% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compose.R
\name{compose}
\alias{compose}
\alias{\%>>>\%}
\title{Compose Functions}
\usage{
compose(...)

fst \%>>>\% snd
}
\arguments{
\item{...}{Functions or lists thereof to compose, in order of application.
Lists of functions are automatically spliced in.
\link[rlang:topic-inject]{Unquoting} of names, via \verb{!!} on the left-hand side
of \verb{:=}, and \link[rlang:topic-inject]{splicing}, via \verb{!!!}, are supported.}

\item{fst, snd}{Functions. These may be optionally named using a colon (\code{:}),
e.g., \code{f \%>>>\% nm: g} names the \code{g}-component \code{"nm"} (see
\sQuote{Exceptions to the Interpretation of Calls as Functions}).
\link[rlang:topic-inject]{Quasiquotation} and the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
semantics are supported (see \sQuote{Semantics of the Composition
Operator}, \sQuote{Quasiquotation} and \sQuote{Examples}).}
}
\value{
Function of class \code{CompositeFunction}, whose
\link[base:formals]{formals} are those of the first function applied (as a
closure).
}
\description{
To compose functions,
\itemize{
\item Use \code{compose()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  compose(f, g, h, ...)
}\if{html}{\out{</div>}}

This makes the function that applies \code{f}, then \code{g}, then \code{h}, etc.
It has the \link[base:formals]{formals} of the first function applied (namely
\code{f}). For example, if

\if{html}{\out{<div class="sourceCode">}}\preformatted{  fun <- compose(paste, toupper)
}\if{html}{\out{</div>}}

then the function \code{fun()} has the same signature as \code{paste()}, and the call

\if{html}{\out{<div class="sourceCode">}}\preformatted{  fun(letters, collapse = ",")
}\if{html}{\out{</div>}}

is equivalent to the composite call

\if{html}{\out{<div class="sourceCode">}}\preformatted{  toupper(paste(letters, collapse = ","))
}\if{html}{\out{</div>}}
\item Use \code{`\%>>>\%`}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  f \%>>>\% g \%>>>\% h \%>>>\% ...
}\if{html}{\out{</div>}}

It comprehends both the semantics of the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator and \link[rlang:topic-inject]{quasiquotation}. For example, if

\if{html}{\out{<div class="sourceCode">}}\preformatted{  sep <- ""
  fun <- sample \%>>>\% paste(collapse = !!sep)
}\if{html}{\out{</div>}}

then the function \code{fun()} has the same signature as \code{sample()}, and the
call

\if{html}{\out{<div class="sourceCode">}}\preformatted{  fun(x, size, replace, prob)
}\if{html}{\out{</div>}}

is equivalent to the composite call

\if{html}{\out{<div class="sourceCode">}}\preformatted{  paste(sample(x, size, replace, prob), collapse = "")
}\if{html}{\out{</div>}}
}

Use \code{\link[=as.list]{as.list()}} to recover the list of composite functions. For example, both

\if{html}{\out{<div class="sourceCode">}}\preformatted{  as.list(compose(paste, capitalize = toupper))

  as.list(paste \%>>>\% capitalize: toupper)
}\if{html}{\out{</div>}}

return the (named) list of functions \code{list(paste, capitalize = toupper)}.
}
\section{Semantics of the Composition Operator}{

The \code{`\%>>>\%`} operator adopts the semantics of the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator:
\enumerate{
\item \strong{Bare names are matched to functions}: For example, in a composition
like

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% foo \%>>>\% ...
}\if{html}{\out{</div>}}

the \sQuote{\code{foo}} is matched to the function of that name.
\item \strong{Function calls are interpreted as a unary function of a point (\code{.})}:
A \emph{call} is interpreted as a \emph{function} (of a point) in one of two ways:
\itemize{
\item If the point matches an argument value, the call is literally
interpreted as the body of the function. For example, in the
compositions

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% foo(x, .) \%>>>\% ...

  ... \%>>>\% foo(x, y = .) \%>>>\% ...
}\if{html}{\out{</div>}}

the \sQuote{\code{foo(x, .)}}, resp. \sQuote{\code{foo(x, y = .)}}, is
interpreted as the function \code{function(..., . = ..1) foo(x, .)}, resp.
\code{function(..., . = ..1) foo(x, y = .)}.
\item Otherwise, the call is regarded as implicitly having the point as its
first argument before being interpreted as the body of the function.
For example, in the compositions

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% foo(x) \%>>>\% ...

  ... \%>>>\% foo(x, y(.)) \%>>>\% ...
}\if{html}{\out{</div>}}

the \sQuote{\code{foo(x)}}, resp. \sQuote{\code{foo(x, y(.))}}, is interpreted
as the function \code{function(..., . = ..1) foo(., x)}, resp.
\code{function(..., . = ..1) foo(., x, y(.))}.
}
\item \strong{Expressions \code{{...}} are interpreted as a function of a point (\code{.})}:
For example, in a composition

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% \{
    foo(.)
    bar(.)
  \} \%>>>\% ...
}\if{html}{\out{</div>}}

the \sQuote{\code{{foo(.); bar(.)}}} is interpreted as the function
\code{function(..., . = ..1) {foo(.); bar(.)}}.

Curly braces are useful when you need to circumvent \code{`\%>>>\%`}'s
usual interpretation of function calls. For example, in a composition

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% \{foo(x, y(.))\} \%>>>\% ...
}\if{html}{\out{</div>}}

the \sQuote{\code{{foo(x, y(.))}}} is interpreted as the function
\code{function(..., . = ..1) foo(x, y(.))}. There is no point as first
argument to \code{foo}.
}

\subsection{Exceptions to the Interpretation of Calls as Functions}{
As a matter of convenience, some exceptions are made to the above
interpretation of calls as functions:
\itemize{
\item \strong{Parenthesis} (\code{(}) applies grouping. (In R, \code{`(`} is indeed a
function.) In particular, expressions within parentheses are literally
interpreted.
\item \strong{Colon} (\code{:}) applies \emph{naming}, according to the syntax
\sQuote{\verb{<name>: <function>}}, where \sQuote{\verb{<function>}} is interpreted
according to the semantics of \code{`\%>>>\%`}. For example, in

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% aName: foo \%>>>\% ...
}\if{html}{\out{</div>}}

the function \code{foo} is named \code{"aName"}.
\item \strong{\code{\link[=fn]{fn()}}}, namespace operators (\code{`::` },
\code{`:::`}) and \strong{\link[base:Extract]{extractors}} (\code{`$`}, \code{`[[`},
\code{`[`}) are literally interpreted. This allows for list extractors to
be applied to composite functions appearing in a \code{`\%>>>\%`} call (see
'Operate on Composite Functions as List-Like Objects'). For example, the
compositions

\if{html}{\out{<div class="sourceCode">}}\preformatted{  paste \%>>>\% tolower

  paste \%>>>\% base::tolower

  (paste \%>>>\% toupper)[[1]] \%>>>\% tolower
}\if{html}{\out{</div>}}

are equivalent functions.
}
}
}

\section{Quasiquotation}{

The \code{`\%>>>\%`} operator supports Tidyverse
\link[rlang:topic-inject]{unquoting} (via \verb{!!}). Use it to:
\itemize{
\item \strong{Enforce immutability}: For example, by unquoting \code{res} in

\if{html}{\out{<div class="sourceCode">}}\preformatted{  res <- "result"
  get_result <- identity \%>>>\% lapply(`[[`, !!res)
}\if{html}{\out{</div>}}

you ensure that the function \code{get_result()} always extracts the component
named \code{"result"}, even if the binding \code{res} changes its value or is
removed altogether.
\item \strong{Interpret the point (\code{.}) in the lexical scope}: Even though
\code{`\%>>>\%`} interprets \sQuote{\code{.}} as a function argument, you can
still reference an object of that name via unquoting. For example,

\if{html}{\out{<div class="sourceCode">}}\preformatted{  . <- "point"
  is_point <- identity \%>>>\% \{. == !!.\}
}\if{html}{\out{</div>}}

determines a function that checks for equality with the string \code{"point"}.
\item \strong{Name composite functions, programmatically}: For example, unquoting
\code{nm} in

\if{html}{\out{<div class="sourceCode">}}\preformatted{  nm <- "aName"
  ... \%>>>\% !!nm: foo \%>>>\% ...
}\if{html}{\out{</div>}}

names the \sQuote{\code{foo}}-component of the resulting composite function
\code{"aName"}.
\item \strong{Accelerate functions by fixing constant dependencies}: For example,
presuming the value of the call \code{f()} is \emph{constant} and that \code{g} is a
\emph{pure} function (meaning that its return value depends only on its
input), both

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ... \%>>>\% g(f()) \%>>>\% ...

  ... \%>>>\% g(!!f()) \%>>>\% ...
}\if{html}{\out{</div>}}

would be functions yielding the same values. But the first would compute
\code{f()} anew with each call, whereas the second would simply depend on a
fixed, pre-computed value of \code{f()}.
}
}

\section{Operate on Composite Functions as List-Like Objects}{

You can think of a composite function as embodying the (possibly nested)
structure of its list of constituent functions. In fact, you can apply
familiar index and assignment operations to a composite function, as if it
were this list, getting a function in return. This enables you to leverage
composite functions as \emph{structured computations}.

\subsection{Indexing}{
For instance, the \sQuote{\code{sum}} in the following composite function

\if{html}{\out{<div class="sourceCode">}}\preformatted{  f <- abs \%>>>\% out: (log \%>>>\% agg: sum)
}\if{html}{\out{</div>}}

can be \link[base:Extract]{extracted} in the usual ways:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  f[[2]][[2]]
  f[[c(2, 2)]]

  f$out$agg
  f[["out"]][["agg"]]
  f[["out"]]$agg

  f$out[[2]]
  f[[list("out", 2)]]
}\if{html}{\out{</div>}}

The last form of indexing with a mixed list is handy when you need to
create an index programmatically.

Additionally, you can excise sub-composite functions with
\code{\link[base:Extract]{[}}, \code{\link[=head]{head()}}, \code{\link[=tail]{tail()}}. For example:
\itemize{
\item Both \code{f[1]} and \code{head(f, 1)} get the \sQuote{\code{abs}} as a composite
function, namely \code{compose(abs)}
\item \code{f[2:1]} reverses the order of the top-level functions to yield

\if{html}{\out{<div class="sourceCode">}}\preformatted{  out: (log \%>>>\% agg: sum) \%>>>\% abs
}\if{html}{\out{</div>}}
\item \code{f$out[c(FALSE, TRUE)]} gets the \sQuote{\code{sum}} as a (named) composite
function
}
}

\subsection{Subset Assignment}{
Similarily, subset assignment works as it does for lists. For instance, you
can replace the \sQuote{\code{sum}} with the identity function:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  f[[2]][[2]] <- identity

  f$out$agg <- identity
  f[["out"]][["agg"]] <- identity

  f$out[[2]] <- identity
  f[[list("out", 2)]] <- identity
}\if{html}{\out{</div>}}

Multiple constituent functions can be reassigned using
\code{\link[base:Extract]{[<-}}. For example

\if{html}{\out{<div class="sourceCode">}}\preformatted{  f[2] <- list(log)

  f["out"] <- list(log)

  f[c(FALSE, TRUE)] <- list(log)
}\if{html}{\out{</div>}}

all replace the second constituent function with \code{log}, so that \code{f} becomes
\code{abs \%>>>\% log}.
}

\subsection{Other List Methods}{
The generic methods \code{\link[=unlist]{unlist()}}, \code{\link[=length]{length()}}, \code{\link[=names]{names()}} also apply to
composite functions. In conjunction with \code{compose()}, you can use
\code{unlist()} to \dQuote{flatten} compositions. For example

\if{html}{\out{<div class="sourceCode">}}\preformatted{  compose(unlist(f, use.names = FALSE))
}\if{html}{\out{</div>}}

gives a function that is identical to

\if{html}{\out{<div class="sourceCode">}}\preformatted{  abs \%>>>\% log \%>>>\% sum
}\if{html}{\out{</div>}}

}
}

\section{Composite Functions Balance Speed and Complexity}{

The speed of a composite function made by \code{compose()} or \code{`\%>>>\%`}
(regardless of its nested depth) is on par with a manually constructed
\emph{serial} composition. This is because \code{compose()} and \code{`\%>>>\%`} are
\strong{associative}, semantically and operationally. For instance, triple
compositions,

\if{html}{\out{<div class="sourceCode">}}\preformatted{  compose(f, g, h)
  f \%>>>\% g \%>>>\% h

  compose(f, compose(g, h))
  f \%>>>\% (g \%>>>\% h)

  compose(compose(f, g), h)
  (f \%>>>\% g) \%>>>\% h
}\if{html}{\out{</div>}}

are all implemented as the \emph{same function}. Lists of functions are
automatically \dQuote{flattened} when composed.

Nevertheless, the original nested structure of constituent functions is
faithfully recovered by \code{\link[=as.list]{as.list()}}. In particular, \code{as.list()} and
\code{compose()} are \strong{mutually invertible}: \code{as.list(compose(fs))} is the same
as \code{fs}, when \code{fs} is a (nested) list of functions. (But note that the
names of the list of composite functions is always a character vector; it
is never \code{NULL}.)
}

\examples{
# Functions are applied in the order in which they are listed
inv <- partial(`/`, 1)  # reciprocal
f0 <- compose(abs, log, inv)
stopifnot(all.equal(f0(-2), 1 / log(abs(-2))))

# Alternatively, compose using the `\%>>>\%` operator
f1 <- abs \%>>>\% log \%>>>\% {1 / .}
stopifnot(all.equal(f1(-2), f0(-2)))

\dontrun{
# Transform a function to a JSON function
library(jsonlite)

# By composing higher-order functions:
jsonify <- {fromJSON \%>>>\% .} \%>>>\% {. \%>>>\% toJSON}

# By directly composing with input/output transformers:
jsonify <- fn(f ~ fromJSON \%>>>\% f \%>>>\% toJSON)}

# Formals of initial function are preserved
add <- function(a, b = 0) a + b
stopifnot(identical(formals(compose(add, inv)), formals(add)))

# Compositions can be provided by lists, in several equivalent ways
f2 <- compose(list(abs, log, inv))
f3 <- compose(!!! list(abs, log, inv))
f4 <- compose(abs, list(log, inv))
f5 <- compose(abs, !!! list(log, inv))
stopifnot(
  all.equal(f2, f0), all.equal(f2(-2), f0(-2)),
  all.equal(f3, f0), all.equal(f3(-2), f0(-2)),
  all.equal(f4, f0), all.equal(f4(-2), f0(-2)),
  all.equal(f5, f0), all.equal(f5(-2), f0(-2))
)

# compose() and as.list() are mutally invertible
f6 <- compose(abs, as.list(compose(log, inv)))
stopifnot(
  all.equal(f6, f0), all.equal(f6(-2), f0(-2))
)
fs <- list(abs, log, inv)
stopifnot(all.equal(check.attributes = FALSE,
  as.list(compose(fs)), fs,
))

# `\%>>>\%` supports names, magrittr `\%>\%` semantics, and quasiquotation
sep <- ""
scramble <- shuffle: sample \%>>>\% paste(collapse = !!sep)
nonsense <- scramble(letters)
stopifnot(
  nchar(nonsense) == 26L,
  identical(letters, sort(strsplit(nonsense, sep)[[1]])),
  identical(scramble$shuffle, sample)
)

}
\seealso{
\code{\link[=constant]{constant()}}; combined with \code{`\%>>>\%`}, this provides a lazy,
structured alternative to the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator.
}
