#include <cpp11.hpp>
using namespace cpp11;
namespace writable = cpp11::writable;

#include <GeographicLib/Gnomonic.hpp>
#include <GeographicLib/Geodesic.hpp>
#include <GeographicLib/Constants.hpp>

using namespace std;
using namespace GeographicLib;

// Forward: Geographic (lon/lat) to Gnomonic (x/y)
// In gnomonic projection, geodesics appear as straight lines
[[cpp11::register]]
cpp11::writable::data_frame gnomonic_fwd_cpp(cpp11::doubles lon, cpp11::doubles lat,
                                              double lon0, double lat0) {
  size_t nn = lon.size();
  
  writable::doubles x(nn);
  writable::doubles y(nn);
  writable::doubles azi(nn);
  writable::doubles rk(nn);
  
  // Create Gnomonic projection centered at origin
  const Geodesic& geod = Geodesic::WGS84();
  Gnomonic gn(geod);
  
  for (size_t i = 0; i < nn; i++) {
    double xx, yy, azz, rkk;
    gn.Forward(lat0, lon0, lat[i], lon[i], xx, yy, azz, rkk);
    
    x[i] = xx;
    y[i] = yy;
    azi[i] = azz;
    rk[i] = rkk;
  }
  
  writable::data_frame out({
    "x"_nm = x,
    "y"_nm = y,
    "azi"_nm = azi,
    "rk"_nm = rk,
    "lon"_nm = lon,
    "lat"_nm = lat
  });
  
  return out;
}

// Reverse: Gnomonic (x/y) to Geographic (lon/lat)
[[cpp11::register]]
cpp11::writable::data_frame gnomonic_rev_cpp(cpp11::doubles x, cpp11::doubles y,
                                              double lon0, double lat0) {
  size_t nn = x.size();
  
  writable::doubles lon(nn);
  writable::doubles lat(nn);
  writable::doubles azi(nn);
  writable::doubles rk(nn);
  
  const Geodesic& geod = Geodesic::WGS84();
  Gnomonic gn(geod);
  
  for (size_t i = 0; i < nn; i++) {
    double la, lo, azz, rkk;
    gn.Reverse(lat0, lon0, x[i], y[i], la, lo, azz, rkk);
    
    lon[i] = lo;
    lat[i] = la;
    azi[i] = azz;
    rk[i] = rkk;
  }
  
  writable::data_frame out({
    "lon"_nm = lon,
    "lat"_nm = lat,
    "azi"_nm = azi,
    "rk"_nm = rk,
    "x"_nm = x,
    "y"_nm = y
  });
  
  return out;
}
