% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/albers.R
\name{albers_fwd}
\alias{albers_fwd}
\alias{albers_rev}
\title{Albers Equal Area projection}
\usage{
albers_fwd(
  x,
  lon0,
  stdlat = NULL,
  stdlat1 = NULL,
  stdlat2 = NULL,
  k0 = 1,
  k1 = 1
)

albers_rev(
  x,
  y,
  lon0,
  stdlat = NULL,
  stdlat1 = NULL,
  stdlat2 = NULL,
  k0 = 1,
  k1 = 1
)
}
\arguments{
\item{x}{For forward conversion: a two-column matrix or data frame of
coordinates (longitude, latitude) in decimal degrees.
For reverse conversion: numeric vector of x (easting) coordinates in meters.}

\item{lon0}{Central meridian in decimal degrees. Can be a vector to specify
different central meridians for each point.}

\item{stdlat}{Standard parallel for single standard parallel projections
(e.g., Lambert Cylindrical Equal Area when stdlat = 0).}

\item{stdlat1, stdlat2}{First and second standard parallels in decimal degrees
for two standard parallel projections.}

\item{k0}{Scale factor at the standard parallel(s). Default is 1.}

\item{k1}{Scale factor at the first standard parallel for two standard
parallel projections. Default is 1.}

\item{y}{Numeric vector of y (northing) coordinates in meters (reverse only).}
}
\value{
Data frame with columns:
\itemize{
\item For forward conversion:
\itemize{
\item \code{x}: Easting in meters
\item \code{y}: Northing in meters
\item \code{convergence}: Grid convergence in degrees
\item \code{scale}: Scale factor at the point
\item \code{lon}, \code{lat}: Input coordinates (echoed)
\item \code{lon0}: Central meridian (echoed)
}
\item For reverse conversion:
\itemize{
\item \code{lon}: Longitude in decimal degrees
\item \code{lat}: Latitude in decimal degrees
\item \code{convergence}: Grid convergence in degrees
\item \code{scale}: Scale factor at the point
\item \code{x}, \code{y}: Input coordinates (echoed)
\item \code{lon0}: Central meridian (echoed)
}
}
}
\description{
Convert geographic coordinates to/from Albers Equal Area conic projection.
This is an equal-area projection commonly used for thematic maps of
regions with greater east-west extent.
}
\details{
The Albers Equal Area conic projection preserves area, making it ideal for:
\itemize{
\item Thematic/choropleth maps where area comparison matters
\item Continental-scale maps (e.g., USGS maps of CONUS)
\item Statistical mapping and analysis
}

Common configurations:
\itemize{
\item \strong{CONUS}: stdlat1 = 29.5, stdlat2 = 45.5, lon0 = -96
\item \strong{Australia}: stdlat1 = -18, stdlat2 = -36, lon0 = 132
\item \strong{Europe}: stdlat1 = 43, stdlat2 = 62, lon0 = 10
}

Special cases:
\itemize{
\item When stdlat1 = -stdlat2, the projection becomes Lambert Cylindrical Equal Area
\item When stdlat1 = stdlat2 = 0, it becomes the cylindrical equal-area projection
}

The \code{lon0} parameter is vectorized, allowing different central meridians
for each point.
}
\examples{
# CONUS Albers Equal Area
pts <- cbind(lon = c(-122, -74, -90), lat = c(37, 41, 30))
albers_fwd(pts, lon0 = -96, stdlat1 = 29.5, stdlat2 = 45.5)

# Australia
aus <- cbind(lon = c(151.2, 115.9, 153.0), lat = c(-33.9, -32.0, -27.5))
albers_fwd(aus, lon0 = 132, stdlat1 = -18, stdlat2 = -36)

# Antarctic projection
ant <- cbind(lon = c(166.67, 77.97, -43.53), lat = c(-77.85, -67.60, -60.72))
albers_fwd(ant, lon0 = 0, stdlat1 = -72, stdlat2 = -60)

# Round-trip conversion
fwd <- albers_fwd(pts, lon0 = -96, stdlat1 = 29.5, stdlat2 = 45.5)
albers_rev(fwd$x, fwd$y, lon0 = -96, stdlat1 = 29.5, stdlat2 = 45.5)

# Single standard parallel (cylindrical-like)
albers_fwd(pts, lon0 = -96, stdlat = 37)
}
\seealso{
\code{\link[=lcc_fwd]{lcc_fwd()}} for Lambert Conformal Conic (conformal, not equal-area).
}
