% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geodist-package.r, R/geodist.R
\docType{package}
\name{geodist}
\alias{geodist-package}
\alias{geodist}
\title{geodist.}
\usage{
geodist(
  x,
  y,
  paired = FALSE,
  sequential = FALSE,
  pad = FALSE,
  measure = "cheap",
  quiet = FALSE
)
}
\arguments{
\item{x}{Rectangular object (matrix, \code{data.frame}, \pkg{tibble},
whatever) containing longitude and latitude coordinates.}

\item{y}{Optional second object which, if passed, results in distances
calculated between each object in \code{x} and each in \code{y}.}

\item{paired}{If \code{TRUE}, calculate paired distances between each entry
in \code{x} and \code{y}, returning a single vector.}

\item{sequential}{If \code{TRUE}, calculate (vector of) distances
sequentially along \code{x} (when no \code{y} is passed), otherwise calculate
matrix of pairwise distances between all points.}

\item{pad}{If \code{sequential = TRUE} values are padded with initial
\code{NA} to return \code{n} values for input with \code{n} rows, otherwise
return \code{n - 1} values.}

\item{measure}{One of "haversine" "vincenty", "geodesic", or "cheap"
specifying desired method of geodesic distance calculation; see Notes.}

\item{quiet}{If \code{FALSE}, check whether max of calculated distances
is greater than accuracy threshold and warn.}
}
\value{
If only \code{x} passed and \code{sequential = FALSE}, a square
symmetric matrix containing distances between all items in \code{x}; If only
\code{x} passed and \code{sequential = TRUE}, a vector of sequential
distances between rows of \code{x}; otherwise if \code{y} is passed, a matrix
of \code{nrow(x)} rows and \code{nrow(y)} columns. All return values are
distances in metres.
}
\description{
Dependency-free, ultra fast calculation of geodesic distances. Includes the reference nanometre-accuracy geodesic distances of Karney (2013) \doi{10.1007/s00190-012-0578-z}, as used by the 'sf' package, as well as Haversine and Vincenty distances. Default distance measure is the "Mapbox cheap ruler" which is generally more accurate than Haversine or Vincenty for distances out to a few hundred kilometres, and is considerably faster. The main function accepts one or two inputs in almost any generic rectangular form, and returns either matrices of pairwise distances, or vectors of sequential distances.

Convert one or two rectangular objects containing lon-lat coordinates into
vector or matrix of geodesic distances in metres.
}
\note{
\code{measure = "cheap"} denotes the mapbox cheap ruler
\url{https://github.com/mapbox/cheap-ruler-cpp}; \code{measure = "geodesic"}
denotes the very accurate geodesic methods given in Karney (2013)
"Algorithms for geodesics" J Geod 87:43-55, and as provided by the
`st_distance()` function from the \pkg{sf} package.
}
\examples{
n <- 50
# Default "cheap" distance measure is only accurate for short distances:
x <- cbind (runif (n, -0.1, 0.1), runif (n, -0.1, 0.1))
y <- cbind (runif (2 * n, -0.1, 0.1), runif (2 * n, -0.1, 0.1))
colnames (x) <- colnames (y) <- c ("x", "y")
d0 <- geodist (x) # A 50-by-50 matrix
d1 <- geodist (x, y) # A 50-by-100 matrix
d2 <- geodist (x, sequential = TRUE) # Vector of length 49
d2 <- geodist (x, sequential = TRUE, pad = TRUE) # Vector of length 50
d0_2 <- geodist (x, measure = "geodesic") # nanometre-accurate version of d0

# Input data can also be 'data.frame' objects:
xy <- data.frame (x = runif (n, -0.1, 0.1), y = runif (n, -0.1, 0.1))
d <- geodist (xy)
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/hypertidy/geodist}
  \item \url{https://hypertidy.github.io/geodist/}
  \item Report bugs at \url{https://github.com/hypertidy/geodist/issues}
}

}
\author{
\strong{Maintainer}: Mark Padgham \email{mark.padgham@email.com}

Authors:
\itemize{
  \item Michael D. Sumner
}

Other contributors:
\itemize{
  \item Charles F.F Karney (Original author of included code for geodesic distances) [copyright holder]
}

}
