% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VarObj.R
\name{VarObj}
\alias{VarObj}
\title{Fréchet Variance Trajectory for densities}
\usage{
VarObj(tgrid, yin = NULL, hin = NULL, din = NULL, qin = NULL, optns = list())
}
\arguments{
\item{tgrid}{Time grid vector for the time varying object data.}

\item{yin}{An array or list of lists holding the samples of observations. If \code{yin} is an array, it has size \code{n} x \code{length(tgrid)} x numbers of samples holding the observation, such that \code{yin[i,j,]} holds the observations to the ith sample at the jth time grid. If \code{yin} is a list of lists, \code{yin[[i]][[j]]} holds the observations to the ith sample at the jth time grid.}

\item{hin}{A list of lists holding the histogram for each subject. \code{hin[[i]][[j]]} holds the histogram to the ith sample at the jth time grid.}

\item{din}{A three dimension array of size \code{n} x \code{length(tgrid)} x \code{length(optns$dSup)} holding the observed densities, such that \code{din[i,j,]} holds the observed density function taking values on \code{optns$dSup} corresponding to the ith sample at the jth time grid.}

\item{qin}{A three dimension array of size \code{n} x \code{length(tgrid)} x \code{length(optns$qSup)} holding the observed quantiles, such that \code{din[i,j,]} holds the observed density function taking values on \code{optns$qSup} corresponding to the ith sample at the jth time grid.
Note that only one of \code{yin}, \code{hin}, \code{din} and \code{qin} needs to be input. If more than one of them are specified, \code{yin} overwrites \code{hin}, \code{hin} overwrites \code{din} and \code{din} overwrites \code{qin}.
where each row holds the observations for one subject on the common grid \code{tGrid}.}

\item{optns}{a list of options control parameters specified by \code{list(name=value)}.}
}
\value{
A list of the following:
\item{tgridout}{Time grid vector for the output time varying object data.}
\item{K}{Numbers of principal components.}
\item{nu}{A vector of dimension \code{length(tgridout)} giving the mean function support on \code{tgridout} of the Fréchet variance function.}
\item{lambda}{A vector of dimension \code{K} containing eigenvalues.}
\item{phi}{A \code{length(tgridout)} X \code{K} matrix containing eigenfunctions support on \code{tgridout} of the Fréchet variance function.}
\item{xiEst}{A \code{n} X \code{K} matrix containing the FPC estimates.}
\item{cumFVE}{A vector of dimension \code{K} with the fraction of the cumulative total variance explained with each additional FPC.}
\item{FPCAObj}{FPCA Object of Fréchet variance function.}
\item{tgridin}{Input \code{tgrid}.}
\item{qSup}{A vector of dimension \code{length(tgridin)} giving the domain grid of quantile functions \code{qout}.}
\item{qout}{A three dimension array of dimension \code{n} x \code{length(tgridin)} x \code{length(qSup)} holding the observed quantiles, such that \code{qout[i,j,]} holds the observed density function taking values on \code{qSup} corresponding to the ith sample at the jth time grid.}
\item{qmean}{A \code{length(tgridin)} X \code{length(qSup)} matrix containing the time varying Fréchet mean function.}
\item{VarTraj}{A \code{n} X \code{length(tgridin)} matrix containing the variance trajectory.}
}
\description{
Modeling time varying density objects with respect to $L^2$-Wasserstein distance by Fréchet variance trajectory
}
\details{
Available control options are \code{qSup}, \code{nqSup}, \code{dSup} and other options in FPCA of fdapace.
}
\examples{
\donttest{
set.seed(1)
#use yin 
tgrid = seq(1, 50, length.out = 50)
dSup = seq(-10, 60, length.out = 100)
yin = array(dim=c(30, 50, 100))
for(i in 1:30){
  yin[i,,] = t(sapply(tgrid, function(t){
    rnorm(100, mean = rnorm(1, mean = 1, sd = 1/t))
  }))
}
result1 = VarObj(tgrid, yin = yin)
plot(result1$phi[,1])
plot(result1$phi[,2])
yin2 = replicate(30, vector("list", 50), simplify = FALSE)
for(i in 1:30){
  for(j in 1:50){
    yin2[[i]][[j]] = yin[i,j,]
  }}
result1 = VarObj(tgrid, yin = yin2)

# use hin
tgrid = seq(1, 50, length.out = 50)
dSup = seq(-10, 60, length.out = 100)
hin =  replicate(30, vector("list", 50), simplify = FALSE)
for(i in 1:30){
  for (j in 1:50){
    hin[[i]][[j]] = hist(yin[i,j,])
  }
}
result2 = VarObj(tgrid, hin = hin)

# use din
tgrid = seq(1, 50, length.out = 50)
dSup = seq(-10, 60, length.out = 100)
din = array(dim=c(30, 50, 100))
for(i in 1:30){
  din[i,,] = t(sapply(tgrid, function(t){
    dnorm(dSup, mean = rnorm(1, mean = t, sd = 1/t))
  }))
}
result3 = VarObj(tgrid, din = din, optns=list(dSup = dSup))

# use qin
tgrid = seq(1, 50, length.out = 50)
qSup = seq(0.00001,1-0.00001,length.out = 100)
qin = array(dim=c(30, 50, 100))
for(i in 1:30){
  qin[i,,] = t(sapply(tgrid, function(t){
    qnorm(qSup, mean = rnorm(1, mean = t, sd = 1/t))
  }))
}
result4 = VarObj(tgrid, qin = qin, optns=list(qSup = round(qSup, 4)))
}
}
\references{
\cite{Dubey, P., & Müller, H. G. (2021). Modeling Time-Varying Random Objects and Dynamic Networks. Journal of the American Statistical Association, 1-33.}
}
