% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{melt_od_matrix}
\alias{melt_od_matrix}
\title{Melt Origin-Destination Matrix to Long Format}
\usage{
melt_od_matrix(od_matrix, nodes = NULL, sort = TRUE)
}
\arguments{
\item{od_matrix}{A numeric matrix with origin-destination flows. Rows represent origins,
columns represent destinations. The matrix should be square (same number of rows and columns).}

\item{nodes}{(Optional) Numeric or integer vector of node IDs in the graph matching the rows
and columns of the matrix. If provided, must have length equal to both \code{nrow(od_matrix)}
and \code{ncol(od_matrix)}. When \code{nodes} is provided, these IDs are used directly,
ignoring row and column names. This is particularly useful when mapping zone-based OD matrices
to graph node IDs (e.g., using \code{\link[sf]{st_nearest_feature}} to find nearest graph nodes).
If omitted, row and column names (if present) will be used as node IDs, coerced to integer
if possible. If names are not available or cannot be coerced to integer, sequential integers
will be used.}

\item{sort}{Sort long OD-matrix in ascending order of from and to columns. This can have computational benefits, e.g., when multithreading with \code{method = "AoN"}.}
}
\value{
A data frame with columns:
  \itemize{
    \item \code{from} - Origin node ID (integer)
    \item \code{to} - Destination node ID (integer)
    \item \code{flow} - Flow value (numeric)
  }
  Only rows with finite, positive flow values are included.
}
\description{
Convert an origin-destination (OD) matrix to a long-format data frame
  with columns \code{from}, \code{to}, and \code{flow}.
}
\details{
This function converts a square OD matrix to long format, which is required by
\code{\link[=run_assignment]{run_assignment()}}. The behavior depends on whether \code{nodes}
is provided:

\strong{When \code{nodes} is provided:}
\itemize{
  \item The \code{nodes} vector is used directly as node IDs for both origins and destinations
  \item Row and column names are ignored (but must match if both are present)
  \item This is the recommended approach when working with zone-based OD matrices that need to be
    mapped to graph nodes, as it ensures the node IDs match those in the graph
}

\strong{When \code{nodes} is omitted:}
\itemize{
  \item Row and column names are extracted from the matrix (if available)
  \item Names are coerced to integer if possible; if coercion fails or names are missing,
    sequential integers are used
  \item This approach works well when the matrix row/column names already correspond to graph node IDs
}

In both cases, the function:
\itemize{
  \item Creates a long-format data frame with all origin-destination pairs
  \item Filters out non-finite and zero flow values
}

The function is useful for converting OD matrices to the long format required by
\code{\link[=run_assignment]{run_assignment()}}.
}
\examples{
library(flownet)
library(sf)

# Load existing network and convert to graph
africa_net <- africa_network[!africa_network$add, ]
graph <- linestrings_to_graph(africa_net)
nodes <- nodes_from_graph(graph, sf = TRUE)

# Map cities/ports to nearest network nodes
nearest_nodes <- nodes$node[st_nearest_feature(africa_cities_ports, nodes)]

# Example 1: Simple gravity-based OD matrix
od_mat <- outer(africa_cities_ports$population, africa_cities_ports$population) / 1e12
dimnames(od_mat) <- list(nearest_nodes, nearest_nodes)
od_long <- melt_od_matrix(od_mat)
head(od_long)

# Example 2: Using nodes argument (when matrix has zone IDs, not node IDs)
# Here zones are 1:n_cities, nodes argument maps them to graph nodes
dimnames(od_mat) <- NULL
od_long2 <- melt_od_matrix(od_mat, nodes = nearest_nodes)
head(od_long2)

}
\seealso{
\code{\link{africa_cities_ports}}, \code{\link{africa_network}},
  \code{\link[=nodes_from_graph]{nodes_from_graph()}},
  \code{\link[=run_assignment]{run_assignment()}}, \link{flownet-package}
}
