% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bkg_geonames.R
\name{bkg_geonames}
\alias{bkg_geonames}
\alias{bkg_endonyms}
\title{Geographical objects and endonyms}
\usage{
bkg_geonames(
  ...,
  names = TRUE,
  ags = FALSE,
  dlm = FALSE,
  status = FALSE,
  bbox = NULL,
  poly = NULL,
  predicate = "intersects",
  filter = NULL,
  epsg = 3035,
  properties = NULL,
  max = NULL
)

bkg_endonyms(..., filter = NULL, properties = NULL, max = NULL)
}
\arguments{
\item{...}{Used to construct CQL filters. Dot arguments accept an R-like
syntax that is converted to CQL queries internally. These queries basically
consist of a property name on the left, an aribtrary vector on the right,
and an operator that links both sides. If multiple queries are provided,
they will be chained with \code{AND}. The following operators and their
respective equivalents in CQL and XML are supported:

\tabular{lll}{
\strong{R} \tab \strong{CQL} \tab \strong{XML}\cr
\code{==} \tab \code{=} \tab \code{PropertyIsEqualTo}\cr
\code{!=} \tab \code{<>} \tab \code{PropertyIsNotEqualTo}\cr
\code{<} \tab \code{<} \tab \code{PropertyIsLessThan}\cr
\code{>} \tab \code{>} \tab \code{PropertyIsGreaterThan}\cr
\code{>=} \tab \code{>=} \tab \code{PropertyIsGreaterThanOrEqualTo}\cr
\code{<=} \tab \code{<=} \tab \code{PropertyIsLessThanOrEqualTo}\cr
\code{\%LIKE\%} \tab \code{LIKE} \tab \code{PropertyIsLike}\cr
\code{\%ILIKE\%} \tab \code{ILIKE} \tab\cr
\code{\%in\%} \tab \code{IN} \tab \code{PropertyIsEqualTo} and \code{Or}
}

To construct more complex queries, you can use the \code{filter} argument
to pass CQL queries directly. Also note that you can switch between
CQL and XML queries using \code{options(ffm_query_language = "xml")}.
See also \code{\link{wfs_filter}}.}

\item{names}{If \code{TRUE}, includes endonyms of the geographical
objects in the output using \code{bkg_endonyms}. Technically, this can be
\code{FALSE}, because the endpoint only returns meta data on geographical
names by default. If this argument is \code{TRUE}, the output is merged with
the endonym table requiring an additional request. Defaults to \code{TRUE}.}

\item{ags}{If \code{TRUE}, resolves AGS codes to geographical names
using \code{\link{bkg_ags}}. Note that setting this to \code{TRUE} requires
an additional web request. Defaults to \code{FALSE}.}

\item{dlm}{If \code{TRUE}, adds the DLM identifier corresponding to the
national name identifiers (NNID) of the output using \code{\link{bkg_dlm}}.
Note that setting this to \code{TRUE} requires an additional web request.
Defaults to \code{FALSE}.}

\item{status}{If \code{TRUE}, adds the date of the objects last edit to
the output. Note that setting this to \code{TRUE} requires an additional
web request. Defaults to \code{FALSE}.}

\item{bbox}{An sf geometry or a boundary box vector of the format
\code{c(xmin, ymin, xmax, ymax)}. Used as a geometric filter to include
only those geometries that relate to \code{bbox} according to the predicate
specified in \code{predicate}. If an sf geometry is provided, coordinates
are automatically transformed to ESPG:25832 (the default CRS), otherwise
they are expected to be in EPSG:25832.}

\item{poly}{An sf geometry. Used as a geometric filter to include
only those geometries that relate to \code{poly} according to the predicate
specified in \code{predicate}. Coordinates are automatically transformed to
ESPG:25832 (the default CRS).}

\item{predicate}{A spatial predicate that is used to relate the output
geometries with the object specified in \code{bbox} or \code{poly}. For
example, if \code{predicate = "within"}, and \code{bbox} is specified,
returns only those geometries that lie within \code{bbox}. Can be one of
\code{"equals"}, \code{"disjoint"}, \code{"intersects"}, \code{"touches"},
\code{"crosses"}, \code{"within"}, \code{"contains"}, \code{"overlaps"},
\code{"relate"}, \code{"dwithin"}, or \code{"beyond"}. Defaults to
\code{"intersects"}.}

\item{filter}{A character string containing a valid CQL or XML filter. This
string is appended to the query constructed through \code{...}. Use this
argument to construct more complex filters. Defaults to \code{NULL}.}

\item{epsg}{An EPSG code specifying a coordinate reference system of the
output. If you're unsure what this means, try running
\code{sf::st_crs(...)$epsg} on a spatial object that you are working with.
Defaults to 3035.}

\item{properties}{Vector of columns to include in the output.}

\item{max}{Maximum number of results to return.}
}
\value{
A dataframe containing the following columns:
\itemize{
\item{\code{nnid}}: National name identifier
\item{\code{landesCode}}: Country identifier; 276 is Germany.
\item{\code{beschreibung}}: Optional details
\item{\code{geoLaenge}}: Geographical longitude
\item{\code{geoBreite}}: Geographical latitude
\item{\code{hoehe}}: Elevation above sea level
\item{\code{hoeheger}}: Computed elevation above sea level
\item{\code{groesse}}: Undocumented, but I guess this relates to the suggested print
size of the labels
\item{\code{ewz}}: Number of inhabitants
\item{\code{ewzger}}: Computed number of inhabitants
\item{\code{ags}}: Official municipality key (Amtlicher Gemeindeschlüssel).
Related to the ARS but shortened to omit position 6 to 9. Structured as
follows: \itemize{
\item{Position 1-2: Federal state}
\item{Position 3: Government region}
\item{Position 4-5: District}
\item{Position 6-8: Municipality}
}
\item{\code{gemteil}}: Whether the place is part of a municipality
\item{\code{virtuell}}: Whether the place is a real or virtual locality
\item{\code{ars}}: Territorial code (Amtlicher Regionalschlüssel).
The ARS is stuctured hierarchically as follows: \itemize{
\item{Position 1-2: Federal state}
\item{Position 3: Government region}
\item{Position 4-5: District}
\item{Position 6-9: Administrative association}
\item{Position 10-12: Municipality}
}
}

If \code{ags = TRUE}, adds the output of \code{\link{bkg_ags}}. If
\code{dlm = TRUE}, adds a column \code{dlm_id} containing identifiers of
\code{\link{bkg_dlm}}.

\code{bkg_endonyms} contains the following columns:
\itemize{
\item{\code{name}}: Name of the geographical object
\item{\code{geschlecht}}: If applicable, the grammatical gender of a geographical name
}

These are also included in the output of \code{bkg_geonames} if
\code{names = TRUE}.
}
\description{
Get geographic names including toponyms and endonyms. \code{bkg_geonames}
retrieves the geographical "objects" based on the digital landscape model
(DLM). These objects contain a set of metadata and a national name identifier
(NNID). These NNIDs can be used to join with the endonyms related to a
geographical object (\code{bkg_endonyms}).

These functions interface the \code{wfs_gnde} product of the BKG.
}
\details{
These functions make use of the GN-DE WFS, just like \code{\link{bkg_ags}},
\code{\link{bkg_ars}}, and \code{\link{bkg_area_codes}}. The infrastructure
behind it is actually quite sophisticated and this function may not live
up to these standards. You can use \code{\link{bkg_feature_types}} and
\code{\link{bkg_wfs}} to manually explore the service's endpoints if
required.
}
\section{Query language}{

While other WFS interfaces like \code{\link{bkg_admin}} allow querying
using CQL or XML, \code{bkg_geonames} and \code{bkg_endonyms} (using the
GNDE service) ONLY support XML. This has implications for the allowed
query filters (see \code{\link{wfs_filter}}).
}

\examples{
\dontshow{if (getFromNamespace("ffm_run_examples", ns = "ffm")() && requireNamespace("ggplot2")) withAutoprint(\{ # examplesIf}
# Plot geographical objects in Cologne
library(sf)
library(ggplot2)
cgn <- st_sfc(st_point(c(6.956944, 50.938056)), crs = 4326)
cgn <- st_buffer(st_transform(cgn, 3035), dist = 500)

cgn_names <- bkg_geonames(poly = cgn)
st_geometry(cgn_names) <- st_centroid(st_geometry(cgn_names))
cgn_names <- cgn_names[lengths(st_intersects(cgn_names, cgn)) > 0, ]
ggplot(cgn_names) + geom_sf_text(aes(label = name)) + theme_void()
\dontshow{\}) # examplesIf}
}
\seealso{
\href{https://mis.bkg.bund.de/trefferanzeige?docuuid=f1fe5b66-25d6-44c7-b26a-88625aca9573}{\code{wfs_gnde} MIS record}

\href{https://sgx.geodatenzentrum.de/web_public/gdz/dokumentation/deu/wfs-gnde.pdf}{\code{wfs_gnde} documentation}

\code{\link{bkg_ags}} and \code{\link{bkg_ars}} for geographical names
of administrative areas
}
