

#' Generalized extreme value distribution
#'
#' Density, distribution function, quantile function and random generation
#' for the generalized extreme value distribution.
#'
#' @param x,q	            vector of quantiles.
#' @param p	              vector of probabilities.
#' @param n	              number of observations. If \code{length(n) > 1},
#'                        the length is taken to be the number required.
#' @param mu,sigma,xi	    location, scale, and shape parameters. Scale must be positive.
#' @param log,log.p	      logical; if TRUE, probabilities p are given as log(p).
#' @param lower.tail	    logical; if TRUE (default), probabilities are \eqn{P[X \le x]}
#'                        otherwise, \eqn{P[X > x]}.
#'
#' @details
#'
#' Probability density function
#' \deqn{ f(x) = \left\{\begin{array}{ll}
#' \frac{1}{\sigma} \left(1+\xi \frac{x-\mu}{\sigma}\right)^{-1/\xi-1} \exp\left(-\left(1+\xi \frac{x-\mu}{\sigma}\right)^{-1/\xi}\right) & \xi \neq 0 \\
#' \frac{1}{\sigma} \exp\left(- \frac{x-\mu}{\sigma}\right) \exp\left(-\exp\left(- \frac{x-\mu}{\sigma}\right)\right)                       & \xi = 0
#' \end{array}\right.
#' }{
#' f(x) = [if \xi != 0:] 1/\sigma * (1+\xi*(x-\mu)/\sigma)^{-1/\xi-1} * exp(-(1+\xi*(x-\mu)/\sigma)^{-1/\xi})
#' [else:] 1/\sigma * exp(-(x-\mu)/\sigma) * exp(-exp(-(x-\mu)/\sigma))
#' }
#'
#' Cumulative distribution function
#' \deqn{ F(x) = \left\{\begin{array}{ll}
#' \exp\left(-\left(1+\xi \frac{x-\mu}{\sigma}\right)^{1/\xi}\right) & \xi \neq 0 \\
#' \exp\left(-\exp\left(- \frac{x-\mu}{\sigma}\right)\right)          & \xi = 0
#' \end{array}\right.
#' }{
#' F(x) = [if \xi != 0:] exp(-(1+\xi*(x-\mu)/\sigma)^{1/\xi})
#' [else:] exp(-exp(-(x-\mu)/\sigma))
#' }
#'
#' Quantile function
#' \deqn{ F^{-1}(p) = \left\{\begin{array}{ll}
#' \mu - \frac{\sigma}{\xi} (1 - (-\log(p))^\xi)  & \xi \neq 0 \\
#' \mu - \sigma \log(-\log(p))                    & \xi = 0
#' \end{array}\right.
#' }{
#' F^-1(p) = [if \xi != 0:] \mu - \sigma/\xi * (1 - (-log(p))^\xi)
#'           [else:] \mu - \sigma * log(-log(p))
#' }
#'
#' @references
#' Coles, S. (2001). An Introduction to Statistical Modeling of Extreme Values.
#' Springer.
#'
#' @examples 
#' 
#' curve(dgev(x, xi = -1/2), -4, 4, col = "green", ylab = "")
#' curve(dgev(x, xi = 0), -4, 4, col = "red", add = TRUE)
#' curve(dgev(x, xi = 1/2), -4, 4, col = "blue", add = TRUE)
#' legend("topleft", col = c("green", "red", "blue"), lty = 1,
#'        legend = expression(xi == -1/2, xi == 0, xi == 1/2), bty = "n")
#' 
#' x <- rgev(1e5, 5, 2, .5)
#' hist(x, 1000, freq = FALSE, xlim = c(0, 50))
#' curve(dgev(x, 5, 2, .5), 0, 50, col = "red", add = TRUE, n = 5000)
#' hist(pgev(x, 5, 2, .5))
#' plot(ecdf(x), xlim = c(0, 50))
#' curve(pgev(x, 5, 2, .5), 0, 50, col = "red", lwd = 2, add = TRUE)
#'
#' @name GEV
#' @aliases GEV
#' @aliases dgev
#' 
#' @keywords distribution
#' @concept Univariate
#' @concept Continuous
#'
#' @export

dgev <- function(x, mu = 0, sigma = 1, xi = 0, log = FALSE) {
  cpp_dgev(x, mu, sigma, xi, log[1L])
}


#' @rdname GEV
#' @export

pgev <- function(q, mu = 0, sigma = 1, xi = 0, lower.tail = TRUE, log.p = FALSE) {
  cpp_pgev(q, mu, sigma, xi, lower.tail[1L], log.p[1L])
}


#' @rdname GEV
#' @export

qgev <- function(p, mu = 0, sigma = 1, xi = 0, lower.tail = TRUE, log.p = FALSE) {
  cpp_qgev(p, mu, sigma, xi, lower.tail[1L], log.p[1L])
}


#' @rdname GEV
#' @export

rgev <- function(n, mu = 0, sigma = 1, xi = 0) {
  if (length(n) > 1) n <- length(n)
  cpp_rgev(n, mu, sigma, xi)
}

