setClassUnion("matrixORnumeric", c("matrix", "numeric"))
setClassUnion("functionORNULL",members=c("function", "NULL"))
setClassUnion("listORNULL",members=c("list", "NULL"))
setClassUnion("matrixORnumericORfunction", c("matrix", "numeric", "function"))
setClassUnion("matrixORnumericORfunctionORNULL", c("matrix", "numeric", "function", "NULL"))


#' An S4 Class Representing an Exponential Marked Hawkes Model
#'
#' This class defines a marked Hawkes model with an exponential kernel.
#' The intensity of the ground process is expressed as:
#' \deqn{\lambda(t) = \mu + \int_{(-\infty,t)\times E} ( \alpha + g(u, z) )  e^{-\beta (t-u)} M(du \times dz).}
#' For more details, refer to the vignettes.
#'
#' \eqn{\mu} is base intensity, typically a constant vector or a function.
#'
#' \eqn{\alpha} is a constant matrix representing the impact on intensities after events, stored in the \code{alpha} slot.
#'
#' \eqn{\beta} is a constant matrix for exponential decay rates, stored in the \code{beta} slot.
#'
#' \eqn{z} represents the mark and can be generated by \code{rmark} slot.
#'
#' \eqn{g} is represented by \code{eta} when it is linear function of \eqn{z}, and by \code{impact} when it is a genenral function.
#'
#' \code{mu}, \code{alpha} and \code{beta} are required slots for every exponential Hawkes model.
#' \code{rmark} and \code{impact} are additional slots.
#'
#' @slot mu A numeric value, matrix, or function. If numeric, it is automatically converted to a matrix.
#' @slot alpha A numeric value, matrix, or function. If numeric, it is automatically converted to a matrix, representing the exciting term.
#' @slot beta A numeric value, matrix, or function. If numeric, it is automatically converted to a matrix, representing the exponential decay.
#' @slot eta A numeric value, matrix, or function. If numeric, it is automatically converted to a matrix, representing the impact of an additional mark.
#' @slot impact A function describing the after-effects of the mark on \eqn{\lambda}, with the first argument always being `param`.
#' @slot dimens The dimension of the model.
#' @slot rmark A function that generates marks for the counting process, used in simulations.
#' @slot dmark A density function for the mark, used in estimation.
#' @slot type_col_map A mapping between type and column number of the kernel used in multi-kernel models.
#' @slot rresidual A function for generating residuals, analogous to the R random number generator function, specifically for the discrete Hawkes model.
#' @slot dresidual A density function for the residual.
#' @slot presidual A distribution function for the residual.
#' @slot qresidual A quantile function for the residual.
#'
#' @examples
#' MU <- matrix(c(0.2), nrow = 2)
#' ALPHA <- matrix(c(0.75, 0.92, 0.92, 0.75), nrow = 2, byrow=TRUE)
#' BETA <- matrix(c(2.25, 2.25, 2.25, 2.25), nrow = 2, byrow=TRUE)
#' mhspec2 <- new("hspec", mu=MU, alpha=ALPHA, beta=BETA)
#' mhspec2
#'
#' @export
setClass("hspec",
  slots = list(
    mu = "matrixORnumericORfunction",
    alpha = "matrixORnumericORfunction",
    beta = "matrixORnumericORfunction",
    eta = "matrixORnumericORfunctionORNULL",
    impact = "functionORNULL",
    dimens = "numeric",
    rmark = "functionORNULL",
    dmark = "functionORNULL",
    type_col_map = "listORNULL",
    rresidual = 'functionORNULL',
    dresidual = 'functionORNULL',
    presidual = 'functionORNULL',
    qresidual = 'functionORNULL'
  )
)

# Initialize the hspec object
#
#
# \code{mu}, \code{alpha} and \code{beta} are required slots for every exponential Hawkes model.
# \code{mark} and \code{impact} are additional slots.
#
# @param .Object \code{hspec}
# @param mu Numeric value or matrix or function, if numeric, automatically converted to matrix.
# @param alpha numeric value or matrix or function, if numeric, automatically converted to matrix, exciting term
# @param beta numeric value or matrix or function, if numeric,, automatically converted to matrix, exponential decay
# @param dimens dimesion of the model
# @param rmark a function that generate mark
# @param impact a function that describe the mark impact whose first argument is always `param`
# @param stability_check check the spectral radius
setMethod(
  "initialize",
  "hspec",
  function(.Object, mu, alpha, beta, eta=NULL, impact=NULL,
           type_col_map = NULL, dimens=NULL,
           rmark=NULL, dmark = NULL,
           dresidual = NULL, presidual = NULL, qresidual = NULL, rresidual = NULL,
           stability_check = FALSE){

    .Object@rmark <- rmark

    if( !is.function(mu) & !is.matrix(mu) ){
      .Object@mu <- as.matrix(mu)
    } else{
      .Object@mu <- mu
    }
    if( !is.function(alpha) & !is.matrix(alpha) ){
      .Object@alpha <- as.matrix(alpha)
    } else{
      .Object@alpha <- alpha
    }
    if( !is.function(beta) & !is.matrix(beta) ){
      .Object@beta <-  as.matrix(beta)
    } else {
      .Object@beta <- beta
    }
    if( !is.null(eta) & !is.function(eta) & !is.matrix(eta) ){
      .Object@eta <-  as.matrix(eta)
    } else {
      .Object@eta <- eta
    }


    .Object@type_col_map <- type_col_map

    .Object@dmark <- dmark
    .Object@impact <- impact

    .Object@rresidual <- rresidual
    .Object@dresidual <- dresidual
    .Object@presidual <- presidual
    .Object@qresidual <- qresidual

    nrow_mu <- ncol_mu <- NULL

    if( is.matrix(.Object@mu) ) {

      nrow_mu <- nrow(.Object@mu)
      ncol_mu <- ncol(.Object@mu)

      if (ncol_mu != 1) stop("The number of column in mu should be one.")


    } else if( is.function(.Object@mu) && length(formals(.Object@mu)) == 1 && names(formals(.Object@mu)[1]) == "param"){

      # In this else if:
      # Check if .Object@mu is a function and the function has exactly one formal argument and
      # Check if the name of the first formal argument is "param"

      nrow_mu <- nrow(evalf(.Object@mu))
      ncol_mu <- ncol(evalf(.Object@mu))

      if (ncol_mu != 1) stop("The number of column in mu should be one.")

    }

    nrow_alpha <- ncol_alpha <- nrow_beta <- ncol_beta <- nrow_eta <- ncol_eta <- NULL

    if ( is.matrix(.Object@alpha) ) {

      nrow_alpha <- nrow(.Object@alpha)
      ncol_alpha <- ncol(.Object@alpha)

    } else if( is.function(.Object@alpha) && length(formals(.Object@alpha)) == 1 && names(formals(.Object@alpha)[1]) == "param"){

      nrow_alpha <- nrow(evalf(.Object@alpha))
      ncol_alpha <- ncol(evalf(.Object@alpha))

    }

    if ( is.matrix(.Object@beta) ) {

      nrow_beta <- nrow(.Object@beta)
      ncol_beta <- ncol(.Object@beta)

    }  else if( is.function(.Object@beta) && length(formals(.Object@beta)) == 1 && names(formals(.Object@beta)[1]) == "param"){

      nrow_beta <- nrow(evalf(.Object@beta))
      ncol_beta <- ncol(evalf(.Object@beta))

    }

    if (!is.null(.Object@eta)){
      if(is.matrix(.Object@eta)){

        nrow_eta <- nrow(.Object@eta)
        ncol_eta <- ncol(.Object@eta)

      } else if (is.function(.Object@eta) &&length(formals(.Object@eta)) == 1 && names(formals(.Object@eta)[1]) == "param"){

        nrow_eta <- nrow(evalf(.Object@eta))
        ncol_eta <- ncol(evalf(.Object@eta))

      }

    }

    # Check if dimens is provided or determinable
    if (is.null(dimens) & is.null(nrow_mu) & is.null(nrow_alpha) & is.null(nrow_beta) & is.null(nrow_eta)) {
      stop("dimens should be provided in this model.")
    }


    # Ensure consistent dimensions across parameters
    if (length(unique(c(dimens, nrow_mu, nrow_alpha, nrow_beta, nrow_eta))) == 1) {
      .Object@dimens <- unique(c(dimens, nrow_mu, nrow_alpha, nrow_beta, nrow_eta))
    } else {
      stop("The number of rows in parameters and given dimens should all be the same.")
    }

    # Check consistency of column numbers in alpha, beta, and eta
    if (length(unique(c(ncol_alpha, ncol_beta, ncol_eta))) != 1) {
      stop("The number of columns in alpha, beta, and eta should all be the same.")
    }

    # Require type_col_map if column numbers exceed dimensions
    if (ncol_alpha > .Object@dimens & is.null(.Object@type_col_map)) {
      stop("type_col_map should be provided for this model as the number of columns in parameters exceeds the dimens.")
    }


    # Check spectral radius for stability if required
    if (stability_check == TRUE && max(abs(eigen(alpha / beta)$values)) > 1) {
      warning("This model may not satisfy the stability condition.")
    }

    # Call the next method in the chain
    methods::callNextMethod()

    # Return the initialized object
    .Object


  }
)

#' Set Residual Distribution Functions for Hawkes Model Specification
#'
#' Sets residual distribution functions (density, CDF, quantile, and random generation)
#' for a Hawkes model specification object with fixed parameters.
#'
#' @param object An object of class `hspec` (Hawkes model specification)
#' @param param A named numeric vector of parameters for the residual distribution
#' @param dresidual Density function of the residual distribution (optional)
#' @param presidual Cumulative distribution function (CDF) of the residual distribution (optional)
#' @param qresidual Quantile function of the residual distribution (optional)
#' @param rresidual Random generation function of the residual distribution (optional)
#' @param ... Additional arguments for future extensions
#'
#' @return An updated `hspec` object with residual functions set
#'
#' @details
#' This method allows setting residual distribution functions for a flexible model.
#'
#' The `param` argument in these functions defaults to the parameters provided during
#' setup and is used for estimation.
#'
#' @examples
#' \dontrun{
#' # Create basic Hawkes specification
#' hspec_obj <- new("hspec",
#'                  mu = matrix(0.1, nrow = 1),
#'                  alpha = matrix(0.5, nrow = 1),
#'                  beta = matrix(1.0, nrow = 1))
#'
#' # Set residual distribution parameters
#' params <- c(a = 0.5, ell = 1.0)
#'
#' # Apply residual functions
#' hspec_obj <- set_residual(
#'   hspec_obj,
#'   param = params,
#'   dresidual = dtzexp,
#'   presidual = ptzexp,
#'   qresidual = qtzexp,
#'   rresidual = rtzexp
#' )
#'
#' # Check resulting functions
#' hspec_obj@dresidual
#' hspec_obj@rresidual
#' }
#'
#' @export
setGeneric(
  "set_residual",
  function(object, param,  dresidual = NULL, presidual = NULL,
           qresidual = NULL, rresidual = NULL, ...) {
    standardGeneric("set_residual")
  }
)

#' @rdname set_residual
#' @aliases set_residual,hspec-method
setMethod(
  "set_residual",
  signature(object = "hspec"),
  function(object, param, dresidual = NULL, presidual = NULL,
           qresidual = NULL, rresidual = NULL) {

    # Copy parameters for closure storage
    stored_param <- param

    # Create dresidual function with desired interface
    if (!is.null(dresidual)) {
      object@dresidual <- function(x, param = stored_param) {
        # Extract required arguments from param vector
        args_list <- list(x = x)
        fun_args <- methods::formalArgs(dresidual)

        # Map parameters to function arguments
        for (arg_name in names(param)) {
          if (arg_name %in% fun_args) {
            args_list[[arg_name]] <- param[arg_name]
          }
        }

        do.call(dresidual, args_list)
      }

      # Set function default parameter values
      formals(object@dresidual)$param <- stored_param
    }

    # Create presidual function (same pattern)
    if (!is.null(presidual)) {
      object@presidual <- function(q, param = stored_param) {
        args_list <- list(q = q)
        fun_args <- methods::formalArgs(presidual)

        for (arg_name in names(param)) {
          if (arg_name %in% fun_args) {
            args_list[[arg_name]] <- param[arg_name]
          }
        }

        do.call(presidual, args_list)
      }
      formals(object@presidual)$param <- stored_param
    }

    # Create qresidual function
    if (!is.null(qresidual)) {
      object@qresidual <- function(p, param = stored_param) {
        args_list <- list(p = p)
        fun_args <- methods::formalArgs(qresidual)

        for (arg_name in names(param)) {
          if (arg_name %in% fun_args) {
            args_list[[arg_name]] <- param[arg_name]
          }
        }

        do.call(qresidual, args_list)
      }
      formals(object@qresidual)$param <- stored_param
    }

    # Create rresidual function (only size argument required)
    if (!is.null(rresidual)) {
      object@rresidual <- function(size, param = stored_param) {
        args_list <- list(n = size)
        fun_args <- methods::formalArgs(rresidual)

        for (arg_name in names(param)) {
          if (arg_name %in% fun_args) {
            args_list[[arg_name]] <- param[arg_name]
          }
        }

        do.call(rresidual, args_list)
      }
      formals(object@rresidual)$param <- stored_param
    }

    return(object)
  }
)


setMethod("show", "hspec", function(object) {

  cat("An object of class \"hspec\" of ", object@dimens, "-dimensional Hawkes process",  "\n\n", sep='')

  if(!is.null(object@mu)){
    cat("Slot mu: \n")
    print(object@mu)
    cat("\n")
  }

  if(!is.null(object@alpha)){
    cat("Slot alpha: \n")
    print(object@alpha)
    cat("\n")
  }

  if(!is.null(object@beta)){
    cat("Slot beta: \n")
    print(object@beta)
    cat("\n")
  }

  if(!is.null(object@eta)){
    cat("Slot eta: \n")
    print(object@eta)
    cat("\n")
  }

  if(!is.null(object@impact)){
    cat("Slot impact: \n")
    print(object@impact)
    cat("\n")
  }

  if(!is.null(object@type_col_map)){
    cat("Slot type_col_map: \n")
    print(object@type_col_map)
    cat("\n")
  }

  if(!is.null(object@rmark)){
    cat("Slot rmark: \n")
    print(object@rmark)
    cat("\n")
  }

  if(!is.null(object@dmark)){
    cat("Slot dmark: \n")
    print(object@dmark)
    cat("\n")
  }

  if(!is.null(object@rresidual)){
    cat("Slot rresidual: \n")
    print(object@rresidual)
    cat("\n")
  }


  if(!is.null(object@dresidual)){
    cat("Slot dresidual: \n")
    print(object@dresidual)
    cat("\n")
  }

  if(!is.null(object@presidual)){
    cat("Slot presidual: \n")
    print(object@presidual)
    cat("\n")
  }

  if(!is.null(object@qresidual)){
    cat("Slot qresidual: \n")
    print(object@qresidual)
    cat("\n")
  }

})
