% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit_one_layer}
\alias{fit_one_layer}
\title{MCMC sampling for one layer GP}
\usage{
fit_one_layer(
  x,
  y,
  nmcmc = 10000,
  sep = FALSE,
  verb = TRUE,
  g_0 = 0.001,
  theta_0 = 0.1,
  true_g = NULL,
  settings = NULL,
  cov = c("matern", "exp2"),
  v = 2.5,
  vecchia = FALSE,
  m = min(25, length(y) - 1),
  ordering = NULL
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{nmcmc}{number of MCMC iterations}

\item{sep}{logical indicating whether to use separable (\code{sep = TRUE})
or isotropic (\code{sep = FALSE}) lengthscales}

\item{verb}{logical indicating whether to print iteration progress}

\item{g_0}{initial value for \code{g}}

\item{theta_0}{initial value for \code{theta}}

\item{true_g}{if true nugget is known it may be specified here (set to a 
small value to make fit deterministic).  Note - values that are too 
small may cause numerical issues in matrix inversions.}

\item{settings}{hyperparameters for proposals and priors (see details)}

\item{cov}{covariance kernel, either Matern or squared exponential 
(\code{"exp2"})}

\item{v}{Matern smoothness parameter (only used if \code{cov = "matern"})}

\item{vecchia}{logical indicating whether to use Vecchia approximation}

\item{m}{size of Vecchia conditioning sets (only used if 
\code{vecchia = TRUE})}

\item{ordering}{optional ordering for Vecchia approximation, must correspond
to rows of \code{x}, defaults to random}
}
\value{
a list of the S3 class \code{gp} or \code{gpvec} with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{settings}: copy of proposal/prior settings
  \item \code{v}: copy of Matern smoothness parameter (\code{v = 999} 
        indicates \code{cov = "exp2"})
  \item \code{g}: vector of MCMC samples for \code{g}
  \item \code{theta}: vector of MCMC samples for \code{theta}
  \item \code{tau2}: vector of MLE estimates for \code{tau2} 
        (scale parameter)
  \item \code{ll}: vector of MVN log likelihood for each Gibbs iteration
  \item \code{time}: computation time in seconds
}
}
\description{
Conducts MCMC sampling of hyperparameters for a one layer 
    GP.  Length scale parameter \code{theta} governs 
    the strength of the correlation and nugget parameter \code{g} 
    governs noise.  In Matern covariance, \code{v} governs smoothness.
}
\details{
Utilizes Metropolis Hastings sampling of the length scale and
    nugget parameters with proposals and priors controlled by 
    \code{settings}.  When \code{true_g} is set to a specific value, the 
    nugget is not estimated.  When \code{vecchia = TRUE}, all calculations 
    leverage the Vecchia approximation with specified conditioning set size 
    \code{m}.  Vecchia approximation is only implemented for 
    \code{cov = "matern"}.
    
    NOTE on OpenMP: The Vecchia implementation relies on OpenMP parallelization
    for efficient computation.  This function will produce a warning message 
    if the package was installed without OpenMP (this is the default for 
    CRAN packages installed on Apple machines).  To set up OpenMP 
    parallelization, download the package source code and install 
    using the gcc/g++ compiler.  
    
    Proposals for \code{g} and \code{theta} follow a uniform sliding window 
    scheme, e.g. 
    
    \code{g_star <- runif(1, l * g_t / u, u * g_t / l)}, 
    
    with defaults \code{l = 1} and \code{u = 2} provided in \code{settings}.
    To adjust these, set \code{settings = list(l = new_l, u = new_u)}.
    Priors on \code{g} and \code{theta} follow Gamma distributions with 
    shape parameters (\code{alpha}) and rate parameters (\code{beta}) 
    controlled within the \code{settings} list object.  
    Defaults have been updated with package version 1.1.3.  Default priors differ
    for noisy/deterministic settings and depend on whether \code{monowarp = TRUE}.  
    All default values are visible in the internal
    \code{deepgp:::check_settings} function.
    These priors are designed for \code{x} scaled 
    to [0, 1] and \code{y} scaled to have mean 0 and variance 1.  These may
    be adjusted using the \code{settings} input.
    
    The output object of class \code{gp} is designed for use with 
    \code{continue}, \code{trim}, and \code{predict}.
}
\examples{
# Additional examples including real-world computer experiments are available at: 
# https://bitbucket.org/gramacylab/deepgp-ex/
\donttest{
# Booth function (inspired by the Higdon function)
f <- function(x) {
  i <- which(x <= 0.58)
  x[i] <- sin(pi * x[i] * 6) + cos(pi * x[i] * 12)
  x[-i] <- 5 * x[-i] - 4.9
  return(x)
}

# Training data
x <- seq(0, 1, length = 25)
y <- f(x)

# Testing data
xx <- seq(0, 1, length = 200)
yy <- f(xx)

plot(xx, yy, type = "l")
points(x, y, col = 2)

# Example 1: full model (nugget fixed)
fit <- fit_one_layer(x, y, nmcmc = 2000, true_g = 1e-6)
plot(fit)
fit <- trim(fit, 1000, 2)
fit <- predict(fit, xx, cores = 1)
plot(fit)

# Example 2: full model (nugget estimated, EI calculated)
fit <- fit_one_layer(x, y, nmcmc = 2000)
plot(fit) 
fit <- trim(fit, 1000, 2)
fit <- predict(fit, xx, cores = 1, EI = TRUE)
plot(fit)
par(new = TRUE) # overlay EI
plot(xx[order(xx)], fit$EI[order(xx)], type = 'l', lty = 2, 
axes = FALSE, xlab = '', ylab = '')

# Example 3: Vecchia approximated model (nugget estimated)
fit <- fit_one_layer(x, y, nmcmc = 2000, vecchia = TRUE, m = 10)
plot(fit)
fit <- trim(fit, 1000, 2)
fit <- predict(fit, xx, cores = 1)
plot(fit)
}

}
\references{
Sauer, A. (2023). Deep Gaussian process surrogates for computer experiments. 
    *Ph.D. Dissertation, Department of Statistics, Virginia Polytechnic Institute and State University.*
     \cr\cr
Sauer, A., Gramacy, R.B., & Higdon, D. (2023). Active learning for deep 
    Gaussian process surrogates. *Technometrics, 65,* 4-18.  arXiv:2012.08015
    \cr\cr
Sauer, A., Cooper, A., & Gramacy, R. B. (2023). Vecchia-approximated deep Gaussian 
   processes for computer experiments. 
   *Journal of Computational and Graphical Statistics, 32*(3), 824-837.  arXiv:2204.02904
}
