% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_histogram_STRAPP_test_for_focal_time.R
\name{plot_histogram_STRAPP_test_for_focal_time}
\alias{plot_histogram_STRAPP_test_for_focal_time}
\title{Plot histogram of STRAPP test statistics to assess results}
\usage{
plot_histogram_STRAPP_test_for_focal_time(
  deepSTRAPP_outputs,
  focal_time = NULL,
  display_plot = TRUE,
  plot_posthoc_tests = FALSE,
  PDF_file_path = NULL
)
}
\arguments{
\item{deepSTRAPP_outputs}{List of elements generated with \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}},
that summarize the results of a STRAPP test for a specific time in the past (i.e. the \code{focal_time}).
\code{deepSTRAPP_outputs} can also be extracted from the output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}} that
run the whole deepSTRAPP workflow and store results inside \verb{$STRAPP_results_over_time}.}

\item{focal_time}{Numerical. (Optional) If \code{deepSTRAPP_outputs} comprises results over multiple time-steps
(i.e., output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}}, this is the time of the STRAPP test targeted for plotting.}

\item{display_plot}{Logical. Whether to display the histogram(s) generated in the R console. Default is \code{TRUE}.}

\item{plot_posthoc_tests}{Logical. For multinominal data only. Whether to plot the histogram for the overall Kruskal-Wallis test across all states (\code{plot_posthoc_tests = FALSE}),
or plot the histograms for all the pairwise post hoc Dunn's test across pairs of states (\code{plot_posthoc_tests = TRUE}). Default is \code{FALSE}.}

\item{PDF_file_path}{Character string. If provided, the plot will be saved in a PDF file following the path provided here. The path must end with ".pdf".}
}
\value{
By default, the function returns a list of classes \code{gg} and \code{ggplot}.
This object is a ggplot that can be displayed on the console with \code{print(output)}.
It corresponds to the histogram being displayed on the console when the function is run, if \code{display_plot = TRUE}, and can be further
modify for aesthetics using the ggplot2 grammar.

If using multinominal data and set \code{plot_posthoc_tests = TRUE}, the function will return a list of objects.
Each object is the ggplot associated with a pairwise post hoc test.
To plot each histogram \code{i} individually, use \code{print(output_list[[i]])}.
To plot all histograms at once in a multifaceted plot, as displayed on the console if \code{display_plot = TRUE}, use \code{cowplot::plot_grid(plotlist = output_list)}.

Each plot also displays summary statistics for the STRAPP test associated with the data displayed.
\itemize{
\item The quantile of null statistic distribution at the significant threshold used to define test significance. This is the value found on the red dashed line.
The test will be considered significant (i.e., the null hypothesis is rejected) if this value is higher than zero (the black dashed line).
\item The p-value of the STRAPP test which correspond the proportion of cases in which the statistics was lower than expected under the null hypothesis
(i.e., the proportion of the histogram found below / on the left-side of the black dashed line).
}

If a \code{PDF_file_path} is provided, the function will also generate a PDF file of the plot. For post hoc tests, this will save the multifaceted plot.
}
\description{
Plot an histogram of the distribution of the test statistics
obtained from a STRAPP test carried out for a unique \code{focal_time}.
(See \code{\link[=compute_STRAPP_test_for_focal_time]{compute_STRAPP_test_for_focal_time()}} and
\code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}).

Returns a single histogram for overall tests.
If \code{plot_posthoc_tests = TRUE}, it will return a faceted plot with an histogram per post hoc tests.

If a PDF file path is provided in \code{PDF_file_path}, the plot will be saved directly in a PDF file.
}
\details{
The main input \code{deepSTRAPP_outputs} is the typical output of \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}.
It provides information on results of a STRAPP test performed at a given \code{focal_time}.

Histograms are built based on the distribution of the test statistics.
Such distributions are recorded in the outputs of a deepSTRAPP run carried out with \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}
when \code{return_perm_data = TRUE} so that the distributions of test stats computed across posterior samples are returned
among the outputs under \verb{$STRAPP_results$perm_data_df}.

For multinominal data (categorical or biogeographic data with more than 2 states), it is possible to plot the histograms of post hoc pairwise tests.
Set \code{plot_posthoc_tests = TRUE} to generate histograms for all the pairwise post hoc Dunn's test across pairs of states.
To achieve this, the \code{deepSTRAPP_outputs} input object must contain a \verb{$STRAPP_results$posthoc_pairwise_tests$perm_data_array} element
that summarizes test statistics computed across posterior samples for all pairwise post hoc tests.
This is obtained from \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} when setting both \code{posthoc_pairwise_tests = TRUE} to carry out post hoc tests,
and \code{return_perm_data = TRUE} to record distributions of test statistics.

Alternatively, the main input \code{deepSTRAPP_outputs} can be the output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}},
providing results of STRAPP tests over multiple time-steps. In this case, you must provide a \code{focal_time} to select the
unique time-step used for plotting.
\itemize{
\item \code{return_perm_data} must be set to \code{TRUE} so that the permutation data used to compute the tests are returned among the outputs
under \verb{$STRAPP_results_over_time[[i]]$perm_data_df}.
\item \code{posthoc_pairwise_tests} must be set to \code{TRUE} so that the permutation data used to performed the post hoc tests are also
returned among the outputs under \verb{$STRAPP_results_over_time[[i]]$posthoc_pairwise_tests$perm_data_array}.
}

For plotting all time-steps at once, see \code{\link[=plot_histograms_STRAPP_tests_over_time]{plot_histograms_STRAPP_tests_over_time()}}.
}
\examples{
if (deepSTRAPP::is_dev_version())
{
 # ----- Example 1: Continuous trait ----- #

 # Load fake trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")
 # Load phylogeny with old calibration
 data(Ponerinae_tree_old_calib, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract continuous trait data as a named vector
 Ponerinae_cont_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cont_tip_data,
                                     nm = Ponerinae_trait_tip_data$Taxa)

 # Select a color scheme from lowest to highest values
 color_scale = c("darkgreen", "limegreen", "orange", "red")

 # Get Ancestral Character Estimates based on a Brownian Motion model
 # To obtain values at internal nodes
 Ponerinae_ACE <- phytools::fastAnc(tree = Ponerinae_tree_old_calib, x = Ponerinae_cont_tip_data)

 \donttest{ # (May take several minutes to run)
 # Run a Stochastic Mapping based on a Brownian Motion model
 # to interpolate values along branches and obtain a "contMap" object
 Ponerinae_contMap <- phytools::contMap(Ponerinae_tree, x = Ponerinae_cont_tip_data,
                                        res = 100, # Number of time steps
                                        plot = FALSE)
 # Plot contMap = stochastic mapping of continuous trait
 plot_contMap(contMap = Ponerinae_contMap,
              color_scale = color_scale)

 ## Set focal time to 10 Mya
 focal_time <- 10

 ## Run deepSTRAPP on net diversification rates for focal time = 10 Mya.

 Ponerinae_deepSTRAPP_cont_old_calib_10My <- run_deepSTRAPP_for_focal_time(
   contMap = Ponerinae_contMap,
   ace = Ponerinae_ACE,
   tip_data = Ponerinae_cont_tip_data,
   trait_data_type = "continuous",
   BAMM_object = Ponerinae_BAMM_object_old_calib,
   focal_time = focal_time,
   rate_type = "net_diversification",
   return_perm_data = TRUE,
   extract_diversification_data_melted_df = TRUE,
   return_updated_trait_data_with_Map = TRUE,
   return_updated_BAMM_object = TRUE)

 ## Explore output
 str(Ponerinae_deepSTRAPP_cont_old_calib_10My, max.level = 1)

 # ----- Plot histogram of STRAPP overall test results from run_deepSTRAPP_for_focal_time() ----- #

 histogram_ggplot <- plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_10My,
    display_plot = TRUE,
    # PDF_file_path = "./plot_STRAPP_histogram_10My.pdf",
    plot_posthoc_tests = FALSE)
 # Adjust aesthetics a posteriori
 histogram_ggplot_adj <- histogram_ggplot +
    ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(histogram_ggplot_adj) }

 # ----- Plot histogram of STRAPP overall test results from run_deepSTRAPP_over_time() ----- #

 ## Load directly outputs from run_deepSTRAPP_over_time()
 data(Ponerinae_deepSTRAPP_cont_old_calib_0_40, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 # Select focal_time = 10My
 focal_time <- 10

 ## Plot histogram for overall test
 histogram_ggplot <- plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    focal_time = focal_time,
    display_plot = TRUE,
    # PDF_file_path = "./plot_STRAPP_histogram_10My.pdf",
    plot_posthoc_tests = FALSE)
 # Adjust aesthetics a posteriori
 histogram_ggplot_adj <- histogram_ggplot +
   ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(histogram_ggplot_adj)

 # ----- Example 2: Categorical trait ----- #

 ## Load data

 # Load phylogeny
 data(Ponerinae_tree, package = "deepSTRAPP")
 # Load trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract categorical data with 3-levels
 Ponerinae_cat_3lvl_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cat_3lvl_tip_data,
                                         nm = Ponerinae_trait_tip_data$Taxa)
 table(Ponerinae_cat_3lvl_tip_data)

 # Select color scheme for states
 colors_per_states <- c("forestgreen", "sienna", "goldenrod")
 names(colors_per_states) <- c("arboreal", "subterranean", "terricolous")

 \donttest{ # (May take several minutes to run)
 ## Produce densityMaps using stochastic character mapping based on an ARD Mk model
 Ponerinae_cat_3lvl_data_old_calib <- prepare_trait_data(
    tip_data = Ponerinae_cat_3lvl_tip_data,
    phylo = Ponerinae_tree_old_calib,
    trait_data_type = "categorical",
    colors_per_states = colors_per_states,
    evolutionary_models = "ARD", # Use default ARD model
    nb_simulations = 100, # Reduce number of simulations to save time
    seed = 1234, # Seet seed for reproducibility
    return_best_model_fit = TRUE,
    return_model_selection_df = TRUE,
    plot_map = FALSE) }

 # Load directly output
 data(Ponerinae_cat_3lvl_data_old_calib, package = "deepSTRAPP")

 ## Set focal time to 10 Mya
 focal_time <- 10

 \donttest{ # (May take several minutes to run)
 ## Run deepSTRAPP on net diversification rates for focal time = 10 Mya.

 Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My <- run_deepSTRAPP_for_focal_time(
    densityMaps = Ponerinae_cat_3lvl_data_old_calib$densityMaps,
    ace = Ponerinae_cat_3lvl_data_old_calib$ace,
    tip_data = Ponerinae_cat_3lvl_tip_data,
    trait_data_type = "categorical",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    focal_time = focal_time,
    rate_type = "net_diversification",
    posthoc_pairwise_tests = TRUE,
    return_perm_data = TRUE,
    extract_diversification_data_melted_df = TRUE,
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE)

 ## Explore output
 str(Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My, max.level = 1)

 # ------ Plot histogram of STRAPP overall test results ------ #

 histogram_ggplot <- plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My,
    display_plot = TRUE,
    # PDF_file_path = "./plot_STRAPP_histogram_overall_test.pdf",
    plot_posthoc_tests = FALSE)
 # Adjust aesthetics a posteriori
 histogram_ggplot_adj <- histogram_ggplot +
    ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(histogram_ggplot_adj)

 # ------ Plot histograms of STRAPP post hoc test results ------ #

 histograms_ggplot_list <- plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My,
    display_plot = TRUE,
    # PDF_file_path = "./plot_STRAPP_histograms_posthoc_tests.pdf",
    plot_posthoc_tests = TRUE)
 # Plot all histograms one by one
 print(histograms_ggplot_list)
 # Plot all histograms on one faceted plot
 cowplot::plot_grid(plotlist = histograms_ggplot_list) }
}

}
\seealso{
Associated functions in deepSTRAPP: \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} \code{\link[=plot_histograms_STRAPP_tests_over_time]{plot_histograms_STRAPP_tests_over_time()}}
}
\author{
Maël Doré
}
