% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interval_dist.R
\name{interval_distance}
\alias{interval_distance}
\alias{int_dist}
\alias{int_dist_matrix}
\alias{int_pairwise_dist}
\alias{int_dist_all}
\title{Distance Measures for Interval Data}
\usage{
int_dist(x, method = "euclidean", gamma = 0.5, q = 1, p = 2, ...)

int_dist_matrix(x, method = "euclidean", gamma = 0.5, q = 1, p = 2, ...)

int_pairwise_dist(x, var_name1, var_name2, method = "euclidean", ...)

int_dist_all(x, gamma = 0.5, q = 1)
}
\arguments{
\item{x}{interval-valued data with symbolic_tbl class, or an array of dimension [n, p, 2]}

\item{method}{distance method: "GD", "IY", "L1", "L2", "CB", "HD", "EHD", "nEHD", "snEHD", "TD", "WD", "euclidean", "hausdorff", "manhattan", "city_block", "minkowski", "wasserstein", "ichino", "de_carvalho"}

\item{gamma}{parameter for the Ichino-Yaguchi distance, 0 <= gamma <= 0.5 (default: 0.5)}

\item{q}{parameter for the Ichino-Yaguchi distance (Minkowski exponent) (default: 1)}

\item{p}{power parameter for Minkowski distance (default: 2)}

\item{...}{additional parameters}

\item{var_name1}{first variable name or column location}

\item{var_name2}{second variable name or column location}
}
\value{
A distance matrix (class 'dist') or numeric vector
}
\description{
Functions to compute various distance measures between interval-valued observations.

\code{int_dist_all} computes all available distance measures at once.
}
\details{
Available distance methods:
\itemize{
  \item \code{GD}: Gowda-Diday distance (Gowda & Diday, 1991)
  \item \code{IY}: Ichino-Yaguchi distance (Ichino, 1988)
  \item \code{L1}: L1 (midpoint Manhattan) distance
  \item \code{L2}: L2 (Euclidean midpoint) distance
  \item \code{CB}: City-Block distance (Souza & de Carvalho, 2004)
  \item \code{HD}: Hausdorff distance (Chavent & Lechevallier, 2002)
  \item \code{EHD}: Euclidean Hausdorff distance
  \item \code{nEHD}: Normalized Euclidean Hausdorff distance
  \item \code{snEHD}: Span Normalized Euclidean Hausdorff distance
  \item \code{TD}: Tran-Duckstein distance (Tran & Duckstein, 2002)
  \item \code{WD}: L2-Wasserstein distance (Verde & Irpino, 2008)
  \item \code{euclidean}: Euclidean distance on interval centers (same as L2)
  \item \code{hausdorff}: Hausdorff distance (same as HD)
  \item \code{manhattan}: Manhattan distance (same as L1)
  \item \code{city_block}: City-block distance (same as CB)
  \item \code{minkowski}: Minkowski distance with parameter p
  \item \code{wasserstein}: Wasserstein distance (same as WD)
  \item \code{ichino}: Ichino-Yaguchi distance (simplified version)
  \item \code{de_carvalho}: De Carvalho distance
}
}
\examples{
# Using symbolic_tbl format
data(mushroom.int)
d1 <- int_dist(mushroom.int[, 3:4], method = "euclidean")
d2 <- int_dist(mushroom.int[, 3:4], method = "hausdorff")
d3 <- int_dist(mushroom.int[, 3:4], method = "GD")

# Using array format: 4 concepts, 3 variables
x <- array(NA, dim = c(4, 3, 2))
x[,,1] <- matrix(c(1,2,3,4, 5,6,7,8, 9,10,11,12), nrow=4)
x[,,2] <- matrix(c(3,5,6,7, 8,9,10,12, 13,15,16,18), nrow=4)
d4 <- int_dist(x, method = "snEHD")
d5 <- int_dist(x, method = "IY", gamma = 0.3)
}
\references{
Gowda, K. C., & Diday, E. (1991). Symbolic clustering using a new dissimilarity measure. 
\emph{Pattern Recognition}, 24(6), 567-578.

Ichino, M. (1988). General metrics for mixed features. 
\emph{Systems and Computers in Japan}, 19(2), 37-50.

Chavent, M., & Lechevallier, Y. (2002). Dynamical clustering of interval data. 
In \emph{Classification, Clustering and Data Analysis} (pp. 53-60). Springer.

Tran, L., & Duckstein, L. (2002). Comparison of fuzzy numbers using a fuzzy distance measure.
\emph{Fuzzy Sets and Systems}, 130, 331-341.

Verde, R., & Irpino, A. (2008). A new interval data distance based on the Wasserstein metric.

Kao, C.-H. et al. (2014). Exploratory data analysis of interval-valued symbolic data with 
matrix visualization. \emph{CSDA}, 79, 14-29.
}
\seealso{
int_dist_matrix int_dist_all int_pairwise_dist
}
\author{
Han-Ming Wu
}
