% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm.R
\name{cpss.glm}
\alias{cpss.glm}
\title{Detecting changes in GLMs}
\usage{
cpss.glm(
  formula,
  family,
  data = NULL,
  algorithm = "BS",
  dist_min = floor(log(n)),
  ncps_max = ceiling(n^0.4),
  pelt_pen_val = NULL,
  pelt_K = 0,
  wbs_nintervals = 500,
  criterion = "CV",
  times = 2
)
}
\arguments{
\item{formula}{a \code{formula} object specifying the GLM with change-points.}

\item{family}{a description of the error distribution and link function to be used in the model, which can be a character string naming a family function or a family function.}

\item{data}{an optional data frame containing the variables in the model.}

\item{algorithm}{a character string specifying the change-point searching algorithm, one of the following choices: "SN" (segment neighborhood), "BS" (binary segmentation), "WBS" (wild binary segmentation) and "PELT" (pruned exact linear time) algorithms.}

\item{dist_min}{an integer specifying minimum searching distance (length of feasible segments).}

\item{ncps_max}{an integer specifying an upper bound of the number of true change-points.}

\item{pelt_pen_val}{a numeric vector specifying candidate values of the penalty only if \code{algorithm = "PELT"}.}

\item{pelt_K}{a numeric value for pruning adjustment only if \code{algorithm = "PELT"}. It is usually taken to be 0 if the negative log-likelihood is used as a cost, see Killick et al. (2012).}

\item{wbs_nintervals}{an integer specifying the number of random intervals drawn only if \code{algorithm = "WBS"}, see Fryzlewicz (2014).}

\item{criterion}{a character string specifying the model selection criterion, "CV" ("cross-validation") or "MS" ("multiple-splitting").}

\item{times}{an integer specifying how many times of sample-splitting should be performed; It should be 2 if \code{criterion = "CV"}.}
}
\value{
\code{cpss.glm} returns an object of an \proglang{S4} class, called "\code{cpss}", which collects data and information required for further change-point analyses and summaries. See \code{\link{cpss.custom}}.
}
\description{
Detecting changes in GLMs
}
\examples{
library("cpss")
set.seed(666)
n <- 200
size <- rpois(n, 20 - 1) + 1
tau <- c(75, 100, 175)
tau_ext <- c(0, tau, n)
be <- list(c(0, 0.5), c(0, -0.5), c(0.5, -0.5), c(-0.5, -0.5))
seg_len <- diff(c(0, tau, n))
x <- rnorm(n)
eta <- lapply(seq(1, length(tau) + 1), function(k) {
  be[[k]][1] + be[[k]][2] * x[(tau_ext[k] + 1):tau_ext[k + 1]]
})
eta <- do.call(c, eta)
p <- 1 / (1 + exp(-eta))
y <- rbinom(n, size = size, prob = p)
\donttest{
pelt_pen_val <- (log(n))^seq(0.5, 2, by = 0.1)
res <- cpss.glm(
  formula = cbind(y, size - y) ~ x, family = binomial(),
  algorithm = "PELT", pelt_pen_val = pelt_pen_val, ncps_max = 10
)
summary(res)
# 75  105  175
coef(res)
# [1,] 0.02540872  0.08389551  0.5284425 -0.4980768
# [2,] 0.57222684 -0.45430385 -0.5203319 -0.4581678
}
}
\references{
Killick, R., Fearnhead, P., and Eckley, I. A. (2012). Optimal Detection of Changepoints With a Linear Computational Cost. Journal of the American Statistical Association, 107(500):1590–1598.

Fryzlewicz, P. (2014). Wild binary segmentation for multiple change-point detection. The Annals of Statistics, 42(6): 2243–2281.
}
\seealso{
\code{\link{cpss.lm}}
}
