% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tetrad-search.R
\docType{class}
\name{TetradSearch}
\alias{TetradSearch}
\title{R6 Interface to Tetrad Search Algorithms}
\description{
High-level wrapper around the Java-based \pkg{Tetrad} causal-discovery
library. The class lets you choose independence tests, scores, and search
algorithms from \pkg{Tetrad}, run them on an R data set, and retrieve the
resulting graph or statistics.
}
\examples{
### tetrad_search R6 class examples ###

# Generally, we do not recommend using the R6 classes directly, but rather
# use the disco() or any method function, for example pc(), instead.

# Requires Tetrad to be installed
if (verify_tetrad()$installed && verify_tetrad()$java_ok) {
  data(num_data)

  # Recommended:
  my_pc <- pc(engine = "tetrad", test = "fisher_z")
  my_pc(num_data)

  # or
  disco(data = num_data, method = my_pc)

  # Example with detailed settings:
  my_pc2 <- pc(
    engine = "tetrad",
    test = "sem_bic",
    penalty_discount = 1,
    structure_prior = 1,
    singularity_lambda = 0.1
  )
  disco(data = num_data, method = my_pc2)

  # Using R6 class:
  s <- TetradSearch$new()

  s$set_data(num_data)
  s$set_test(method = "fisher_z", alpha = 0.05)
  s$set_alg("pc")

  g <- s$run_search()

  print(g)
}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{data}}{Java object that stores the (possibly converted) data set
used by \pkg{Tetrad}.}

\item{\code{rdata}}{Original \strong{R} \code{data.frame} supplied by the user.}

\item{\code{score}}{Java object holding the scoring function selected with
\code{set_score()}. Supply one of the method strings for
\code{set_score()}. Recognised values are:

\strong{Continuous - Gaussian}
\itemize{
\item \code{"ebic"} - Extended BIC score.
\item \code{"gic"} - Generalized Information Criterion (GIC) score.
\item \code{"poisson_prior"} - Poisson prior score.'
\item \code{"rank_bic"} - Rank-based BIC score.
\item \code{"sem_bic"} - SEM BIC score.
\item \code{"zhang_shen_bound"} - Zhang and Shen bound score.
}

\strong{Discrete - categorical}
\itemize{
\item \code{"bdeu"} - Bayes Dirichlet Equivalent score with uniform priors.
\item \code{"discrete_bic"} - BIC score for discrete data.
}

\strong{Mixed Discrete/Gaussian}
\itemize{
\item \code{"basis_function_bic"} - BIC score for basis-function models.
This is a generalization of the Degenerate Gaussian score.
\item \code{"basis_function_blocks_bic"} - BIC score for mixed data using basis-function models.
\item \code{"basis_function_sem_bic"} - SEM BIC score for basis-function models.
\item \code{"conditional_gaussian"} - Conditional Gaussian BIC score.
\item \code{"degenerate_gaussian"} - Degenerate Gaussian BIC score.
\item \code{"mag_degenerate_gaussian_bic"} - MAG Degenerate Gaussian BIC Score.
}}

\item{\code{test}}{Java object holding the independence test selected with
\code{set_test()}. Supply one of the method strings for
\code{set_test()}. Recognised values are:

\strong{Continuous - Gaussian}
\itemize{
\item \code{"fisher_z"} - Fisher \eqn{Z} (partial correlation) test.
\item \code{"poisson_prior"} - Poisson prior test.
\item \code{"rank_independence"} - Rank-based independence test.
\item \code{"sem_bic"} - SEM BIC test.
}

\strong{Discrete - categorical}
\itemize{
\item \code{"chi_square"} - chi-squared test
\item \code{"g_square"} - likelihood-ratio \eqn{G^2} test.
\item \code{"probabilistic"} - Uses BCInference by Cooper and Bui to calculate
probabilistic conditional independence judgments.
}

\strong{General}
\itemize{
\item \code{"gin"} - Generalized Independence Noise test.
\item \code{"kci"} - Kernel Conditional Independence Test (KCI) by Kun Zhang.
\item \code{"rcit"} - Randomized Conditional Independence Test (RCIT).
}

\strong{Mixed Discrete/Gaussian}
\itemize{
\item \code{"basis_function_blocks"} - Basis-function blocks test.
\item \code{"basis_function_lrt"} - basis-function likelihood-ratio.
\item \code{"conditional_gaussian"} - Conditional Gaussian test as a likelihood ratio test.
\item \code{"degenerate_gaussian"} - Degenerate Gaussian test as a likelihood ratio test.
}}

\item{\code{alg}}{Java object representing the search algorithm.
Supply one of the method strings for \code{set_alg()}.
Recognised values are:

\strong{Constraint-based}
\itemize{
\item \code{"fci"} - FCI algorithm. See \code{\link[=fci]{fci()}}.
\item \code{"pc"} - Peter-Clark (PC) algorithm. See \code{\link[=pc]{pc()}}.
\item \code{"rfci"} - Restricted FCI algorithm. See \code{\link[pcalg:rfci]{pcalg::rfci()}}.
}

\strong{Hybrid}
\itemize{
\item \code{"boss_fci"} - BOSS-FCI algorithm. See \code{\link[=boss_fci]{boss_fci()}}.
\item \code{"gfci"} - GFCI algorithm. See \code{\link[=gfci]{gfci()}}.
\item \code{"grasp_fci"} - GRaSP-FCI algorithm. See \code{\link[=grasp_fci]{grasp_fci()}}.
\item \code{"sp_fci"} - Sparsest Permutation using FCI. See \code{\link[=sp_fci]{sp_fci()}}.
}

\strong{Score-based}
\itemize{
\item \code{"boss"} - BOSS algorithm. See \code{\link[=boss]{boss()}}.
\item \code{"ges" ("fges")} - (Fast) Greedy Equivalence Search (GES) algorithm. See \code{\link[=ges]{ges()}}.
\item \code{"grasp"} - GRaSP (Greedy Relations of Sparsest Permutation) algorithm. See \code{\link[=grasp]{grasp()}}.
}}

\item{\code{mc_test}}{Java independence-test object used by the Markov checker.}

\item{\code{java}}{Java object returned by the search (typically a graph).}

\item{\code{result}}{Convenience alias for \code{java}; may store additional
metadata depending on the search type.}

\item{\code{knowledge}}{Java \code{Knowledge} object carrying background
constraints (required/forbidden edges).}

\item{\code{params}}{Java \code{Parameters} object holding algorithm settings.}

\item{\code{bootstrap_graphs}}{Java \code{List} of graphs produced by bootstrap
resampling, if that feature was requested.}

\item{\code{mc_ind_results}}{Java \code{List} with Markov-checker test results.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-TetradSearch-new}{\code{TetradSearch$new()}}
\item \href{#method-TetradSearch-set_test}{\code{TetradSearch$set_test()}}
\item \href{#method-TetradSearch-set_score}{\code{TetradSearch$set_score()}}
\item \href{#method-TetradSearch-set_alg}{\code{TetradSearch$set_alg()}}
\item \href{#method-TetradSearch-set_knowledge}{\code{TetradSearch$set_knowledge()}}
\item \href{#method-TetradSearch-set_params}{\code{TetradSearch$set_params()}}
\item \href{#method-TetradSearch-get_parameters_for_function}{\code{TetradSearch$get_parameters_for_function()}}
\item \href{#method-TetradSearch-run_search}{\code{TetradSearch$run_search()}}
\item \href{#method-TetradSearch-set_bootstrapping}{\code{TetradSearch$set_bootstrapping()}}
\item \href{#method-TetradSearch-set_data}{\code{TetradSearch$set_data()}}
\item \href{#method-TetradSearch-set_verbose}{\code{TetradSearch$set_verbose()}}
\item \href{#method-TetradSearch-set_time_lag}{\code{TetradSearch$set_time_lag()}}
\item \href{#method-TetradSearch-get_data}{\code{TetradSearch$get_data()}}
\item \href{#method-TetradSearch-get_knowledge}{\code{TetradSearch$get_knowledge()}}
\item \href{#method-TetradSearch-get_java}{\code{TetradSearch$get_java()}}
\item \href{#method-TetradSearch-get_string}{\code{TetradSearch$get_string()}}
\item \href{#method-TetradSearch-get_dot}{\code{TetradSearch$get_dot()}}
\item \href{#method-TetradSearch-get_amat}{\code{TetradSearch$get_amat()}}
\item \href{#method-TetradSearch-clone}{\code{TetradSearch$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-new"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-new}{}}}
\subsection{Method \code{new()}}{
Initializes the \code{TetradSearch} object, creating new Java objects for
\code{knowledge} and \code{params}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_test"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_test}{}}}
\subsection{Method \code{set_test()}}{
Sets the independence test to use in Tetrad.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_test(method, ..., use_for_mc = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{(character) Name of the test method (e.g., "chi_square", "fisher_z").
\itemize{
\item \code{"basis_function_blocks"} - Basis-function blocks test
\item \code{"basis_function_lrt"} - basis-function likelihood-ratio
\item \code{"chi_square"} - chi-squared test
\item \code{"conditional_gaussian"} - Mixed discrete/continuous test
\item \code{"degenerate_gaussian"} - Degenerate Gaussian test as a likelihood ratio test
\item \code{"fisher_z"} - Fisher \(Z\) (partial correlation) test
\item \code{"gin"} - Generalized Independence Noise test
\item \code{"kci"} - Kernel Conditional Independence Test (KCI) by Kun Zhang
\item \code{"poisson_prior"} - Poisson prior test
\item \code{"probabilistic"} - Uses BCInference by Cooper and Bui to calculate
probabilistic conditional independence judgments.
\item \code{"rcit"} - Randomized Conditional Independence Test (RCIT)
\item \code{"rank_independence"} - Rank-based independence test
\item \code{"sem_bic"} - SEM BIC test
}}

\item{\code{...}}{Additional arguments passed to the private test-setting methods.
For the following tests, the following parameters are available:
\itemize{
\item \code{"basis_function_blocks"} - Basis-function blocks test.
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{basis_type = "polynomial"} - The type of basis to use. Supported
types are \code{"polynomial"}, \code{"legendre"}, \code{"hermite"}, and
\code{"chebyshev"},
\item \code{truncation_limit = 3} - Basis functions 1 through
this number will be used. The Degenerate Gaussian category
indicator variables for mixed data are also used.
}
\item \code{"basis_function_lrt"} - basis-function likelihood-ratio
\itemize{
\item \code{truncation_limit = 3} - Basis functions 1 through
this number will be used. The Degenerate Gaussian category
indicator variables for mixed data are also used,
\item \code{alpha = 0.05} - Significance level for the
likelihood-ratio test,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse,
\item \code{do_one_equation_only = FALSE} - If TRUE, only one
equation should be used when expanding the basis.
}
\item \code{"chi_square"} - chi-squared test
\itemize{
\item \code{min_count = 1} - Minimum count for the chi-squared
test per cell. Increasing this can improve accuracy of the test
estimates,
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{cell_table_type = "ad"} - The type of cell table to
use for optimization. Available types are:
\code{"ad"} - AD tree, \code{"count"} - Count sample.
}
\item \code{"conditional_gaussian"} - Mixed discrete/continuous test
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{discretize = TRUE} - If TRUE for the conditional
Gaussian likelihood, when scoring X --> D where X is continuous
and D discrete, one should to simply discretize X for just
those cases.
If FALSE, the integration will be exact,
\item \code{num_categories_to_discretize = 3} - In case the exact
algorithm is not used for discrete children and continuous
parents is not used, this parameter gives the number of
categories to use for this second (discretized) backup copy of
the continuous variables,
\item \code{min_sample_size_per_cell = 4} - Minimum sample size
per cell for the independence test.
}
\item \code{"degenerate_gaussian"} - Degenerate Gaussian
likelihood ratio test
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"fisher_z"} - Fisher \(Z\) (partial correlation) test
\itemize{
\item \code{alpha = 0.05} - Significance level for the independence test,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"gin"} - Generalized Independence Noise test.
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{gin_backend = "dcor"} - Unconditional test for residual
independence. Available types are \code{"dcor"} - Distance correlation (for non-linear)
and \code{"pearson"} - Pearson correlation (for linear),
\item \code{num_permutations = 200} - Number of permutations used for
\code{"dcor"} backend. If \code{"pearson"} backend is used, this parameter is ignored.
\item \code{gin_ridge = 1e-8} - Ridge parameter used when computing residuals.
A small number >= 0.
\item \code{seed = -1} - Random seed for the independence test. If -1, no seed is set.
}
\item \code{"kci"} - Kernel Conditional Independence Test (KCI) by Kun Zhang
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{approximate = TRUE} - If TRUE, use the approximate
Gamma approximation algorithm. If FALSE, use the exact,
\item \code{scaling_factor = 1} - For Gaussian kernel: The
scaling factor * Silverman bandwidth.
\item \code{num_bootstraps = 1000} - Number of bootstrap
samples to use for the KCI test. Only used if \code{approximate = FALSE}.
\item \code{threshold = 1e-3} - Threshold for the KCI test.
Threshold to determine how many eigenvalues to use --
the lower the more (0 to 1).
\item \code{kernel_type = "gaussian"} - The type of kernel to
use. Available types are \code{"gaussian"}, \code{"linear"}, or
\code{"polynomial"}.
\item \code{polyd = 5} - The degree of the polynomial kernel,
if used.
\item \code{polyc = 1} - The constant of the polynomial kernel,
if used.
}
\item \code{"poisson_prior"} - Poisson prior test
\itemize{
\item \code{poisson_lambda = 1} - Lambda parameter for the Poisson
distribution (> 0),
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"probabilistic"} - Uses BCInference by Cooper and Bui
to calculate probabilistic conditional independence judgments.
\itemize{
\item \code{threshold = FALSE} - Set to TRUE if using the cutoff
threshold for the independence test,
\item \code{cutoff = 0.5} - Cutoff for the independence test,
\item \code{prior_ess = 10} - Prior equivalent sample size
for the independence test. This number is added to the sample
size for each conditional probability table in the model and is
divided equally among the cells in the table.
}
\item \code{"rcit"} - Randomized Conditional Independence Test (RCIT).
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test,
\item \code{rcit_approx = "lpb4"} - Null approximation method. Recognized values are:
\code{"lpb4"} - Lindsay-Pilla-Basak method with 4 support points,
\code{"hbe"} - Hall-Buckley-Eagleson method,
\code{"gamma"} - Gamma (Satterthwaite-Welch),
\code{"chi_square"} - Chi-square (normalized),
\code{"permutation"} - Permutation-based (computationally intensive),
\item \code{rcit_ridge = 1e-3} - Ridge parameter used when computing residuals.
A small number >= 0,
\item \code{num_feat = 10} - Number of random features to use
for the regression of X and Y on Z. Values between 5 and 20 often suffice.
\item \code{num_fourier_feat_xy = 5} - Number of random Fourier features to use for
the tested variables X and Y. Small values often suffice (e.g., 3 to 10),
\item \code{num_fourier_feat_z = 100} - Number of random Fourier features to use for
the conditioning set Z. Values between 50 and 300 often suffice,
\item \code{center_features = TRUE} - If TRUE, center the random features to have mean zero. Recommended
for better numerical stability,
\item \code{use_rcit = TRUE} - If TRUE, use RCIT; if FALSE, use RCoT
(Randomized Conditional Correlation Test),
\item \code{num_permutations = 500} - Number of permutations used for
the independence test when \code{rcit_approx = "permutation"} is selected.
\item \code{seed = -1} - Random seed for the independence test. If -1, no seed is set.
}
\item \code{"rank_independence"} - Rank-based independence test.
\itemize{
\item \code{alpha = 0.05} - Significance level for the
independence test.
}
\item \code{"sem_bic"} - SEM BIC test.
\itemize{
\item \code{penalty_discount = 2} - Penalty discount factor used in
BIC = 2L - ck log N, where c is the penalty. Higher c yield sparser
graphs,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
}}

\item{\code{use_for_mc}}{(logical) If TRUE, sets this test for the Markov checker \code{mc_test}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly returns \code{self}, for chaining.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_score"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_score}{}}}
\subsection{Method \code{set_score()}}{
Sets the scoring function to use in Tetrad.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_score(method, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{(character) Name of the score (e.g., "sem_bic", "ebic", "bdeu").
\itemize{
\item \code{"bdeu"} - Bayes Dirichlet Equivalent score with uniform priors.
\item \code{"basis_function_bic"} - BIC score for basis-function models.
This is a generalization of the Degenerate Gaussian score.
\item \code{"basis_function_blocks_bic"} - BIC score for mixed data using basis-function models.
\item \code{"basis_function_sem_bic"} - SEM BIC score for basis-function models.
\item \code{"conditional_gaussian"} - Mixed discrete/continuous BIC score.
\item \code{"degenerate_gaussian"} - Degenerate Gaussian BIC score.
\item \code{"discrete_bic"} - BIC score for discrete data.
\item \code{"ebic"} - Extended BIC score.
\item \code{"gic"} - Generalized Information Criterion (GIC) score.
\item \code{"mag_degenerate_gaussian_bic"} - MAG Degenerate Gaussian BIC Score.
\item \code{"poisson_prior"} - Poisson prior score.
\item \code{"rank_bic"} - Rank-based BIC score.
\item \code{"sem_bic"} - SEM BIC score.
\item \code{"zhang_shen_bound"} - Zhang and Shen bound score.
}}

\item{\code{...}}{Additional arguments passed to the private score-setting methods.
For the following scores, the following parameters are available:
\itemize{
\item \code{"bdeu"} - Bayes Dirichlet Equivalent score with uniform priors.
\itemize{
\item \code{sample_prior = 10} - This sets the prior equivalent
sample size. This number is added to the sample size for each
conditional probability table in the model and is divided equally
among the cells in the table,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"basis_function_bic"} - BIC score for basis-function models.
This is a generalization of the Degenerate Gaussian score.
\itemize{
\item \code{truncation_limit = 3} - Basis functions 1 though this
number will be used. The Degenerate Gaussian category indicator
variables for mixed data are also used,
\item \code{penalty_discount = 2} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse,
\item \code{do_one_equation_only = FALSE} - If TRUE, only one
equation should be used when expanding the basis.
}
\item \code{"basis_function_blocks_bic"} - BIC score for mixed data using basis-function models.
\itemize{
\item \code{basis_type = "polynomial"} - The type of basis to use. Supported
types are \code{"polynomial"}, \code{"legendre"}, \code{"hermite"}, and
\code{"chebyshev"},
\item \code{penalty_discount = 2} - Penalty discount factor used in
BIC = 2L - ck log N, where c is the penalty. Higher c yield sparser
graphs,
\item \code{truncation_limit = 3} - Basis functions 1 through this number will be used.
The Degenerate Gaussian category indicator variables for mixed data are also used.
}
\item \code{"basis_function_sem_bic"} - SEM BIC score for basis-function models.
\itemize{
\item \code{penalty_discount = 2} - Penalty discount factor used in
BIC = 2L - ck log N, where c is the penalty. Higher c yield sparser
graphs,
\item \code{jitter = 1e-8} - Small non-negative constant added to the diagonal of
covariance/correlation matrices for numerical stability,
\item \code{truncation_limit = 3} - Basis functions 1 through this number will be used.
The Degenerate Gaussian category indicator variables for mixed data are also used.
}
\item \code{"conditional_gaussian"} - Mixed discrete/continuous BIC score.
\itemize{
\item \code{penalty_discount = 1} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{discretize = TRUE} - If TRUE for the conditional
Gaussian likelihood, when scoring X --> D where X is continuous and
D discrete, one should to simply discretize X for just those cases.
If FALSE, the integration will be exact,
\item \code{num_categories_to_discretize = 3} -  In case the exact
algorithm is not used for discrete children and continuous parents
is not used, this parameter gives the number of categories to use
for this second (discretized) backup copy of the continuous
variables,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution.
}
\item \code{"degenerate_gaussian"} - Degenerate Gaussian BIC score.
\itemize{
\item \code{penalty_discount = 1} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"discrete_bic"} - BIC score for discrete data.
\itemize{
\item \code{penalty_discount = 2} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution.
}
\item \code{"ebic"} - Extended BIC score.
\itemize{
\item \code{gamma} - The gamma parameter in the EBIC score.
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"gic"} - Generalized Information Criterion (GIC) score.
\itemize{
\item \code{penalty_discount = 1} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{sem_gic_rule = "bic"} - The following rules are available:
\code{"bic"} - \eqn{\ln n},
\code{"gic2"} - \eqn{p n^{1/3}},
\code{"ric"} - \eqn{2 \ln(p n)},
\code{"ricc"} - \eqn{2(\ln(p n) + \ln\ln(p n))},
\code{"gic6"} - \eqn{\ln n \ln(p n)}.
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"mag_degenerate_gaussian_bic"} - MAG Degenerate Gaussian BIC Score.
\itemize{
\item \code{penalty_discount = 1} - Penalty discount. Higher penalty
yields sparser graphs,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution,
}
\item \code{"poisson_prior"} - Poisson prior score.
\itemize{
\item \code{poisson_lambda = 1} - Lambda parameter for the Poisson
distribution (> 0),
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"sem_bic"} - SEM BIC score.
\itemize{
\item \code{penalty_discount = 2} - Penalty discount factor used in
BIC = 2L - ck log N, where c is the penalty. Higher c yield sparser
graphs,
\item \code{structure_prior = 0} - The default number of parents
for any conditional probability table. Higher weight is accorded
to tables with about that number of parents. The prior structure
weights are distributed according to a binomial distribution,
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
\item \code{"rank_bic"} - Rank-based BIC score.
\itemize{
\item \code{gamma = 0.8} - Gamma parameter for Extended BIC (Chen and Chen, 2008). Between 0 and 1,
\item \code{penalty_discount = 2} - Penalty discount factor used in
BIC = 2L - ck log N, where c is the penalty. Higher c yield sparser
graphs.
}
\item \code{"zhang_shen_bound"} - Zhang and Shen bound score.
\itemize{
\item \code{risk_bound = 0.2} - This is the probability of getting
the true model if a correct model is discovered. Could underfit.
\item \code{singularity_lambda = 0.0} - Small number >= 0: Add
lambda to the diagonal, < 0 Pseudoinverse.
}
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly returns \code{self}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_alg"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_alg}{}}}
\subsection{Method \code{set_alg()}}{
Sets the causal discovery algorithm to use in Tetrad.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_alg(method, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{(character) Name of the algorithm (e.g., "fges", "pc",
"fci", etc.).}

\item{\code{...}}{Additional parameters passed to the private algorithm-setting
methods.
For the following algorithms, the following parameters are available:
\itemize{
\item \code{"boss"} - BOSS algorithm.
\itemize{
\item \code{num_starts = 1} - The number of times the algorithm
should be started from different initializations. By default, the
algorithm will be run through at least once using the initialized
parameters,
\item \code{use_bes = TRUE} - If TRUE, the algorithm uses the
backward equivalence search from the GES algorithm as one of its
steps,
\item \code{use_data_order = TRUE} - If TRUE, the data variable
order should be used for the first initial permutation,
\item \code{output_cpdag = TRUE} - If TRUE, the DAG output of the
algorithm is converted to a CPDAG.
}
\item \code{"boss_fci"} - BOSS-FCI algorithm.
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
Set to -1 for unlimited,
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
Set to -1 for unlimited,
\item \code{use_bes = TRUE} - If TRUE, the algorithm uses the
backward equivalence search from the GES algorithm as one of its
steps,
\item \code{use_heuristic = FALSE} - If TRUE, use the max p heuristic
version,
\item \code{complete_rule_set_used = TRUE} -  FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness,
\item \code{guarantee_pag = FALSE} - Ensure the output is a legal PAG
(where feasible).
}
\item \code{"fci"} - FCI algorithm.
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{stable_fas = TRUE} - If TRUE, the "stable" version of
the PC adjacency search is used, which for k > 0 fixes the graph
for depth k + 1 to that of the previous depth k.
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
\item \code{complete_rule_set_used = TRUE} - FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness.
\item \code{guarantee_pag = FALSE} - Ensure the output is a legal
PAG (where feasible).
}
\item \code{"ges" ("fges")} - Fast Greedy Equivalence Search (FGES) algorithm.
\itemize{
\item \code{symmetric_first_step = FALSE} - If TRUE, scores for both
X --> Y and X <-- Y will be calculated and the higher score used.
\item \code{max_degree = -1} - Maximum degree of any node in the
graph. Set to -1 for unlimited,
\item \code{parallelized = FALSE} - If TRUE, the algorithm should
be parallelized,
\item \code{faithfulness_assumed = FALSE} - If TRUE, assume that if
\eqn{X \perp\!\!\!\perp Y} (by an independence test) then
\eqn{X \perp\!\!\!\perp Y} | Z for nonempty Z.
}
\item \code{"gfci"} - GFCI algorithm. Combines FGES and FCI.
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{max_degree = -1} - Maximum degree of any node in the
graph. Set to -1 for unlimited,
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
\item \code{complete_rule_set_used = TRUE} - FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness,
\item \code{guarantee_pag = FALSE} - Ensure the output is a legal
PAG (where feasible),
\item \code{use_heuristic = FALSE} - If TRUE, use the max p heuristic.
\item \code{start_complete = FALSE} - If TRUE, start from a complete
graph.
}
\item \code{"grasp"} - GRaSP (Greedy Relations of Sparsest Permutation)
algorithm.
\itemize{
\item \code{covered_depth = 4} - The depth of recursion for first
search,
\item \code{singular_depth = 1} - Recursion depth for singular
tucks,
\item \code{nonsingular_depth = 1} - Recursion depth for nonsingular
tucks,
\item \code{ordered_alg = FALSE} - If TRUE, earlier GRaSP stages
should be performed before later stages,
\item \code{raskutti_uhler = FALSE} - If TRUE, use Raskutti and
Uhler's DAG-building method (test); if FALSE, use Grow-Shrink
(score).
\item \code{use_data_order = TRUE} - If TRUE, the data variable
order should be used for the first initial permutation,
\item \code{num_starts = 1} - The number of times the algorithm
should be started from different initializations. By default, the
algorithm will be run through at least once using the initialized
parameters.
}
\item \code{"grasp_fci"} - GRaSP-FCI algorithm. Combines GRaSP and FCI.
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{stable_fas = TRUE} - If TRUE, the "stable" version of
the PC adjacency search is used, which for k > 0 fixes the graph
for depth k + 1 to that of the previous depth k.
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
\item \code{complete_rule_set_used = TRUE} - FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness,
\item \code{covered_depth = 4} - The depth of recursion for first
search,
\item \code{singular_depth = 1} - Recursion depth for singular
tucks,
\item \code{nonsingular_depth = 1} - Recursion depth for nonsingular
tucks,
\item \code{ordered_alg = FALSE} - If TRUE, earlier GRaSP stages
should be performed before later stages,
\item \code{raskutti_uhler = FALSE} - If TRUE, use Raskutti and
Uhler's DAG-building method (test); if FALSE, use Grow-Shrink
(score).
\item \code{use_data_order = TRUE} - If TRUE, the data variable
order should be used for the first initial permutation,
\item \code{num_starts = 1} - The number of times the algorithm
should be started from different initializations. By default, the
algorithm will be run through at least once using the initialized
parameters,
\item \code{guarantee_pag = FALSE} - If TRUE, ensure the output is a
legal PAG (where feasible).
}

\item \code{"pc"} - Peter-Clark (PC) algorithm
\itemize{
\item \code{conflict_rule = 1} -
The value of \code{conflict_rule} determines how collider conflicts are handled. \code{1}
corresponds to the "overwrite" rule as introduced in the \pkg{pcalg} package, see
\code{\link[pcalg:pc]{pcalg::pc()}}. \code{2} means that all collider conflicts using bidirected edges
should be prioritized, while \code{3} means that existing colliders should be prioritized,
ignoring subsequent conflicting information.
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{stable_fas = TRUE} - If TRUE, the "stable" version of
the PC adjacency search is used, which for k > 0 fixes the graph
for depth k + 1 to that of the previous depth k.
\item \code{guarantee_cpdag = FALSE} - If TRUE, ensure the output is
a legal CPDAG.
}
\item \code{"rfci"} - Restricted FCI algorithm
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{stable_fas = TRUE} - If TRUE, the "stable" version of
the PC adjacency search is used, which for k > 0 fixes the graph
for depth k + 1 to that of the previous depth k.
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
\item \code{complete_rule_set_used = TRUE} - FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness.
\item \code{guarantee_pag = FALSE} - Ensure the output is a legal
PAG (where feasible).
}
\item \code{"sp_fci"} - Sparsest Permutation using FCI
\itemize{
\item \code{depth = -1} - Maximum size of conditioning set,
\item \code{max_disc_path_length = -1} - Maximum length for any
discriminating path,
\item \code{complete_rule_set_used = TRUE} - FALSE if the (simpler)
final orientation rules set due to P. Spirtes, guaranteeing arrow
completeness, should be used; TRUE if the (fuller) set due to
J. Zhang, should be used guaranteeing additional tail completeness,
\item \code{guarantee_pag = FALSE} - Ensure the output is a legal
PAG (where feasible),
\item \code{use_heuristic = FALSE} - If TRUE, use the max p heuristic version.
}
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly returns \code{self}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_knowledge"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_knowledge}{}}}
\subsection{Method \code{set_knowledge()}}{
Sets the background \code{Knowledge} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_knowledge(knowledge_obj)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{knowledge_obj}}{An object containing \pkg{Tetrad} knowledge.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_params"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_params}{}}}
\subsection{Method \code{set_params()}}{
Sets parameters for the Tetrad search.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_params(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Named arguments for the parameters to set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_parameters_for_function"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_parameters_for_function}{}}}
\subsection{Method \code{get_parameters_for_function()}}{
Retrieves the argument names of a matching private function.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_parameters_for_function(
  fn_pattern,
  score = FALSE,
  test = FALSE,
  alg = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{fn_pattern}}{(character) A pattern that should match a private method name.}

\item{\code{score}}{If TRUE, retrieves parameters for a scoring function.}

\item{\code{test}}{If TRUE, retrieves parameters for a test function.}

\item{\code{alg}}{If TRUE, retrieves parameters for an algorithm.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
(character) The names of the parameters.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-run_search"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-run_search}{}}}
\subsection{Method \code{run_search()}}{
Runs the chosen Tetrad algorithm on the data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$run_search(
  data = NULL,
  bootstrap = FALSE,
  int_cols_as_cont = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{(optional) If provided, overrides the previously set data.}

\item{\code{bootstrap}}{(logical) If TRUE, bootstrapped graphs will be
generated.}

\item{\code{int_cols_as_cont}}{(logical) If \code{TRUE}, integer columns are treated
as continuous, since \pkg{Tetrad} does not support ordinal data, but only
either continuous or nominal data. Default is \code{TRUE.}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{Disco} object (a list with a \code{caugi} and a \code{Knowledge} object).
Also populates \code{self$java}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_bootstrapping"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_bootstrapping}{}}}
\subsection{Method \code{set_bootstrapping()}}{
Configures bootstrapping parameters for the Tetrad search.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_bootstrapping(
  number_resampling = 0,
  percent_resample_size = 100,
  add_original = TRUE,
  with_replacement = TRUE,
  resampling_ensemble = 1,
  seed = -1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{number_resampling}}{(integer) Number of bootstrap samples.}

\item{\code{percent_resample_size}}{(numeric) Percentage of sample size for each bootstrap.}

\item{\code{add_original}}{(logical) If TRUE, add the original dataset to the bootstrap set.}

\item{\code{with_replacement}}{(logical) If TRUE, sampling is done with replacement.}

\item{\code{resampling_ensemble}}{(integer) How the resamples are used or aggregated.}

\item{\code{seed}}{(integer) Random seed, or -1 for none.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_data"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_data}{}}}
\subsection{Method \code{set_data()}}{
Sets or overrides the data used by Tetrad.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_data(data, int_cols_as_cont = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{(data.frame) The new data to load.}

\item{\code{int_cols_as_cont}}{(logical) If \code{TRUE}, integer columns are treated
as continuous, since Tetrad does not support ordinal data, but only
either continuous or nominal data. Default is \code{TRUE.}}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_verbose"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_verbose}{}}}
\subsection{Method \code{set_verbose()}}{
Toggles the verbosity in Tetrad.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_verbose(verbose)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{verbose}}{(logical) TRUE to enable verbose logging, FALSE otherwise.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-set_time_lag"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-set_time_lag}{}}}
\subsection{Method \code{set_time_lag()}}{
Sets an integer time lag for time-series algorithms.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$set_time_lag(time_lag = 0)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{time_lag}}{(integer) The time lag to set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_data"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_data}{}}}
\subsection{Method \code{get_data()}}{
Retrieves the current Java data object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_data()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
(Java object) Tetrad dataset.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_knowledge"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_knowledge}{}}}
\subsection{Method \code{get_knowledge()}}{
Returns the background \code{Knowledge} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_knowledge()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
(Java object) Tetrad Knowledge.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_java"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_java}{}}}
\subsection{Method \code{get_java()}}{
Gets the main Java result object (usually a graph) from the last search.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_java()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
(Java object) The Tetrad result graph or model.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_string"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_string}{}}}
\subsection{Method \code{get_string()}}{
Returns the string representation of a given Java object or \code{self$java}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_string(java_obj = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{java_obj}}{(Java object, optional) If NULL, uses \code{self$java}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
(character) The \code{toString()} of that Java object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_dot"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_dot}{}}}
\subsection{Method \code{get_dot()}}{
Produces a DOT (Graphviz) representation of the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_dot(java_obj = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{java_obj}}{(Java object, optional) If NULL, uses \code{self$java}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
(character) The DOT-format string.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-get_amat"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-get_amat}{}}}
\subsection{Method \code{get_amat()}}{
Produces an amat representation of the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$get_amat(java_obj = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{java_obj}}{(Java object, optional) If NULL, uses \code{self$java}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
(character) The adjacency matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TetradSearch-clone"></a>}}
\if{latex}{\out{\hypertarget{method-TetradSearch-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TetradSearch$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
