##
# ------------------------------------------------------------------------
#
# "regenboot(x,func,B,...,atom,s=mean(x),eps)" --
#
# Regenerative bootstrap for Markov chains.
#
# ------------------------------------------------------------------------
##
#' @aliases regenboot
#' @title Regenerative and Approximative Regenerative Block Bootstrap.
#' @description Performs regenerative block bootstrap and approximately regenerative
#' block bootstrap on a Markov chain, either in the atomic case 
#'              or in the general Harris case.
#' @param x A numeric vector representing a Markov process.
#' @param func The function to apply to each sample.
#' @param B A positive integer; the number of bootstrap replications.
#' @param ... Optional additional arguments for the \code{func} function.
#' @param atom A numeric value or a string; an atom of the Markov chain in the atomic case.
#' @param s A real number specifying the center of the small set. Default is the median of \code{x}.
#' @param small An object of class \code{\link{smallEnsemble}}. 
#' It can be created optimally using the function \code{\link{findBestEpsilon}}.
#' @param plotIt Logical. If \code{TRUE} then the function returns a plot of the time 
#' series with the approximative regenerative blocks. Does not plot anything in 
#' the atomic case.
#'  Default is \code{FALSE}.
#' @param moon A positive integer. Default is length of \code{x}.
#' \code{moon} should be smaller than the length of \code{x}, then it creates bootstrap 
#' samples of size \code{moon}.  
#' 
#' @details This function \code{regenboot} implements two different kinds of 
#' regenerative bootstrap:
#' \itemize{
#'   \item \emph{A regenerative block bootstrap} used for atomic Markov 
#'   chains.
#'   \item \emph{An approximate regenerative block bootstrap} used to bootstrap 
#'   Harris recurrent Markov chains 
#'   based on a given small set of the form \eqn{[s-eps,s+eps]} where \emph{s} 
#'   is the center and \emph{eps} the radius.
#' }
#' 
#' One must specify either the \code{atom} argument or the \code{small}
#' argument. In the first case, \code{atom} is the state used to split the
#' Markov chain into blocks ending with the atom. In the second case,
#' \code{small} is an object of class \code{smallEnsemble} representing the
#' small ensemble. Such objects are typically
#' obtained using the \code{findBestEpsilon} function but may also be constructed
#' manually using the \code{smallEnsemble} function. 
#' 
#' @return returns an object of class \code{boodd}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P., Clémençon, S. (2006a). Regenerative Block
#'   Bootstrap for Markov Chains. \emph{Bernoulli},  \bold{12}, 689-712.
#' 
#' Bertail, P. and Clémençon, S. (2006b).
#' Regeneration-based statistics for Harris recurrent
#'   Markov chains.  
#' \emph{Lecture notes in Statistics}, vol. \bold{187}, pp. 1-54, Springer.
#' 
#' Radulović, D. (2004). Renewal type bootstrap for Markov chains. \emph{Test},
#'  \bold{13}, 147-192.
#' @seealso \code{\link{boots}},
#' \code{\link{blockboot}},
#' \code{\link{plot.boodd}},
#' \code{\link{confint.boodd}}, \code{\link{findBestEpsilon}},\code{\link{smallEnsemble}}.
#' @keywords "Regenerative Block Bootstrap" "Atom" "Small set"
#'  "Approximative Regenerative Block Bootstrap" "Markov chain"
#' @export
#' @examples 
#' \donttest{
#' B <- 299
#' n <- 200
#' 
#' # Atomic Boostrap
#' acgt <- c("A","C","G","T")
#' probs <- c(.3,.1,.3,.3)
#' atom <- "C"
#' set.seed(1)
#' x <- sample(acgt,n,prob=probs,repl=TRUE)
#' propAtom <- function(x) {
#'   tbl <- as.vector(table(x))
#'   prop <- tbl[3]/length(x)
#'    return(prop)
#' }
#' boo <- regenboot(x,propAtom,B,atom=atom)
#' plot(boo)
#' 
#' # Approximate regenerative bootstrap with estimated small set
#' ar <- arima.sim(list(c(1,0,0),ar=0.6),n=500)
#' # Find the small ensemble with the largest number of regenerations
#' sm <- findBestEpsilon(ar,s=0,plotIt=TRUE)
#' # Approximate regenerative bootstrap of the mean
#' rboo <- regenboot(ar,mean,small=sm,B=999, plotIt=TRUE)
#' # Plot the corresponding bootstrap distribution
#' plot(rboo)
#' # Compute the bootstrap percentile confidence interval
#' confint(rboo)
#' }
##
regenboot <- function(x,func,B,...,atom,small=NULL,s=median(x),plotIt=FALSE,moon=length(x)) {
  # Test the value returned by func
  y <- func(x,...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  
  if (!missing(atom)) {
    # This is the atomic case
    res <- regenboot.atom(x,func,B,atom=atom,m=moon,...)
  } else {
    if (is.null(small)) {
      small <- findBestEpsilon(x,s)
    }
    res <- regenboot.smallEnsemble(x,func,B,small,m=moon,...,plotIt=plotIt)
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "regenerative"
  attr(obj,"func") <- func
  return(obj)
}


##
# ------------------------------------------------------------------------
#
# "regenboot.atom <- function(x,func,B,atom,m,...)" --
#
# Atomic regenerative bootstrap for finite states Markov chains.
#
# ------------------------------------------------------------------------
##
regenboot.atom <- function(x,func,B,atom,m=length(x),...) {
  n <- length(x)
  
  if (m>n)  {
    stop(" moon size must be much smaller than n")
  }
  if (length(atom) != 1) {
    stop("atom must be a single value")
  }
  if ( !(atom %in% x) ) {
    stop("atom must be an element of x")
  }
  n <- length(x)
  y <- func(x,...)
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  idx <- which(x==atom)
  li <- length(idx)
  starts <- idx+1
  lens <- diff(c(idx,n))
  # Drop the last block, either because it is incomplete or because it is
  # past the last position
  nb <- li - 1
  
  for (i in 1:B) {
    nx <- numeric(n+max(lens)-1)
    tot <- 0
    pos <- 1
    while (pos <= n) {
      # Draw a block
      bi <- sample(1:nb,1)
      start <- starts[bi]
      len <- lens[bi]
      nx[pos:(pos+len-1)] <- x[start:(start+len-1)]
      pos <- pos + len
    }
    res[i,] <- func(nx[1:m],...)
  }
  
  return(res)
}


##
# ------------------------------------------------------------------------
#
# "regenboot.smallEnsemble <- function(x,func,B,small,m=length(x),...)" --
#
# Small ensemble regenerative bootstrap for homogeneous Markov chains.
#
# ------------------------------------------------------------------------
##
regenboot.smallEnsemble <- function(x,func,B,small,m=length(x),...,plotIt=FALSE) {
  
  n=length(x)
  if (m>n)  {
    stop("moon size must be much smaller than n")
  }
  
  if (class(small)[1] != "smallEnsemble") {
    stop("expecting an object of class 'smallEnsemble'.")
  }
  
  y <- func(x,...)
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  
  s <- small$s
  eps <- small$epsilon
  delta <- small$delta
  p_XiXip1 <- small$trans
  
  # Find indices of pairs (X_i,X_{i+1}) belonging to the small ensemble
  isInSmall <- (x[1:(n-1)]>=s-eps)*(x[1:(n-1)]<=s+eps)*(x[2:n]>=s-eps)*(x[2:n]<=s+eps)
  
  # Compute the Bernoulli parameter if (X_i,X_{i+1}) \in S
  prob_regen <- delta*isInSmall/p_XiXip1
  
  # Simulate the Bernoulli drawing
  regen <- c((prob_regen>runif(n-1)),0)
  
  # Build the partition in blocks (ignoring first and last blocks)
  blocknums <- cumsum(c(0,regen[1:(n-1)]))
  if (regen[n-1]==1) {
    nb <- max(blocknums)
  } else {
    nb <- max(blocknums)-1
  }
  data <- cbind(1:n,blocknums)
  # S: start index of block - L: block length
  S <- numeric(nb)
  L <- numeric(nb)
  for (i in 1:nb) {
    aux <- subset(data,data[,2]==i,1)
    S[i] <- aux[1]
    L[i] <- nrow(aux)
  }
  
  # Bootstrap now
  for (i in 1:B) {
    nx <- numeric(n+max(L)-1)
    tot <- 0
    pos <- 1
    while (pos <= n) {
      # Draw a block
      bi <- sample(1:nb,1)
      start <- S[bi]
      len <- L[bi]
      nx[pos:(pos+len-1)] <- x[start:(start+len-1)]
      pos <- pos + len
    }
    res[i,] <- func(nx[1:(n-1)],...)
  }
  
  if (plotIt) {
    # Display the series
    plot(x,type="l",col="gray",xlab="",ylab="x",pch=16,main="(a)RBB Blocks")
    abline(h=s,col="gray")
    abline(h=c(s-eps,s+eps),lty=2,col="red")
    
    #  Draw the blocks
    for (i in 1:nb) {
      b <- S[i]
      l <- L[i]
      lines(b:(b+l),x[b:(b+l)],col=i)
      abline(v=b,lty=3,col="blue")
    }
  }
  if (m==n) return(res)
  else return(res[1:m,])
}