

## 
# ------------------------------------------------------------------------
# 
# "jackVarRegen.atom(x,func,atom,...)" --
# 
# Regenerative jackknife estimator of the variance of function
# 'func' for finite states Markov chains.
# 
# ------------------------------------------------------------------------
##
#' @aliases jackVarRegen.atom
#' @title Jackknife Variance Estimator for Atomic Markov Chains
#' @description Provides a regenerative jackknife estimator of the variance 
#'              of a function applied to atomic Markov chains.
#' @param x A vector or a matrix representing the Markov chain.
#' @param func The function to apply to each sample.
#' @param atom A numeric value or a string; an atom of the Markov chain in the atomic case.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details This function uses a regenerative approach to estimate the 
#'          jackknife variance of a statistic for atomic Markov chains. 
#'          It segments the chain at the specified \code{atom} into independent
#'          blocks. The function \code{func}, having output size equal to \emph{p}, 
#'  is applied to the data with each regenerative block removed in turn to finally compute an empirical
#' variance of the obtained values. This approach is particularly useful 
#' for atomic Markov chains.
#' 
#' @return Returns a scalar or a covariance matrix, depending on whether the function \code{func} 
#' is univariate or multivariate. For a function returning a vector of length 
#' \emph{p}, the output will be a covariance matrix of size \emph{p x p}.
#' 
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#'             
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' Quenouille, M. H. (1956). Notes on bias in estimation. \emph{Biometrika}, \bold{43},
#'   353–360.
#' 
#' @seealso  \code{\link{jackVar}}, 
#' \code{\link{jackFunc}},
#' \code{\link{regenboot}},
#' \code{\link{jackFuncRegen}}, 
#' \code{\link{jackFuncBlock}},
#' \code{\link{jackVarRegen}}.
#' @keywords Jackknife "Variance estimation" "Regenerative processes" "Atomic Markov chains"
#' @export
#' @examples
#'  B=1000
#'  set.seed(5)
#'  bb=0*(1:B)
#'  cc=0*(1:B)
#'  for (i in 1:B) {
#'    ts=genMM1(100,2,4)
#'    vv=function(ts){as.numeric(var(ts))}
#'    bb[i]=mean(ts)
#'    cc[i]=jackVarRegen.atom(ts,mean,atom=0)}
#'  var(bb)  # true variance of the mean (evaluated by Monte-Carlo)
#'  mean(cc) # mean of the variance estimators over the Monte-Carlo simulations

jackVarRegen.atom <- function(x,func,atom,...) {
  # Ensure func is a function
  func <- match.fun(func)
  
  if (length(atom) != 1) {
    stop("atom must be a single value")
  }
  if ( !(atom %in% x) ) {
    stop("atom must be an element of x")
  }
  Tn <- func(x,...)
  if (!is.vector(Tn)) {
    stop("Function 'func' must return a vector")
  }
  n <- length(x)

  p <- length(Tn)
  Tnames <- names(Tn)	
  
  idx <- which(x==atom)
  li <- length(idx)
  starts <- idx+1
  lens <- diff(c(idx,n))
  # Drop the last block, either because it is incomplete or because it is
  # past the last position
  nb <- li - 1
  nbTn <- nb*Tn
  if(nb>0){
    
    J <- matrix(nrow=nb,ncol=p)
    for (i in 1:nb) {
      start <- starts[i]
      len <- lens[i]
      # Indices of the block to remove
      bidx <- start:(start+len-1)
      Tni <- func(x[-bidx],...)
      J[i,] <- (nbTn - (nb-1)*Tni)
    }
    muJ <- colMeans(J)
    V <- matrix(0,nrow=p,ncol=p)
    for (i in 1:nb) {
      L <- J[i,]-muJ
      V <- V + L%*%t(L)
    }
    V <- V/(nb*(nb-1))
    if (!is.null(Tnames)) {
      rownames(V) <- Tnames
      colnames(V) <- Tnames
    }
    if (p == 1) {V <- as.vector(V)}		
    return(V)}
  else{
    message("There is no regenerative blocks, the variance is 0")
    return(0)}
}
