
## 
# ------------------------------------------------------------------------
# 
# "fieldbootP(arr,length.block,method)" --
# 
# return a bootstrap sample of the random field 
# ------------------------------------------------------------------------
##
#' @aliases fieldbootP
#' @title Bootstrap Sample from the Random Field 
#' @description Function returns a bootstrap sample of the random field on a lattice, using 
#' various block bootstrap methods such as moving block, circular block, or 
#' nonoverlapping block bootstrap.
#' @param arr A multidimensional real-valued array; it represents a random field on a grid of dimension
#' equals to dimension of the \code{arr}.
#' @param length.block An integer or vector of integers; it specified the block lengths for blocks.
#'  If a scalar is provided, the same block length is used for all dimensions.
#' @param method The method for array reconstruction:
#'  * \code{"movingblock"} - Moving Block Bootstrap,
#'  * \code{"nonoverlapping"} - Nonoverlapping Block Bootstrap,
#'  * \code{"circular"} - Circular Block Bootstrap (obtained by wrapping the field on the torus). 
#'  Default is \code{"movingblock"}.
#' @details In the case of random fields, the blocks \eqn{B_{i}}'s are random matrices of size
#' \code{length.block}. From the set of all blocks 
#' \eqn{\mathcal{B}=\{B_{1},B_{2},\dots,B_{q}\},} we select 
#' randomly with replacement blocks \eqn{B_{1}^{\ast},\dots,B_{l}^{\ast}} and bind them together. The
#' probability of choosing any block is \eqn{1/q}.
#' @return Returns a bootstrap sample of given \code{arr}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#'  Bertail, P.  Politis, D. N. Rhomari, N. (2000). Subsampling continuous 
#' parameter random fields and a Bernstein inequality,  \emph{Statistics}, 
#' \bold{33}, 367-392.
#' 
#' Nordman, D.J.  Lahiri, S.N.(2004). On optimal spatial subsample size for 
#' variance estimation, \emph{The Annals of Statistics}, \bold{32}, 1981-2027.
#' 
#' Politis,  D.N. Romano, J.P. (1993). Nonparametric Resampling for Homogeneous 
#' Strong Mixing Random Fields, \emph{J. Multivar. Anal.}, \bold{47}, 301-328.
#' @seealso \code{\link{blockboot}},
#' \code{\link{jackVarField}},
#' \code{\link{field.sub}}, \code{\link{fieldboot}}.
#' @keywords "Bootstrap" "Random fields"
#' @export
#' @examples 
#' \donttest{
#' library(geoR)
#' N=10
#' n=N^2
#' sim <- grf(n, grid="reg", cov.pars=c(1, .25), nsim=1)
#' image(sim) 
#' arr=array(sim$data,dim=c(N,N))
#' mb=fieldbootP(arr,length.block=c(2,2),method="movingblock")
#' simb=sim
#' simb$data=as.vector(mb)
#' image(simb)
#' }

fieldbootP <- function(arr,length.block,method=c("movingblock","nonoverlapping","circular")) {
  
  # Data lengths n_1, n_2, ..., n_d
  dlens <- dim(arr)
  ndims <- length(dlens)
  if (ndims < 2) {
    stop("expected at least 2-dimensional array")
  }
  if (length(length.block) == 1) {
    length.block = rep(length.block,ndims)
  }
  if (length(length.block) != ndims) {
    stop("wrong number of block lengths")
  }
  if (any(length.block > dlens)) {
    stop("block lengths must be less than data lengths in all dimensions")
  }
  if (any(length.block <= 0)) {
    stop("block lengths must be positive")
  }
  method <- match.arg(method)
  # Block lengths b_1, b_2, ..., b_d
  blens <- length.block
  last <- dlens-blens+1
  fulldims <- ((dlens-1)%/%blens +1)*blens
  
  # Array indices as string "1:n_1,1:n_2,...,1:n_d"
  indstr <- blockString(rep(1,ndims),dlens)
  
  # Build the tile indices (as strings) defining the covering
  # of the reconstructed array
  tiles <- blockTiles(blens,dlens)
  tnum <- length(tiles)
  
  # Initialize the new array (a slightly bigger array containing an 
  # entire number of tiles that will be trimmed later)
  narr <- array(0,dim=fulldims)
  origs <- integer(ndims)
  if (method[1] == "circular") {
    # Ensure circularity
    arr <- completeArray(arr,blens)	
  }
  
  for (j in 1:tnum) {
    tile <- tiles[j]
    
    # Draw a random block in the original array
    for (k in 1:ndims) {
      if (method[1] == "movingblock") {
        last <- dlens[k]-blens[k]+1
        origs[k] <- sample(1:last,1)
      } else if (method[1] == "nonoverlapping") {
        origs[k] <- sample(1+blens[k]*(0:(dlens[k]%/%blens[k]-1)),1)
      } else if (method[1] == "circular") {
        last <- dlens[k]
        origs[k] <- sample(1:last,1)
      }  
    }
    rndblock <- blockString(origs,blens)
    
    # Build the command string
    cmd <- paste0("narr[",tile,"] <- ","arr[",rndblock,"]")
    # Eval the command
    eval(parse(text=cmd))
  }		
  return(narr)
}
