% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Hierarchical.Functions.R
\name{hierarchical.desc2}
\alias{hierarchical.desc2}
\title{Operating characteristics for hierarchical group testing}
\usage{
hierarchical.desc2(p, I2 = NULL, I3 = NULL, se = 1, sp = 1, order.p = TRUE)
}
\arguments{
\item{p}{vector of probabilities corresponding to each individual's 
risk of disease.}

\item{I2}{a numeric vector of pool sizes for stage 2 testing (used in 
hierarchical testing with at least three stages).}

\item{I3}{a numeric vector of pool sizes for stage 3 testing (used in 
hierarchical testing with at least four stages).}

\item{se}{the sensitivity of the diagnostic test.}

\item{sp}{the specificity of the diagnostic test.}

\item{order.p}{a logical value indicating whether the vector of 
individual probabilities needs to be sorted.}
}
\value{
A list containing:
\item{ET}{the expected number of tests.}
\item{stages}{the number of stages in the testing algorithm.}
\item{group.size}{the total number of individuals tested in the 
algorithm.}
\item{I2}{pool sizes for the second stage of testing, or
"individual testing" if there are only two stages of testing.}
\item{I3}{pool sizes for the third stage of testing, or
"individual testing" if there are only three stages of testing.}
\item{m1}{the initial (stage 1) group size for two stage testing, 
or the number of subgroups originating from the initial group.}
\item{m2}{the number of subgroups for each preceding group 
containing more than one individual, or "individual testing" 
if there are only two stages of testing.}
\item{m3}{the number of subgroups for each preceding group 
containing more than one individual, or "individual testing" 
if there are only three stages of testing. NULL if there are 
only two stages of testing.}
\item{individual.testerror}{a data frame containing:
\describe{
\item{pse.vec}{a vector containing each individual's pooling 
sensitivity.}
\item{psp.vec}{a vector containing each individual's pooling 
specificity.}
\item{pppv.vec}{a vector containing each individual's pooling 
positive predictive value.}
\item{pnpv.vec}{a vector containing each individual's pooling
negative predictive value.}}}
\item{group.testerror}{a vector containing:
\describe{
\item{PSe}{the overall pooling sensitivity for the algorithm. 
Further details are given under 'Details'.}
\item{PSp}{the overall pooling specificity for the algorithm. 
Further details are given under 'Details'.}
\item{PPPV}{the overall pooling positive predictive value 
for the algorithm. Further details are given under 'Details'.}
\item{PNPV}{the overall pooling negative predictive value
for the algorithm. Further details are given under 'Details'.}}}
\item{individual.probabilities}{a vector containing each 
individual's probability of disease. If \kbd{order.p=TRUE}, this
is the sorted vector of individual probabilities.}
}
\description{
Calculate operating characteristics for hierarchical 
group testing with up to four stages, given a vector of individual 
probabilities and a testing configuration.
}
\details{
This function calculates the operating characteristics for
hierarchical group testing with up to four stages of testing. 
Operating characteristics calculated are expected number of tests, 
and pooling sensitivity, pooling specificity, pooling positive 
predictive value, and pooling negative predictive value, for each 
individual and for the configuration overall.

If \kbd{I2} is NULL, there are two stages of testing. If \kbd{I3} 
is NULL but \kbd{I2} has values, there are three stages of testing.
If both \kbd{I2} and \kbd{I3} have values, there are four stages 
of testing.

Vectors \kbd{I2} and \kbd{I3} should be entered using notation
that keeps track of all individuals through all stages (e.g. for
a group of 10 individuals that splits into 5, 4, and 1 individual
at stage 2, then into 3, 2, 2, 1, and 1 individual at stage 3
before individual testing at stage 4, then I2=c(5,4,1) and 
I3=c(3,2,2,1,1,1) so that the specimen that was tested 
individually at stage 2 is still numbered at stage 3 even though
it will not be tested again).

The displayed pooling sensitivity, pooling specificity, pooling positive 
predictive value, and pooling negative predictive value are weighted 
averages of the corresponding individual accuracy measures for all 
individuals within the initial group for a hierarchical algorithm.
Expressions for these averages are provided in the Supplementary 
Material for Hitt et al. (2018). These expressions are based on accuracy 
definitions given by Altman and Bland (1994a, 1994b).
}
\section{Note}{
 This function returns the pooling positive and negative
predictive values for all individuals even though these measures are 
diagnostic specific; i.e., PPPV (PNPV) should only be considered
for those individuals who have tested positive (negative).
}

\examples{
# Calculate the operating characteristics for 
#   non-informative two-stage hierarchical testing
#   with an overall disease prevalence of p = 0.015
#   and an initial group size of 12.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
p.vec <- rep(x=0.015, times=12)
hierarchical.desc2(p=p.vec, I2=NULL, I3=NULL, se=0.95, 
sp=0.95, order.p=FALSE)

# Calculate the operating characteristics for 
#   non-informative three-stage hierarchical testing
#   with an overall disease prevalence of p = 0.04, 
#   where an initial group of 20 individuals is 
#   split into equally sized subgroups of 5 each.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
p.vec <- rep(x=0.04, times=20)
hierarchical.desc2(p=p.vec, I2=rep(x=5, times=4),
I3=NULL, se=0.99, sp=0.99, order.p=FALSE)

# Calculate the operating characteristics for 
#   informative three-stage hierarchical testing
#   where an initial group of 10 individuals is 
#   split into subsequent groups of 5, 4, and 1 
#   individual.
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta 
#   distribution with p = 0.02 and a heterogeneity level 
#   of alpha = 0.5. Depending on the specified probability, 
#   alpha level, and overall group size, simulation may 
#   be necessary in order to generate the vector of individual
#   probabilities. This is done using p.vec.func() and 
#   requires the user to set a seed in order to reproduce 
#   results.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
set.seed(1002)
p.vec <- p.vec.func(p=0.02, alpha=0.5, grp.sz=10)
hierarchical.desc2(p=p.vec, I2=c(5,4,1), I3=NULL,
se=0.90, sp=0.90, order.p=TRUE)
}
\references{
\insertRef{Black2015}{binGroup}
}
\seealso{
\code{\link{Array.Measures}} for calculating operating characteristics
under array testing without master pooling, 
\code{\link{MasterPool.Array.Measures}} for non-informative array 
testing with master pooling, and \code{\link{inf.dorf.measures}} 
for informative two-stage hierarchical testing. See 
\code{\link{p.vec.func}} for generating a vector of 
individual risk probabilities for informative group testing.  

\url{http://chrisbilder.com/grouptesting/}

Other Operating characteristic functions: 
\code{\link{Array.Measures}()},
\code{\link{MasterPool.Array.Measures}()},
\code{\link{inf.dorf.measures}()}
}
\author{
This function was originally written by Michael S. 
Black for Black et al. (2015). The function was obtained from 
\url{http://chrisbilder.com/grouptesting/}. Minor modifications were made 
to the function for inclusion in the binGroup package.
}
\concept{Operating characteristic functions}
