% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{bhetGP}
\alias{bhetGP}
\title{MCMC sampling for Heteroskedastic GP}
\usage{
bhetGP(
  x = NULL,
  y = NULL,
  reps_list = NULL,
  nmcmc = 1000,
  sep = TRUE,
  inits = NULL,
  priors = NULL,
  reps = TRUE,
  cov = c("exp2", "matern", "ARD matern"),
  v = 2.5,
  stratergy = c("default", "flat"),
  vecchia = FALSE,
  m = min(25, length(y) - 1),
  ordering = NULL,
  verb = TRUE,
  omp_cores = 4
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{reps_list}{list object from hetGP::find_reps}

\item{nmcmc}{number of MCMC iterations}

\item{sep}{logical indicating whether to fit isotropic GP (\code{sep = FALSE})
or seperable GP (\code{sep = TRUE})}

\item{inits}{set initial values for hyparameters: \code{llam}, \code{theta_y}, 
\code{theta_lam}, \code{g}, \code{mean_y}, \code{mean_lam},
\code{scale_y}, \code{scale_lam}.
Additionally, set initial conditions for tuning:
\itemize{ 
  \item \code{theta_check}: logical; if \code{theta_check = TRUE},
  then ensures that theta_lam > theta_y i.e., decay of correlation
  for noise process is slower than mean process.
  \item \code{prof_ll_lam}: logical; if \code{prof_ll_lam = TRUE},
  infers tau2_lam i.e., scale parameter for latent noise process
  \item \code{noise}: logical; if \code{noise = TRUE}, infers nugget
  \code{g} throught M-H for latent noise process.
}}

\item{priors}{hyperparameters for priors and proposals (see details)}

\item{reps}{logical; if \code{reps = TRUE} uses Woodbury inference adjusting for
replication of design points and \code{reps = FALSE} does not
use Woodbury inference}

\item{cov}{covariance kernel, either Matern, ARD Matern 
or squared exponential (\code{"exp2"})}

\item{v}{Matern smoothness parameter (only used if \code{cov = "matern"})}

\item{stratergy}{choose initialization stratergy; "default" uses hetGP for 
\code{vecchia = FALSE} settings and sVecchia for \code{vecchia = TRUE}. 
See details.}

\item{vecchia}{logical indicating whether to use Vecchia approximation}

\item{m}{size of Vecchia conditioning sets (only used if 
\code{vecchia = TRUE})}

\item{ordering}{optional ordering for Vecchia approximation, must correspond
to rows of \code{x}, defaults to random, is applied to \code{x} (only used if 
\code{vecchia = TRUE})}

\item{verb}{logical indicating whether to print progress}

\item{omp_cores}{logical; if \code{vecchia = TRUE}, user may specify the number of cores to
use for OpenMP parallelization. Uses min(4, limit) where limit is max openMP 
cores available on the machine.}
}
\value{
a list of the S3 class \code{bhetgp} or \code{bhetgp_vec} with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response mean at inputs (x)
  \item \code{Ylist}: list of all responses observed per location (x)
  \item \code{A}: number of replicates at each location
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{priors}: copy of proposal/priors
  \item \code{settings}: setting for predictions using \code{bhetgp} or \code{bhetgp_vec}
  object
  \item \code{theta_y}: vector of MCMC samples for \code{theta_y} (length
        scale of mean process)
  \item \code{theta_lam}: matrix of MCMC samples for \code{theta_lam} (length 
        scale of latent noise process)
  \item \code{llam_samples}: matrix of ESS samples for \code{log lambda} (latent 
  noise process samples)
  \item \code{g}: vector of MCMC samples for \code{g} if infered
  \item \code{tau2}: vector of MAP estimates for \code{tau2} (scale 
        parameter of mean process)
  \item \code{tau2_lam}: vector of MAP estimates for \code{tau2_lam} (scale 
        parameter of latent noise process)
  \item \code{llik_y}: vector of MVN log likelihood of Y for reach Gibbs iteration
  \item \code{llik_lam}: vector of MVN log likelihood of \code{llam} i.e.
             the latent noise process for reach Gibbs iteration
  \item \code{x_approx}: conditioning set, NN and ordering for \code{vecchia = TRUE}
  \item \code{m}: copy of size of conditioning set for \code{vecchia = TRUE}            
  \item \code{time}: computation time in seconds
  
}
}
\description{
Conducts MCMC sampling of hyperparameters and latent noise 
    process \code{llam} for a hetGP.  Separate length scale 
    parameters \code{theta_lam} and \code{theta_y} govern the correlation 
    strength of the hidden layer and outer layer respectively.  
    \code{lam} layer may have a non-zero nugget \code{g} which governs 
    noise for the latent noise layer. \code{tau2_y} and \code{tau2_lam}
    control the amplitude of the mean and noise process respectively.
   In Matern covariance, \code{v} governs smoothness.
}
\details{
Maps inputs \code{x} to mean response \code{y} and noise levels
    \code{llam}. Conducts sampling of the latent noise process using Elliptical 
    Slice sampling.  Utilizes Metropolis Hastings sampling of the length 
    scale and nugget parameters with proposals and priors controlled by 
    \code{priors}. \code{g} for the noise process is set to a specific 
    value, and by default, is not estimated.  When \code{vecchia = TRUE}, 
    all calculations leverage the Vecchia approximation with 
    specified conditioning set size \code{m}. \code{tau2_y} is always 
    inferred from likelihood; \code{tau2_lam} is inferred by default but 
    may be pre-specified and fixed.
    
    NOTE on OpenMP: The Vecchia implementation relies on OpenMP parallelization
    for efficient computation.  This function will produce a warning message 
    if the package was installed without OpenMP (this is the default for 
    CRAN packages installed on Apple machines).  To set up OpenMP 
    parallelization, download the package source code and install 
    using the gcc/g++ compiler.    
    
    Proposals for \code{g} and \code{theta} follow a uniform sliding window 
    scheme, e.g. 
    
    \code{theta_star <- runif(1, l * theta_t / u, u * theta_t / l)}, 
    
    with defaults \code{l = 1} and \code{u = 2} provided in \code{priors}.
    To adjust these, set \code{priors = list(l = new_l, u = new_u)}.    
    Priors on \code{g}, \code{theta_y}, and \code{theta_lam} follow Gamma 
    distributions with shape parameters (\code{alpha}) and rate parameters 
    (\code{beta}) controlled within the \code{priors} list object.  
    Defaults are
    \itemize{
        \item \code{priors$alpha$theta_lam <- 1.5}
        \item \code{priors$beta$theta_lam <- 4}
        \item \code{priors$alpha$theta_y <- 1.5}
        \item \code{priors$beta$theta_y <- 4}
        \item \code{priors$alpha$g <- 1.5}
        \item \code{priors$beta$g <- 4}
    }
    
    \code{tau2_y} and \code{tau2_lam} are not sampled; rather directly inferred
    under conjugate Inverse Gamma prior with shape (\code{alpha}) and scale 
    parameters (\code{beta}) within the \code{priors} list object
    \itemize{       
        \item \code{priors$a$tau2_y <- 10}
        \item \code{priors$a$tau2_y <- 4}
        \item \code{priors$a$tau2_lam <- 10}
        \item \code{priors$a$tau2_lam <- 4}
    }
    These priors are designed for \code{x} scaled to 
    [0, 1] and \code{y} having mean \code{mean_y}.  These may be 
    adjusted using the \code{priors} input.
    
    Initial values for \code{theta_y}, \code{theta_lam}, \code{llam} may be
    specified by the user. If no initial values are specified, \code{stratergy}
    will determine the initialization method. \code{stratergy = "default"} 
    leverages mleHetGP for initial values of hyper-parameters if 
    \code{vecchia = FALSE} and Scaled-Vecchia with Stochastic Kriging (Sk-Vec)
    hybrid approach if \code{vecchia = TRUE}.
    
    For SK-Vec hybrid approach, code appropriated from 
    \url{https://github.com/katzfuss-group/scaledVecchia/blob/master/vecchia_scaled.R}
    is used to fit two GPs using the Vecchia approximation. The first for (x, y) pairs,
    which result in estimated residual sums of squares 
    based on predicted y values. Another GP on (x, s) to obtain
    latent noise estimates which are smoothed.  A script is leveraged
    internally within this package that fits this method. 
    
    Optionally, choose stratergy = "flat" which which will start at 
    uninformative initial values; \code{llam} = log(var(y) * 0.1) or
    specify initial values. 
    
    The output object of class \code{bhetgp} or \code{bhetgp_vec} is designed for 
    use with \code{trim}, \code{predict}, and \code{plot}.
}
\examples{

# 1D function with 1D noise 

# Truth
fx <- function(x){
result <- (6 * x - 2)^2* sin(12 * x - 4)
}

# Noise
rx <- function(x){
result <- (1.1 + sin(2 * pi * x))^2
return(result)
}

# Training data
r <- 10 # replicates
xn <- seq(0, 1, length = 25)
x <- rep(xn, r)

rn <- drop(rx(x))
noise <- as.numeric(t(mvtnorm::rmvnorm(r, sigma = diag(rn, length(xn)))))

f <- fx(x) 
y <- f + noise

# Testing data
xx <- seq(0, 1, length = 100)
yy <- fx(xx)

#--------------------------------------------------------------------------- 
# Example 1: Full model, no Vecchia 
#---------------------------------------------------------------------------

# Fitting a bhetGP model using all the data
fit <- bhetGP(x, y, nmcmc = 100, verb = FALSE)

# Trimming the object to remove burn in and thin samples
fit <- trim(fit, 50, 10)

# Prediction using the bhetGP object (independent predictions)
fit <- predict(fit, xx, cores = 2) 

# Visualizing the mean predictive surface. 
# Can run plot(fit, trace = TRUE) to view trace plots
plot(fit) 


#---------------------------------------------------------------------------
# Example 2: Vecchia approximated model
#---------------------------------------------------------------------------

# Fitting a bhetGP model with vecchia approximation. Two cores for OpenMP
fit <- bhetGP(x, y, nmcmc = 100, vecchia = TRUE, m = 5, omp_cores = 2, verb = FALSE)

# Trimming the object to remove burn in and thin samples
fit <- trim(fit, 50, 10)

# Prediction using the bhetGP_vec object with joint predictions (lite = FALSE)
# Two cores for OpenMP, default setting (omp_cores = 2). No SNOW
fit <- predict(fit, xx, lite = FALSE, vecchia = TRUE) 

# Visualizing the mean predictive surface
plot(fit)

\donttest{
#--------------------------------------------------------------------------- 
# Example 3: Vecchia inference, non-vecchia predictions
#---------------------------------------------------------------------------

# Fitting a bhetGP model with vecchia approximation. Two cores for OpenMP
fit <- bhetGP(x, y, nmcmc = 200, vecchia = TRUE, m = 5, omp_cores = 2)

# Trimming the object to remove burn in and thin samples
fit <- trim(fit, 100, 10)

# Prediction using the bhetGP object with joint predictions (lite = FALSE)
# Two cores for OpenMP which is default setting (omp_cores = 2)
# Two cores for SNOW (cores = 2)
fit <- predict(fit, xx, vecchia = FALSE, cores = 2, lite = FALSE)

# Visualizing the mean predictive surface
plot(fit)
}

}
\references{
Binois, Mickael, Robert B. Gramacy, and Mike Ludkovski. "Practical heteroscedastic Gaussian process 
modeling for large simulation experiments." Journal of Computational and Graphical 
Statistics 27.4 (2018): 808-821.

Katzfuss, Matthias, Joseph Guinness, and Earl Lawrence. "Scaled Vecchia approximation for 
fast computer-model emulation." SIAM/ASA Journal on Uncertainty Quantification 10.2 (2022): 537-554.

Sauer, Annie Elizabeth. "Deep Gaussian process surrogates for computer experiments." (2023).
}
