% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimations.R
\name{auction_model}
\alias{auction_model}
\title{Estimates a first-price auction model.}
\usage{
auction_model(
  dat = NULL,
  init_param = NULL,
  num_cores = 1,
  method = "BFGS",
  control = list(),
  std_err = FALSE,
  hessian_args = list()
)
}
\arguments{
\item{dat}{A data.frame containing input columns in the following order: the winning bids, number of bids, and \code{X}
variables that represent observed heterogeneity.}

\item{init_param}{Vector of initial values for mu, alpha, sigma, and beta vector, provided in order specified.
Note that the Weibull distribution requires mu and alpha to be positive. The standard deviation of unobserved heterogeneity, sigma, must be positive as well. The Beta vector may take any values.
If \code{init_params} is not provided, all values will be set to 1 by default.}

\item{num_cores}{The number of cores for running the model in parallel. The default value is 1.}

\item{method}{Optimization method to be used in optim() (see ?optim for details).}

\item{control}{A list of control parameters to be passed to optim() (see ?optim for details).}

\item{std_err}{If TRUE, the standard errors of the parameters will also be calculated. Note that it may significantly increase the computation time.}

\item{hessian_args}{A list of arguments passed as the \code{method.args} argument of the hessian() function if standard errors are calculated (see ?hessian for details).}
}
\value{
A list returned by \code{optim()}. See \code{?optim} for more details. If \code{std_err} was set to TRUE and the routine succeeded in inverting
the estimated Hessian, the list will have an additional component:

\item{std_err}{A vector of standard errors for parameter estimates.}
}
\description{
Estimates a first-price auction model.
}
\details{
This function estimates a first-price auction model with conditionally independent private values.
This version of the package estimates a procurement auction, where the winning bid is the amount that a single buyer
will pay to the top bidding supplier, and values correspond to costs. The model allows for unobserved heterogeneity
that is common to all bidders in addition to observable heterogeneity. The winning bid (Y) takes the form

Y = B * U * h(X)

where B is the proportional winning bid, U is the unobserved heterogeneity, and h(X) controls for
observed heterogeneity. The model is log-linear so that
log(Y) = log(B) + log(U) + log(h(X)) and log(h(X)) = beta1 * X1 + beta2 * X2 + ...

The (conditionally) independent private costs are drawn from a Weibull distribution
with parameters mu (mean) and alpha (shape). The CDF of this distribution is given by

F(c) = 1 - exp(- (c * 1/mu * Gamma(1 + 1/alpha))^(alpha))

The unobserved heterogeneity U is sampled from log-normal distribution with mean 1 and a free parameter sigma representing its standard deviation.

\code{init_params}, the initial guess for convergence, must be supplied.

This function utilizes the \code{Rsnow} framework within the \code{Rparallel} package. If \code{numcores} is not specified, this will be run using only
one CPU/core. One can use \code{parallel::detectCores()} to determine how many are available on your system, but you are not advised
to use all at once, as this may make your system unresponsive. Please see \code{Rparallel} and \code{Rsnow} for more details.

Note that the supplied data can not have missing values.
}
\examples{
###########################################################################
## Estimating parameters and standard errors with custom "control" argument
set.seed(100)
dat <- auction_generate_data(obs = 15, mu = 10, alpha = 2,
                              sigma = 0.2, beta = c(-1,1),
                              new_x_mean= c(-1,1),
                              new_x_sd = c(0.5,0.8))

res <- auction_model(dat, init_param =  c(8, 2, .5, .4, .6),
                      num_cores = 1,
                      control = list(parscale = c(1,0.1,0.1,1,1)),
                      std_err = TRUE)
res

## run vignette("auctionr") to view a more detailed example

}
\references{
Mackay, Alexander. 2020. "Contract Duration and the Costs of Market Transactions." Working paper, Appendix G.
}
\seealso{
\code{\link{auction_generate_data}}
}
\author{
Mackay, Alexander. \email{amackay@hbs.edu}, HBS Research Computing \email{research@hbs.edu}.
}
