% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary.effect_curve.R
\name{summary.effect_curve}
\alias{summary.effect_curve}
\alias{print.summary.effect_curve}
\title{Test omnibus hypotheses about an effect curves}
\usage{
\method{summary}{effect_curve}(
  object,
  hypothesis,
  method,
  subset = NULL,
  transform = TRUE,
  df = NULL,
  nsim = 1e+06,
  ...
)

\method{print}{summary.effect_curve}(x, digits = max(4L, getOption("digits") - 3L), ...)
}
\arguments{
\item{object}{an object of class \code{\link{effect_curve}}; the result of a call to \code{\link[=adrf]{adrf()}} or a function that modifies it.}

\item{hypothesis}{the hypothesis to be tested. Allowable options include \code{"flat"} (the default), \code{"linear"}, \code{"quadratic"}, \code{"cubic"}, a one-sided formula corresponding to a projection model, or a single number (e.g., 0). See Details. The default is \code{"flat"} for ADRFs and 0 otherwise.}

\item{method}{string; the method used to compute the p-value of the test. Allowable options include \code{"sim"} for simulation-based, \code{"imhof"} for Imhof's approximation, \code{"davies"} for Davies's approximation, \code{"liu"} for Liu's approximation, \code{"satterthwaite"} for Satterthwaite's approximation, and \code{"saddlepoint"} for a saddlepoint approximation. Default is \code{"imhof"} when the \pkg{CompQuadForm} package is installed, otherwise \code{"saddlepoint"} when the \pkg{survey} package is installed, and \code{"sim"} otherwise. See Details.}

\item{subset}{an optional logical expression indicating the subset of the subgroups for which to perform tests. Can only be used when \code{by} was supplied to the original call to \code{\link[=adrf]{adrf()}}, and only to refer to variables defining subgroups.}

\item{transform}{whether to perform the test on the transformed estimates. Allowable options include \code{TRUE}, \code{FALSE}, or a function specifying a transformation. Ignored unless \code{object} is an ADRF object. See Details.}

\item{df}{the "denominator" degrees of freedom to use for the tests. Default is to use the residual degrees of freedom from the original model if it is a linear model and \code{Inf} otherwise.}

\item{nsim}{when \code{method} is \code{"sim"}, the number of iterations used to simulate the p-values. Higher numbers give more accurate p-values subject to less Monte Carlo error but are slower and require more memory. Default is 1,000,000.}

\item{\dots}{when \code{method} is \code{"imhof"}, \code{"davies"}, or \code{"liu"}, further arguments passed to \pkgfun{CompQuadForm}{imhof}, \pkgfun{CompQuadForm}{davies}, or \pkgfun{CompQuadForm}{liu}, respectively.}

\item{x}{a \code{summary.effect_curve} object.}

\item{digits}{numeric; the number of digits to print.}
}
\value{
An object of class \code{"summary.effect_curve"}, which is a data.frame with a column for the p-value and, for stratified effect curves or contrasts thereof, additional columns identifying the subset to which the p-value refers.
}
\description{
\code{summary()} tests an omnibus hypothesis about an effect curve. For example, it can be used to test that the ADRF is flat, that the contrast between two ADRFs is 0 everywhere, or that the AMEF is 0 everywhere.
}
\details{
\code{summary()} performs an omnibus test for an effect curve. The hypothesis tested is determined by the argument to \code{"hypothesis"}. When supplied as a single number, \code{summary()} tests whether all values on the effect curve are equal to that number. When supplied as a one-sided formula, \code{summary()} tests whether the projection of the effect curve model onto the model represented in the formula is sufficient to describe the effect curve. The test itself tests whether the residuals around the projection are all equal to 0, incorporating the uncertainty in estimating the effect curve. See \code{\link[=curve_projection]{curve_projection()}} for more information on how the curve projection and the uncertainty in the residuals are computed.

\code{"flat"} tests whether all values on the curve are equal to each other (i.e., whether the curve is flat), without specifying what value they are equal to. This is equivalent to testing whether the variance around the mean estimate is different from 0 or whether an intercept-only projection model is sufficient. \code{"linear"} tests whether the curve is linear, i.e., whether the residuals around linear projection of the curve are all 0. \code{"quadratic"} and \code{"cubic"} test whether the curve is quadratic or cubic, respectively, using the same method.

Rejecting the null hypothesis means that the curve is more complicated than the specified model. For example, rejecting the null hypothesis that the curve is linear implies that the curve is nonlinear (and, therefore, not flat either).

The test involves computing a test statistic, specifying its distribution under the null hypothesis, and computing the p-value using the compliment of the cumulative density function of the distribution evaluated at the test statistic value. The test statistic depends on \code{"hypothesis"}. For \code{hypothesis} equal to a constant, say, \eqn{c}, the test statistic is
\deqn{T^* = \int_\mathcal{A} (\theta(a) - c)^2 \ da}
Otherwise, the test statistic is
\deqn{T^* = \int_\mathcal{A} (\theta(a) - \hat{\theta}_0(a))^2 \ da}
where \eqn{\hat{\theta}_0} is the projection of \eqn{\theta(a)} onto the null subspace specified by \code{hypothesis}.

Each of these can be approximated as a quadratic form, \eqn{T=\mathbf{\theta}'\mathbf{W}\mathbf{\theta}} where \eqn{\mathbf{\theta}} is a vector of estimates at evaluation points along the curve and \eqn{\mathbf{W}} is a diagonal matrix of weights implementing a trapezoidal approximation to the integral. The null hypothesis is that \eqn{T=0}, which approximates the null hypothesis that \eqn{T^*=0}. Each of the allowable options to \code{method} corresponds to a method of approximating the distribution of \eqn{T} under the null hypothesis:
\itemize{
\item \code{"sim"} simulates the null distribution by simulating from a multivariate normal distribution under the null hypothesis and computing the test statistic in each simulation. The p-value is the proportion of simulated estimates greater than the observed test statistic. When \code{df} is not \code{Inf}, the simulation is done from a multivariate t-distribution.
\item \code{"imhof"}, \code{"davies"}, and \code{"liu"} assume the test statistic follows a generalized \eqn{\chi^2}-distribution and approximate its CDF numerically. \code{"imhof"} tends to be the most accurate and is recommended. These methods require the \CRANpkg{CompQuadForm} package to be installed.
\item \code{"satterthwaite"} also assumes the test statistic follows a generalized \eqn{\chi^2}-distribution, but this distribution is approximated using a scaled F-distribution with the same first two moments.
\item \code{"saddlepoint"} also assumes the test statistic follows a generalized \eqn{\chi^2}-distribution, and this distribution is approximated using a saddlepoint method as implemented in \pkgfun{survey}{pFsum}. This method requires the \CRANpkg{survey} package to be installed.
}

In general, we recommend using \code{method = "imhof"}, though this requires \pkg{CompQuadForm} to be installed (and is the default when it is). \code{method = "saddlepoint"} has also been shown to be quite accurate and very fast. When using \code{"sim"}, increasing \code{nsim} improves the accuracy of the p-value by reducing Monte Carlo error. The default value of \code{1e6} ensures that the simulated p-value is within .0005 of the true p-value with at least 98\\% confidence for p-values less than .05. For greater precision at higher p-value thresholds, \code{nsim} should be increased. The other methods exist primarily for comparison purposes. \code{"satterthwaite"} is particularly fast and doesn't require any other packages, but it can be somewhat inaccurate.
\subsection{Transform}{

When the effect curve is an ADRF and the outcomes are bounded (e.g., a probability between 0 and 1), the \code{transform} argument can be specified, which changes the details of the tests.

The tests above assume the estimates along the effect curve are normally distributed (or t-distributed when \code{df} is not \code{Inf}). However, when the outcome is bounded (e.g., a probability bounded between 0 and 1), this assumption may not be valid for the ADRF in finite samples. \code{transform} transforms the estimates to ones that are unbounded and computes the corresponding distribution of transformed estimates using the delta method. By default, if a generalized linear model is used for the outcome with a non-identity link function, the estimates are transformed by the link function to be on an unbounded scale. Note this is not the same as using the linear predictor for the effect curve; this is simple a transformation of the estimated points along the curve already computed. When \code{hypothesis} is a number, that number is also transformed.

Tests on the transformed and untransformed ADRFs correspond to different hypotheses; the difference is not simply a matter of the appropriate distribution of the statistic. For example, for binary outcome model with a logistic transformation, testing that the transformed ADRF is linear corresponds to testing whether the ADRF is a sigmoid function, whereas testing that the untransformed ADRF is linear corresponds to testing whether the ADRF is a straight line. These choices correspond to how the projection of the ADRF is formed; see \code{\link[=curve_projection]{curve_projection()}} for details. See Examples for an example with a binary outcome. In the example, there is evidence to reject that the transformed ADRF is linear for one of the groups, indicating that a sigmoid function is not sufficient for describing the ADRF, but there is indeterminate evidence (at the .05) to reject that the untransformed ADRF is linear for either group, indicating that a linear function is sufficient for describing the ADRF (at least in the treatment range examined).
}
}
\examples{
data("nhanes3lead")

fit <- glm(Block >= 12 ~ poly(logBLL, 3) *
             Male * (Age + Race + PIR + NICU +
                       Smoke_Pregnant),
           data = nhanes3lead,
           family = binomial)

# ADRFs of logBLL on P(Block >= 12) within
# groups defined by `Male`
adrf1 <- adrf(fit, treat = "logBLL",
              by = ~Male)

adrf1

# Test if ADRFs are flat
summary(adrf1)

# Test if logit-transformed ADRFs are linear
# (i.e., if ADRFs have sigmoid shape)
summary(adrf1, hypothesis = "linear")
# summary(adrf1, hypothesis = ~logBLL) # same test

proj1 <- curve_projection(adrf1, "linear")

plot(adrf1, proj = proj1, conf_level = 0)

# Test if un-transformed ADRFs are linear
summary(adrf1, hypothesis = "linear",
        transform = FALSE)

proj2 <- curve_projection(adrf1, "linear",
                          transform = FALSE)

plot(adrf1, proj = proj2, conf_level = 0)

## Test if ADRFs differ from each other by
## testing if the ADRF contrast is 0
curve_contrast(adrf1) |>
  summary()
}
\seealso{
\itemize{
\item \code{\link[=adrf]{adrf()}} for computing the ADRF
\item \code{\link[=summary.curve_est]{summary.curve_est()}} for performing inference on individual points on an effect curve
\item \code{\link[=plot.effect_curve]{plot.effect_curve()}} for plotting the effect curve
\item \code{\link[=curve_projection]{curve_projection()}} for projecting a simpler model onto an effect curve
}
}
