% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_basis_lambda_LOOCV.R
\name{select_basis_lambda_LOOCV}
\alias{select_basis_lambda_LOOCV}
\title{Select optimal number of basis functions and smoothing parameter}
\usage{
select_basis_lambda_LOOCV(
  data,
  time,
  nbasis_range = seq(10, 100, 5),
  lambda_range = 10^seq(-6, 2, length.out = 10),
  norder = 4,
  penalty_order = 2
)
}
\arguments{
\item{data}{A numeric vector of observed values (one curve).}

\item{time}{A numeric vector of time points corresponding to `data`.}

\item{nbasis_range}{A numeric vector specifying the range of basis
function numbers to test.}

\item{lambda_range}{A numeric vector of lambda values to test
(e.g. \code{10^seq(-6, 2, length = 10)}).}

\item{norder}{Integer specifying the order of the B-spline basis
(default = 4, cubic).}

\item{penalty_order}{Integer specifying the derivative order used in
the roughness penalty (default = 2).}
}
\value{
A list with:
\item{best_nbasis}{Number of basis functions minimizing LOOCV error.}
\item{best_lambda}{Lambda value minimizing LOOCV error.}
\item{cv_errors}{Matrix of mean squared LOOCV errors across all
  combinations.}
\item{nbasis_range}{Tested nbasis values.}
\item{lambda_range}{Tested lambda values.}
}
\description{
This function performs leave-one-timepoint-out cross-validation (LOOCV)
on a single functional trajectory to determine the optimal number of
B-spline basis functions (`nbasis`) and smoothing parameter (`lambda`)
for fitting a functional data object using \code{fda::smooth.basis}.
}
\details{
Each time point is left out in turn, the curve is refit using the
remaining observations, and prediction error at the omitted time
point is recorded. Mean squared prediction error is used as the
selection criterion.

This function is computationally expensive and intended as a helper
routine for selecting smoothing parameters prior to large-scale
functional analyses.


The function searches for the best combination of number of basis functions
and roughness penalty by minimizing leave-one-out cross-validation error.
The smoothing parameter (\eqn{\lambda}) controls the trade-off between
goodness of fit and smoothness; larger values enforce smoother curves.
}
\examples{
\donttest{
time <- seq(0, 48, length.out = 100)
y <- sin(2 * pi * time / 24) + rnorm(100, 0, 0.1)
result <- select_basis_lambda_LOOCV(y, time,
                                    nbasis_range = seq(5, 25, 5),
                                    lambda_range = 10^seq(-6, 0, length = 5))
print(result)}

}
