% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forecasting_functions.R
\name{expanding_tvmvp}
\alias{expanding_tvmvp}
\title{#' Expanding Window Time-Varying Minimum Variance Portfolio Optimization}
\usage{
expanding_tvmvp(
  obj,
  initial_window,
  rebal_period,
  max_factors,
  return_type = "daily",
  kernel_func = epanechnikov_kernel,
  rf = NULL,
  M0 = 10,
  rho_grid = seq(0.005, 2, length.out = 30),
  floor_value = 1e-12,
  epsilon2 = 1e-06
)
}
\arguments{
\item{obj}{An object of class TVMVP with the data.}

\item{initial_window}{An integer specifying the number of periods used in the initial estimation window.}

\item{rebal_period}{An integer specifying the number of periods between portfolio rebalancing (e.g., weekly, monthly).}

\item{max_factors}{An integer indicating the maximum number of latent factors to consider in the factor model.}

\item{return_type}{A string indicating the return frequency. Options: `"daily"`, `"weekly"`, or `"monthly"`. Used for annualization of evaluation metrics.}

\item{kernel_func}{A kernel function to be used in the local PCA procedure. Default is `epanechnikov_kernel`.}

\item{rf}{A numeric vector or single value representing the risk-free rate. If `NULL`, it defaults to zero.}

\item{M0}{An integer specifying the number of observations to leave out between the two sub-samples for the adaptive thresholding of the residual covariance estimation.}

\item{rho_grid}{A numeric sequence specifying the grid of rho values for threshold selection in covariance shrinkage. Default is `seq(0.005, 2, length.out = 30)`.}

\item{floor_value}{A small positive value to ensure numerical stability in the covariance matrix. Default is `1e-12`.}

\item{epsilon2}{A small positive value used in the adaptive thresholding of the residual covariance estimation. Default is `1e-6`.}
}
\value{
An R6 object of class \code{ExpandingWindow} with the following accessible elements:
\describe{
  \item{\code{summary}}{A data frame of summary statistics for the TV-MVP and equal-weight portfolios, including cumulative excess return (CER), mean excess returns (MER), Sharpe ratio (SR), and standard deviation (SD) (raw and annualized).}
  \item{\code{TVMVP}}{A list containing rebalancing dates, estimated portfolio weights, and excess returns for the TV-MVP strategy.}
  \item{\code{Equal}}{A list with similar structure for the equal-weight portfolio.}
}
}
\description{
This function performs time-varying minimum variance portfolio (TV-MVP) optimization using
time-varying covariance estimation based on Local Principal Component Analysis (Local PCA). The
optimization is performed over a Expanding window, with periodic rebalancing.
The procedure is available either as a stand-alone function or as a method in
the `TVMVP` R6 class.
}
\details{
Two usage styles:
#' \preformatted{
# Function interface
results <- expanding_tvmvp(
  obj = tv,
  initial_window = 50,
  rebal_period = 20,
  max_factors = 3,
  return_type = "daily",
  rf = NULL
)

# R6 method interface
tv <- TVMVP$new()
tv$set_data(returns)
results <- tv$expanding_tvmvp(
  initial_window = 50,
  rebal_period = 20,
  max_factors = 3,
  return_type = "daily",
  rf = NULL)
}

The function implements a Expanding time-varying PCA approach to estimate latent factor structures
and uses a sparse residual covariance estimation method to improve covariance matrix estimation.
The covariance matrix is used to determine the global minimum variance portfolio (MVP), which is
rebalanced periodically according to the specified `rebal_period`. The number of factors is
determined by a BIC-type information criterion using the function `determine_factors`, updated
yearly. The bandwidth is determined by Silverman's rule of thumb, updated each rebalancing period.

If `rf` is `NULL`, the risk-free rate is assumed to be zero.
}
\section{References}{
 
Lillrank, E. (2025). \ifelse{html}{
    \out{<a href='../doc/thesis.pdf'>A Time-Varying Factor Approach to Covariance Estimation</a>}
  }{Master’s thesis (PDF in inst/doc)}
  
Fan, Q., Wu, R., Yang, Y., & Zhong, W. (2024). Time-varying minimum variance portfolio. Journal of Econometrics, 239(2), 105339.
}

\examples{
\donttest{
# Generate random returns for 20 assets over 100 periods
set.seed(123)
returns <- matrix(rnorm(20*100), nrow = 100, ncol = 20)

# Initialize object
tv <- TVMVP$new()
tv$set_data(returns)

# Run Expanding TV-MVP optimization
results <- expanding_tvmvp(
  obj = tv,
  initial_window = 50,
  rebal_period = 20,
  max_factors = 3,
  return_type = "daily",
  kernel_func = epanechnikov_kernel,
  rf = NULL
)

# R6 method interface
results <- tv$expanding_tvmvp(
  initial_window = 50,
  rebal_period = 20,
  max_factors = 3,
  return_type = "daily",
  rf = NULL)

# Print summary
print(results)

# Plot cumulative log excess returns
plot(results)
}
}
