#' @title Myerson value
#' @description Given a game and a communications network, this function computes the Myerson value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param communications An undirected communications network, as a list of vectors (their order being irrelevant), each containing two different players (their order being irrelevant). If two players are able to communicate with each other, a bidimensional vector containing them should be present in the list; otherwise, the vector should be absent. When \code{communications} is not specified, it is assumed that all players can communicate with each other.
#' @param game A logical value. By default, \code{game=FALSE}. If set to \code{TRUE}, the game with restricted communication is also returned.
#' @return The corresponding Myerson value, as a vector.
#' @details Let \eqn{v\in G^N}. Assuming that communication between players is necessary for their cooperation,
#' the game with restricted communication, \eqn{v^{A}}, is defined by
#' \eqn{v^{A}(S)=v(S)} if the players of S can communicate and \eqn{v^{A}(S)=0} otherwise, for each \eqn{S\in 2^N}.
#'
#' The Myerson value is the Shapley value of the game \eqn{v^{A}}.
#' @examples
#' v <- c(0,0,0,0,30,30,40,40,50,50,60,70,80,90,100)
#' communications <- list(c(1,2), c(1,3), c(1,4))
#' myersonvalue(v, binary=FALSE, communications)
#' @references Myerson, R. B. (1977). Graphs and cooperation in games. \emph{Mathematics of Operations Research}, 2(3), 225-229.
#' @seealso \link{shapleyvalue}
#' @export

myersonvalue <- function(v, binary=FALSE, communications, game=FALSE) {
  # Se fabrica matriz G a partir de communications.
  # El valor de Myerson es el valor de Shapley del graph game vG asociado.
  # Ahora bien, vG es descomponible por la particion determinada por las
  # componentes conexas del grafo G: C1,...,Cr
  # Por tanto, vG=vGC1+...+vGCr
  # Siendo el valor de Shapley lineal, Sh(vG)=sh(vGC1)+...+sh(vGCr)

  nCoal=length(v) # Número de coaliciones
  n=log(nCoal+1)/log(2) # Número de jugadores
  # Comprobaciones de posibles errores:
  if (n>floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (length(unlist(communications))%%2 == 1) { # Si communications contiene número impar de jugadores, mal
    stop("'communications' must be a list of vectors, each containing two different players.")
  }
  for (i in length(communications)) { # Si en communications hay vectores que repiten jugador, mal
    if(length(unique(communications[[i]]))<2) {
      stop("'communications' must be a list of vectors, each containing two different players.")
    }
  }
  if(sum(unlist(communications)>n)>0 | sum(unlist(communications)<1)>0) { # Si en communications hay numeros que no identifican jugador, mal
    stop("'communications' must be a list of vectors, each containing two different players.")
  }
  # Acabaron las comprobaciones de errores

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  # Se fabrica matriz G a partir de communications:
  G <- matrix(unlist(communications),nrow=length(communications),byrow=TRUE)

  # Número de arcos en el grafo G
  nA <- nrow(G)

  # Inicializamos el vector para el valor de Myerson
  M <- numeric(n)

  # ##############
  # ##############
  # Paso 1: Calculo de las componentes conexas de G
  ################
  # ##############

  C <- graphcomponents(1:n, G)  # Componentes conexas de G
  nComps <- length(C)               # Número de componentes conexas

  # ##############
  ################
  # Paso 2: Calculamos el grafo restringido a cada componente
  ################
  # ##############

  # %
  # Recorro los arcos del grafo G y los clasifico en la componente que les
  # corresponde.
  # Creo una matriz 'lista' con la información, para cada componente conexa,
  # de los arcos que se han de borrar del grafo inicial.
  # La matriz lista es binaria, con nA filas y nComps columnas.
  # lista(m,c)=1 -  el arco m se borra en la componente c
  # lista(m,c)=0 -  el arco m está en la componente c

  # Creamos una matriz 'lista' para indicar los arcos a eliminar

  lista <- matrix(1, nrow = nA, ncol = nComps)

  #lista <- vector("list", nA)

  # Recorremos los arcos del grafo G
  for (m in 1:nA) {
    #  % Si el primer nodo del arco no está en la componente conexa,
    # lo pongo en la lista de los que voy a borrar.
    # Nota: como ya tengo separadas las componentes conexas, solo
    # tengo que fijarme en el primer nodo ya que este NUNCA podrá
    # estar conectado con otro de una componente conexa diferente
    control <- TRUE
    c <- 1
    while (control && c <= nComps) {
      if (G[m, 1] %in% C[[c]]) {
        control <- FALSE
        lista[m, c] <- 0
      }
      c <- c + 1
    } # fin del while
  } # fin del bucle de los arcos del grafo G

  # Ahora utilizamos la matriz lista para calcular explicitamente los subgrafos
  # Creamos los subgrafos restringidos para cada componente
  GC <- vector("list", nComps)
  for (c in 1:nComps) {
    GC[[c]] <- G # Partimos inicialmente de la matriz con todos los arcos
    GC[[c]] <- GC[[c]][lista[, c] == 0, , drop = FALSE] # Borramos aquellos arcos con coeficiente 1 en la lista
  }


  ###########################################
  # Paso 3: Cálculo de los valores de Shapley para cada componente
  ###########################################

  for (c in 1:nComps) {
    # Número de jugadores en el elemento Cc de la particion
    nCc <- length(C[[c]])

    #######
    # Subjuego asociado a Cc
    #######

    vC <- numeric(2^nCc - 1)
    Cpos <- 2^(C[[c]] - 1)

    # % Recorro las subcoaliciones
    # numeradas de 1 en adelante, es decir,
    # renumero los jugadores de Cc={j1,...,js} como 1,...,s
    # Luego tendré que recuperar sus posiciones para obtener
    # el valor correcto de la función característica

    for (R in 1:(2^nCc - 1)) {
      Rjug <- integer(nCc)
      # Primero calculo los jugadores que pertenecen a R
      num <- R
      J <- floor(log2(num)) + 1 # el mayor jugador en R

      for (ii in 1:J) {
        Rjug[ii] <- num %% 2
        num <- floor(num / 2)
      } # Fin del bucle para determinar los jugadores en R

      # Calculamos el valor de vC para la coalición R
      vC[R] <- v[sum(Cpos * Rjug)]
    }

    # Transformamos el grafo restringido para el subjuego
    GG <- GC[[c]]
    for (ii in 1:length(C[[c]])) {
      GG[GG == C[[c]][ii]] <- ii
    }

    # Calculamos el juego restringido por el grafo
    vGC <- graphgame(vC, GG)

    # Añadimos el valor de Shapley de vGC al valor de Myerson
    M[C[[c]]] <- M[C[[c]]] + shapleyvalue(vGC,binary=TRUE)
  }

  if (game == FALSE) { # Si el usuario no pide vGC...
    return(M)
  } else { # Si el usuario pide vGC...
    if (binary == FALSE) { # Devolvemos vGC en orden lexicográfico.
      return(list(value=M,game=bin2lex(vGC)))
    } else { # Devolvemos vGC en orden binario.
      return(list(value=M,game=vGC))
    }
  }

  return(M)
}
