#' @title Balanced family check
#' @description This function checks if the given family is balanced.
#' @param Fam A vector containing the binary order positions of a family of coalitions.
#' @param n The number of players in the set of players from which \code{Fam} is taken. When not specified, \code{n} is assumed to be the the number of players present in \code{Fam}.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return This function returns three outputs: \code{check}, \code{minimal} and \code{delta}.
#' If \code{Fam} is not a balanced family: \code{check=FALSE} and both \code{minimal} and \code{delta} are \code{NULL}.
#' If \code{Fam} is a balanced family: \code{check=TRUE}, \code{minimal=TRUE} if \code{Fam} is minimal (\code{minimal=FALSE} otherwise), and \code{delta} returns an associated weight family.
#' @details A family \eqn{F} of non-empty coalitions of a set of players \eqn{N}
#' is balanced if there exists a weight family \eqn{\delta^{F} = \{ \delta^{F}_{S} \}_{S \in F}} such that
#' \eqn{\delta^{F}_{S} > 0} for each \eqn{S \in F} and \eqn{\sum_{S \in F} \delta^{F}_{S} e^{S} = e^{N}},
#' being \eqn{e^{S}} the characteristic vector of \eqn{S}, that is, the vector \eqn{(e_{i}^{S})_{i \in N}}
#' in which \eqn{e_{i}^{S}=1} if \eqn{i \in S} and \eqn{e_{i}^{S}=0} if \eqn{i \notin S}).
#'
#' A balanced family \eqn{F} is said to be minimal if there does not exist
#' a balanced family \eqn{F'} such that \eqn{F' \subsetneq F}.
#' @examples
#' balancedfamilycheck(c(3,6,13,8)) # balanced and minimal
#' balancedfamilycheck(c(3,5,9,4,8,14)) # balanced but not minimal
#' balancedfamilycheck(c(1,2,4,12,13)) # not balanced
#' @seealso \link{balancedcheck}, \link{kohlbergcriterion}, \link{totallybalancedcheck}
#' @references Maschler, M., Solan, E., & Zamir, S. (2013). \emph{Game Theory}. Cambridge University Press.
#' @export

balancedfamilycheck <- function(Fam, n = NULL, tol = 100*.Machine$double.eps) {

  # balancedfamilycheck Checks if a family of coalitions is balanced

  # INPUT
  # Fam = A vector [S1,...,Sm] with the binary number associated with each
  # coalition of the family
  # n = Number of players
  # tol = tolerance (by default 100*eps)
  # OUTPUT
  # B=1 if Fam is a balanced family. B=0 otherwise.
  # L=[0,...,0] if B=0. Otherwise, L=(L1,...Lm) is a feasible vector for the family Fam
  # minimal=1 if B=1 and Fam is a minimal balanced family
  # minimal=0 if B=1 and Fam is not a minimal balanced family
  # minimal=NULL if B=0
  #
  # EXAMPLES
  # Fam <- c(3, 6, 13, 8);  Equilibrada minimal (4 jugadores) {1,2}, {2,3}, {1,3,4}, {4}
  # Fam <- c(1, 2, 4, 12, 13);  No es equilibrada (4 jugadores) {1}, {2}, {3,4}, {1,3,4}
  # Fam <- c(3, 5, 9, 4, 8, 14)  Es equilibrada. No minimal (4 jugadores)
  # {1,2}, {1,3}, {1,4}, {3}, {4}, {2,3,4}
  #######################################
  #  Default number of players
  if (is.null(n)) {
    n <- floor(log2(max(Fam))) + 1 # La coalicion de mayor número nos permite calcular
  # el mayor jugador posible de toda la familia
  }
  #  Inconsistencia Fam y n
  if (n < floor(log2(max(Fam))) + 1) {
    stop("The number of players in 'Fam' cannot be greater than 'n'.")
  }
  #######################################
  #############
  ## DATOS
  #############
  # Size of the family
  m <- length(Fam)
  #  Inicializamos
  B <- FALSE # Familia NO equilibrada
  L <- NULL # de momento, nada
  minimal <- NULL # de momento, nada
  #################################
  #################################
  #         THEORY (Zumsteg)
  #################################
  # A family Fam=[S1,...,Sm] is balanced if there are l=(l1,..,lm)>0 such that:
  #          l1*I(i,S1)+ ... + lm*I(i,Sm)=1 para todo i=1,...,n  (1)
  # donde I(i,Sj)=1 si i está en Sj, I(i,Sj)=0 en caso contrario.
  # En forma matricial (1) se escribe:
    #                                     A*l = 1n, l>0              (2)
  # siendo A (n x m) la matriz a(i,j)=I(i,Sj), y 1n el vector nx1 de 1's.
  # Dado que A es positiva, si (2) tiene solución ha de ser l<=1.
  # Ahora transformamos el sistema (2) multiplicando las ecuaciones por un
  # número w>=1  de forma que Y=wl>=1. De modo que encontrar
  # una solución factible de (2) equivale a encontrar una solución factible
  # de
  #     A*Y=W*1n, Y,w>=1  (2')
  # Ahora, denotando y=Y-1m (siendo 1m el vector mx1 de 1's)
  # tenemos que (2') equivale a
  #      A(y+1m)=w*1n, y,w>=0   (2'')
  # (En efecto, si (Y,w) es solución de (2') evidentemente (Y-1,w) lo es de (2'').
  # Veamos ahora que si (y,w) es solución de (2'') entonces (Y,w) lo es de
  # (2'). Basta con demostrar que w>=1. Ahora bien, w*1n=Ay+A1m y Ay>=0.
  # Observemos que  A*1m es el vector nx1 dado por la suma de las filas de A
  # (o sea, el número de elementos de la familia Fam al que pertenece cada
  # jugador). Al menos un elemento de A1m>=1 y, por tanto, w>=1).
  # Reordenando, (2'') equivale a:
  #                                 -A*y+w*1n=A*1m, y,w>=0  (3)
  #
  # Finalmente, (3) has a feasible solution if (clasical initialization problem)
    # the optimum of the following LP
  #
  #              Min 1'*z=z1+...+zn
  #              s.t.
  #                   -Ay+w1n+Iz=A1m               (P)
  #                   y,w,z>=0
  #
  #is zero.
  # Problem (P) has n equality constraints and n+m+1 variables:
  # y1,...,ym , w , z1, ... ,zn
  # 1, ..., m ,m+1, m+2,...,m+n+1  ------ Orden de las variables
  #
  # Fam is minimal if and only if there exists a unique l>0, such that A*l = 1n.
  # If Fam is minimal then m<=n
  ##########################################
  ##########################################

  # Primero comprobamos si Fam es minimal (porque algebraicamente es más fácil)
  # Matriz A
  A <- matrix(0, nrow = n, ncol = m)
  for (j in seq_along(Fam)) {
    A[, j] <- as.integer(intToBits(Fam[j])[1:n])
  }
  ########  Familia equilibrada minimal ##########

  if (m <= n) { # Si Fam tiene más de n coaliciones entonces no es minimal
    # Resolvemos el sistema  A * l = 1n
   sol <- solvels(cbind(A,rep(1, n)))
   # Si el sistema es compatible determinado y todas las coordenadas
   # de la solución x son estrictamente positivas


    if (sol$flag == 1 && sum(sol$solution > tol) == m){
      B <- TRUE # La familia es equilibrada
      L <- sol$solution[1:m] # sol es el único vector L factible
      minimal <- TRUE #  Fam es minimal
      return(list(check = B, minimal = minimal, delta = L))
    }
  }

  ###################################
  #            CASO Fam NO MINIMAL
  ###################################

  if (is.null(minimal)) {

    #Variables básicas. Inicialmente z1,...,zn

    Basic <- (m + 2):(m + n + 1)

    #Variables no básicas. Inicialmente y1,...,ym, w

    NBasic <- 0:m+1

    # Tableau (sin las básicas)
    Tab <- matrix(0, nrow = n + 1, ncol = m + 2)
    # Todas las filas excepto la última
    Tab[1:n, 1:m] <- -A # Matriz A
    Tab[1:n, m + 1] <- rep(1, n) # Columna de w
    Tab[1:n, m + 2] <- rowSums(A) # Término independiente. Suma de las filas de A

   #############
    # FIRST STEP: Update the last row
   #############
     # Inicialmente, la última fila del tableau (la definida por la función objetivo)
     # tiene 1's bajo las n variables básicas iniciales (las z's) y ceros en las demás
     # (las que reflejamos en la matriz Tab).
     # Para empezar a pivotar, necesitamos que bajo las variables básicas haya
     # 0's. Este será nuestro primer paso
    ############

    # Inicializamos la última fila
    for (ii in 1:n) {
      Tab[n + 1, ] <- Tab[n + 1, ] - Tab[ii, ] # Ultima fila menos fila ii
    }
    ####################
    # SECOND STEP: Simplex procedure
    #
    # Criterio de parada
    # Regla de Bland: Buscamos la primera entrada estrictamente negativa
    # en la última fila


    # Proceso del símplex
    C <- which(Tab[n + 1, 1:(ncol(Tab) - 1)] < -tol)[1]

    while (!is.na(C)) { # Si todas las coordenadas de la fila r son positivas o nulas FIN
      # Regla de Bland (entra en la base la columna de menor índice con el
                        # coeficiente de control estrictamente negativo)
      # Meteremos la columna C en la base
      #iterada=iterada+1, C
      ColPivote <- Tab[, C]
      ####
        # Buscamos la fila para pivotar.
      # OJO. Para elegir la fila en la que pivotar tenemos que considerar sólo
      # los cocientes correspondientes a valores de la columna pivote estrictamente mayores que c
      Fam <- 0 # Fila de pivote
      cociente <- Inf
      for (ii in 1:n) {
        # Los cocientes correspondientes a coeficientes negativos o nulos
        # de la columna pivote no los consideramos.
        # Regla de Bland. Tomamos el menor índice con el menor cociente
        if (ColPivote[ii] > tol && Tab[ii, ncol(Tab)] / ColPivote[ii] < cociente - tol) {
            Fam <- ii
            cociente <- Tab[ii, ncol(Tab)] / ColPivote[ii]
          }
      }
      # Elemento de pivote (Fam,C).
      ################ PIVOTE %%%%%%%%%%%%%%%%%%%%%%%%%
      # Guardamos la fila Fam

      Faux <- Tab[Fam, ]
      # Cambiamos la  columna C por el vector Fam de la base canonica (y
      # añadimos la ultima posición nula)
      Tab[, C] <- rep(0, n + 1)
      Tab[Fam, C] <- 1
      # También ajustamos la fila Fam de pivote
      FF <- Faux
      FF[C] <- 1

      # Pivotamos en el elemento (Fam,C)
      for (ii in 1:(n + 1)) {
        if (abs(ColPivote[ii]) > tol) {
          Tab[ii, ] <- Tab[ii, ] - ColPivote[ii] / ColPivote[Fam] * FF
        }
      }
      # Recomponemos la fila Fam
      Tab[Fam, ] <- Faux / ColPivote[Fam]
      Tab[Fam, C] <- 1 / ColPivote[Fam]
      # Actualizamos las variables basicas y no basicas
      momentanea <- Basic[Fam]
      Basic[Fam] <- NBasic[C]
      NBasic[C] <-  momentanea


      orden <- order(NBasic)
      NBasic <- NBasic[orden]
      # Reordenamos las columnas de Tab para tener siempre el orden inicial de las
      # variables
      Tab[, 1:(ncol(Tab) - 1)] <- Tab[, orden]
      # Actualizamos el control
      # Primera entrada en la fila r con coordenada estrictamente positiva
      C <- which(Tab[n + 1, 1:(ncol(Tab) - 1)] < -tol)[1]
    }

    ############# Si el valor optimo es NULO entonces la familia es equilibrada

    if (abs(Tab[n + 1, m + 2]) < tol) {
      B <- TRUE
      minimal <- FALSE
    }
  } # Bucle if minimal==0
  ###############################
  ###############################
  #############     CALCULO DE UN VECTOR FACTIBLE
  ###############################
  ###############################

  if (B == 1 && minimal == 0) {
    # %     % En la solución optima puede que tengamos alguna zi como variable básica.
   # Siempre podemos quitarlas de la base a coste cero.
    V <- Basic[Basic >= (m + 2)]
    #if (any(Basic >= m + 2)) {
     # stop("Tengo zs como básicas")
    #}
    #   Ahora, si (y,w) es una solución optima, entonces l=(y+1)/w es solución de (2)
    y <- rep(0, m + 1)
    # Calculamos la solucion optima basica del programa (P).
    #     (yB,z)=Tab(1:n,end), donde
    # yB son las y's basicas.
    #y[Basic] <- Tab[1:n, m + 2]
    y[Basic] <- Tab[1:(nrow(Tab) - 1), ncol(Tab)]
    #  El valor optimo w
    w <- y[length(y)]
    # La solución de (2) es L=(y+1)/w
    L <- (y[1:(length(y) - 1)] + 1) / w
  } # Bucle if de pesos

  return(list(check = B, minimal = minimal, delta = L))
}
