% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2e_optimize_act.R
\name{e2e_optimize_act}
\alias{e2e_optimize_act}
\title{Optimize StrathE2E fishing gear activity multipliers to maximumize the likelihood of either observed ecosystem target data or know harvest ratios.}
\usage{
e2e_optimize_act(
  model,
  selection = "HR",
  n_iter = 3000,
  start_temperature = 0.5,
  cooling = 0.985,
  csv.output = FALSE,
  runtime.plot = TRUE,
  n_traj = 100,
  deltaHi = 0.2,
  attenuationstep = 500,
  deltaG = 0.25,
  nyears = 40,
  quiet = TRUE
)
}
\arguments{
\item{model}{R-list object generated by the e2e_read() function which defined the model configuration.}

\item{selection}{Text string from a list to select the method for optimization. Select from: "HR", "ECO", Remember to include the phrase within "" quotes (default = "HR"). "HR" selects a scheme for optimizing the activity multipler to known harvest ratios and uses just the fishing fleet model. "ECO" selects a scheme for optimizing to the observational data in the state of the ecosystem and so uses both the fishing fleet and ecology models.}

\item{n_iter}{Number of iterations of the model (per trajectory in the case of selection = "HR") (default=3000). Suggest a value of 500 if selection = "ECO".}

\item{start_temperature}{Initial value of the simulated annealing temperature parameter (default=1). Suggested a value of 0.5 if selection = "ECO". Values in the range 0.0005 - 5 are reasonable. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood.}

\item{cooling}{Rate at which the simulated annealing temperature declines with iterations (default=0.985). Suggested a value of 0.975 if selection = "ECO".}

\item{csv.output}{Logical. If TRUE then enable writing of csv output files (default=FALSE).}

\item{runtime.plot}{Logical. If FALSE then disable runtime plotting of the progress of the run - useful for testing (default=TRUE)}

\item{n_traj}{Specific to selection="HR" : Number of repeats (trajectories) of the simulated annealing process (default=100).}

\item{deltaHi}{Specific to selection="HR" : Initial coefficient of variation for jiggling the activity multiplier values (default=0.2). This is set in the parameter file "optimize_fishing.csv" in the case where selection="ECO" and can be manually varied during a run.}

\item{attenuationstep}{Specific to selection="HR" : Number of itereations between down-steps of the jiggling factor applied to the multiplier values. The jiggling rate is attenuated by a factor of 2 every xx iterations (default=500).}

\item{deltaG}{Specific to selection="HR" : Coefficient of variation for jiggling the gear linkage values (default=0.25).}

\item{nyears}{Specific to selection="ECO" : Number of years to run the ecology model in each iteration (default=40).}

\item{quiet}{Logical. Specific to selection="ECO" : If TRUE then suppress informational messages at the start of each iteration (default=TRUE).}
}
\value{
A list object containing the data from each trajectory and the final accepted parameter values. Optionally (by default), csv files of the histories and the final accepted parameter values. The latter are returned to the model parameter folder in a format to be read back into the model.
}
\description{
Launches a simulated annealing process to find the set of fishing fleet model gear activity multipliers
producing the maximum likelihood of either a) observed target data on the state of the ecosystem
given specified environmental driving data and ecology model parameters, and effort-harvest ratio scaling parameters, or b) known harvest ratios for each
guild in each spatial zone. The former version (optimizing to the state of the ecosystem uses the full StrathE2E system, ie the combined fishing fleet and 
ecology models. However, the latter version, i.e. optimizing to known harvest ratios, uses only the fishing fleet model. The version of optimization procedure is selected by an argument in the function call.
}
\details{
Generic details for both optimization procedures

Simulated annealing is is a probabilistic technique for approximating the global optimum of a given function. As implemented here the process searches the parameter space
of a model to locate the combination which maximises the likelihood of a set of observed
data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1)

Model configuration and initial values of the ecology model parameters need to be
assembled by a prior call of the e2e_read() function.

NOTE that the models.path argument in the e2e_read() function call needs to point to a user workspace folder, not the default
North Sea model provided with the package. This is because the optimization function needs write-access to the model /Param folder,
but the /extdata/Models folder in the package installation is read-only.
To use the annealing function on the North Sea model, use the e2e_copy() function to make a copy of the
North Sea model in the user workspace.

The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
the likelihoods of the proposed and accepted parameter sets at each iteration.

At the end of the procedure (provided that the argument csv.output=TRUE) a new version of the gear activity multipliers file is exported
to the folder /Param of the model version, with a user defined identifier specified by the model.ident argument
in the e2e_read() function. The histories of parameter combinations and their likelihood during the course of the optimization procedure
are saved as CSV files in the results folder (provided that the argument csv.output=TRUE), and returned as a list object.

The returned or saved data on the progress of the procedures can be be further analysed using the function e2e_plot_opt_diagnostics() to 
assess the performance of the optimization process.

To preserve the new activity multipliers and incorporate them into the fishing fleet model parameterisation
the multiplier values need to be applied to the activity rates specified in the data input file /Param/fishing_activity_*.csv.
Manually update the values in fishing_activity_*.csv, by multiplying the existing values by the new multipliers emerging from the annealing process.

If the edited file fishing_activity_*.csv is saved with a new identifier (*) then in order to use it in a subsequent run of the
StrathE2E model (using the e2e_run() function) it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new file.

Specific details for optimization to observed data in the state of the ecosytem

The observational data to which the activity multipliers are optimized are loaded from the folder Modelname/Variantname/Target/annual_observed_*.csv as part of a e2e_read() function call and are built into the R-list object generated by e2e_read().
Column 3 of annual_observed_* (header: "Use_1.yes_0.no") is a flag to set whether any given row is used in calculating the likelihood of the observed data given the model setup and parameters. Un-used rows of data are omitted from calculations.

The coefficient of variation for jiggling the activity multipliers can be varied in real-time
during the run by editing the file "optimize_fishing.csv" in the folder
/Param/control/ of the model version. Suggested vaues for the SD are in the range 0.1 to 0.01

The y-axis (likelihood of the target data) range of the real time plot can be varied during the run by 
editing the setup file "optimize_fishing.csv".

At the end of the procedure the proposed and accepted activity multiplier values and corresponding likleihoods from each iteration of the procedure are saved as csv files 
in the results folder (provided that the argument csv.output=TRUE), and in a list object which is returned by the function. 
The two csv files generated by the procedure have names:  
annealing_ACTmult_proposalhistory-*, annealing_ACTmult_acceptedhistory-*, where * denotes the value
of model.ident defined in the preceding e2e_read() function call.
The returned list object contains three dataframes: parameter_proposal_history, parameter_accepted_history, new_parameter_data.
The proposal and accepted histories can be further analysed with the function e2e_plot_opt_diagnostics() to assess the performance of the optimization process.

Specific details for optimization to known harvest ratios

The task here is a bit harder than normal
because of the potentially large overlap in the selectivity patterns of the fishing gears with respect to the living guilds in the ecosystem. So there are
some non-standard features to try and avoid local likelihood maxima.

The zonal harvest ratios to which the activity multipliers are optimized are loaded from the folder Modelname/Variantname/Target/zonal_harvest_r_*.csv as part of a e2e_read() function call and are built into the R-list object generated by e2e_read().
Column 4 of taget data file (header: "Use_1.yes_0.no") is a flag to set whether any given row is used in calculating the likelihood of the observed data given the model setup and parameters. Un-used rows of data are omitted from calculations.

To cope with the overlap in selectivity of the fishing gears, the function uses a set of linkage parameters specified in the the file /Param/fishing_gear_linkages.csv
These parameters force selected gears activities to vary in concert +/- some variation, rather than independently. The parameters for the gear linkages are located in the file
../Modelname/Variantname/Param/fishing_gear_linkages.csv. The table of linkages specifies which gear activity rates are forced to 
vary in concert during the fitting process, as opposed to varying independently. The value of the linkage coefficient defines the 
scaling of changes in the activity rate of a dependent gear relative to its linked independent gear. For example, if gear 8 is permitted to vary 
independently (value in column "Gear to which linked" = NA and "Linkage coefficient" = NA). If gear 9 is dependent on gear 8 then the activity rate 
of gear 9 this would be specified by e.g. "Gear to which linked" = 8 and "Linkage coefficient" = 0.645. This would force the activity of gear 9 to be 
set at gear8_activity * (0.645 +/- a random variation defined by the argument deltaG). 

The initial coefficient of variation for searching the parameter space is set by a function argument, and cannot be user-varied dring the run. The CV decreases in steps with increasing iterations.
The function repeats the iteration process multiple times. The repeats are referred to as 'trajectories'. Each trajectory starts from and follows a different pathway through the parameter space.
At the end of the process, the best-fit set of activity multipliers is selected from across all the trajectories.

At the end of the procedure the activity multiplier values and harvest ratios from the end of each trajectory are saved as csv files 
in the results folder (provided that the argument csv.output=TRUE), and to a list object which is returned by the function. These data are presented as both raw values, and as relative to the initial
values (in the case of the activity multipliers) or the target values (in the case of the harvest ratios). The relativity is expressed as "(optimized - target)/target" for each trajectory
(in the case of harvest ratios). The four csv files generated by the procedure have names 
activity_optim_gearmult_history-*, activity_optim_gearmult_relinitial_history-*, activity_optim_harvestratio_history-*, activity_optim_harvestratio_reltarget_history-*, where * denotes the value
of model.ident defined inthe preceding e2e_read() function call.
The returned list object contains four dataframes: gear_mult_results, gear_mult_rel_initial, harvest_ratio_results, harvest_ratio_rel_target, new_parameter_data.
The relative multiplier and target data can be further analysed with the function e2e_plot_opt_diagnostics() to assess the performance of the optimization process.
}
\examples{
\donttest{
# Load the 1970-1999 version of the North Sea model supplied with the package and generate a
# quick test data object with only 8 iterations and running the model for only 3 years 
# optimising activity rates to the database of observed indices of the state of the ecosystem.
# In this example, the final parameter values are not saved back to the model Param folder.
# More realistic would be at least 500 iterations and running for 50 years.
# Even so this example will take a few minutes to run:
    model<-e2e_read(model.name="North_Sea",
                    model.variant="1970-1999",
                    model.ident="test")
# This model is already optimized to the observed ecosystem data supplied with the package.
# Here, so as to illustrate the optimization process in action, here we perturb the temperature
# driving data to knock the model away from its maximum likelihood state relative to the 
# target data:
# add 3 degC to upper layer offshore temperatures:
    model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3
# add 3 degC to inshore temperatures:
    model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3
# add 3 degC to lower layer offshore temperatures:
    model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3
    test_run  <- e2e_optimize_act(model, selection="ECO", n_iter=8, start_temperature=0.4,
                 cooling=0.975, csv.output=FALSE, nyears=3 )
# View the structure of the returned list:
    str(test_run,max.level=1)
# View the structure of the returned list element containing parameter objects:
    str(test_run$new_parameter_data,max.level=1)
# View the new, final accepted parameter data:
    test_run$new_parameter_data
}

# --------------------------------------------------------------------------

# This is a dummy example to illustrate a realistic run in which optimised
# parameters are written back to the model Param folder. To try it out substitute 
# your own relative folder path in place of \Folder in the e2e_copy() function...
# WARNING - this will take about 26 hours to run...
# Copy the 1970-1999 version of the North Sea model supplied with the package into a
# user workspace relative to the current working directory (../Folder):
#    e2e_copy("North_Sea", "1970-1999",
#                dest.path="Folder")
# Load the copied version of the North Sea/1970-1999 model from the user workspace
# and assign a path for results data:
# (REPLACE "Folder/Models" and "Folder/results" with your own paths before running)
#    model<-e2e_read(model.name="North_Sea",
#                    model.variant="1970-1999",
#                    models.path="Folder/Models",
#                    results.path="Folder/results",
#                    model.ident="fittingrun")
# Launch the fitting process
#    fitting_data<-e2e_optimize_act(model, selection="ECO", n_iter=500,
#                  start_temperature=0.5, cooling=0.975, csv.output=TRUE, nyears=50)

# --------------------------------------------------------------------------

\donttest{
# Examples of using the function for optimizing to given zonal harvest ratios
    model<-e2e_read(model.name="North_Sea",
                    model.variant="1970-1999",
                    model.ident="test")
# Activity rates in this model are already optimized to the target harvest ratios supplied with
# the package but we woud not expect to recover these values in this very short demonstration run
    test_run  <- e2e_optimize_act(model, selection="HR", n_iter=30, start_temperature=1.0,
                                  cooling=0.985, csv.output=FALSE, n_traj=5)
# View the structure of the returned list:
    str(test_run,max.level=1)
# View the structure of the returned list element containing parameter objects:
    str(test_run$new_parameter_data,max.level=1)
# View the new, final accepted parameter data:
    test_run$new_parameter_data
}

# --------------------------------------------------------------------------

# This is a dummy example to illustrate a realistic run in which optimised
# parameters are written back to the model Param folder. To try it out substitute 
# your own relative folder path in place of \Folder in the e2e_copy() function...
# WARNING - this will take about 14 hours to run...
# Copy the 1970-1999 version of the North Sea model supplied with the package into a
# user workspace relative to the current working directory (../Folder):
#    e2e_copy("North_Sea", "1970-1999",
#               dest.path="Folder")
# Load the copied version of the North Sea/1970-1999 model from the user workspace
# and assign a path for results data:
# (REPLACE "Folder/Models" and "Folder/results" with your own paths before running)
#    model<-e2e_read(model.name="North_Sea",
#                    model.variant="1970-1999",
#                    models.path="Folder/Models",
#                    results.path="Folder/results",
#                    model.ident="fittingrun")
# Launch the fitting process
#    fitting_data<-e2e_optimize_act(model, selection="HR", n_iter=3000, start_temperature=1.0,
#                                   cooling=0.985, csv.output=TRUE, n_traj=100 )

# --------------------------------------------------------------------------

}
\seealso{
\code{\link{e2e_ls}} , \code{\link{e2e_read}} , \code{\link{e2e_plot_opt_diagnostics}} , \code{\link{e2e_optimize_eco}} , \code{\link{e2e_optimize_hr}}
}
