% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neg_loglik.R
\name{neg_loglik}
\alias{neg_loglik}
\title{Negative Log-Likelihood for Semi-Supervised FMM with a Mixed-Missingness Mechanism}
\usage{
neg_loglik(theta, Y, m_j, Z, d2_yj, xi, alpha_k, unpack_fn)
}
\arguments{
\item{theta}{Numeric vector of packed model parameters to be unpacked by \code{unpack_fn}.}

\item{Y}{Numeric matrix of observations (n x p).}

\item{m_j}{Integer or logical vector of length n indicating missingness:
0 for observed (labeled block), 1 for unlabeled/missingness block.}

\item{Z}{Integer vector of length n with class labels for labeled samples
(1..g); use \code{NA} for unlabeled rows.}

\item{d2_yj}{Numeric vector of length n with the entropy-like score used in the
MAR mechanism (e.g., posterior entropy or any scalar proxy).}

\item{xi}{Numeric length-2 vector \code{c(xi0, xi1)} for the logistic MAR model
\eqn{q_j = \text{logit}^{-1}(\xi_0 + \xi_1 d2_{yj})}.}

\item{alpha_k}{Numeric scalar in (0,1), the MCAR mixing proportion in the
missingness mechanism.}

\item{unpack_fn}{Function that takes \code{theta} and returns a list with elements:
\describe{
\item{\code{pi}}{Numeric vector of length g with mixture weights.}
\item{\code{mu}}{List of length g; each element is a numeric mean vector (length p).}
\item{\code{sigma}}{Shared covariance matrix (p x p).}
}}
}
\value{
A single numeric value: the negative log-likelihood.
}
\description{
Computes the negative log-likelihood for a semi-supervised Gaussian mixture
model under a mixed missingness mechanism (MCAR + entropy-based MAR).
Assumes a covariance matrix \eqn{\Sigma} across all mixture
components.
}
\details{
The total log-likelihood is composed of three parts:
\enumerate{
\item Labeled samples (\eqn{m_j=0}) with observed class labels \eqn{Z_j}.
\item Unlabeled samples attributed to MCAR with probability mass \eqn{m_{1j}}.
\item Unlabeled samples attributed to MAR with probability mass \eqn{m_{2j}}.
}
The MAR probability for each sample is \eqn{q_j = \text{logit}^{-1}(\xi_0 + \xi_1 d2_{yj})}.
Internally, the function uses a numerically stable \code{logSumExp}.
}
\note{
This implementation is for the equal covariance case (shared \eqn{\Sigma}).
}
\examples{
# Minimal example (illustrative only):
library(mvtnorm)
set.seed(1)
n <- 20; p <- 2; g <- 2
Y <- matrix(rnorm(n*p), n, p)
Z <- sample(c(1:g, rep(NA, n - g)), n, replace = TRUE)
m_j <- ifelse(is.na(Z), 1L, 0L)
d2_yj <- runif(n)
xi <- c(-1, 2)
alpha_k <- 0.4
unpack_fn <- function(theta) {
  list(pi = c(0.6, 0.4),
       mu = list(c(0,0), c(1,1)),
       sigma = diag(p))
}
theta <- numeric(1) # not used in this toy unpack_fn
neg_loglik(theta, Y, m_j, Z, d2_yj, xi, alpha_k, unpack_fn)

}
