\name{plot}
\docType{methods}
\alias{plot}
\alias{plot.strata}
\title{Plot method for stratigraphic and sedimentological data\cr
       (strata class object)}
\description{
Plot method for stratigraphic and sedimentological data. This function will produce a graphic log 
of outcrop section or borehole log in a PDF format (check the working directory for the PDF output file).
}
\usage{
\method{plot}{strata}(x,
  datum = "top",
  data.units = "feet",
  scale = 100,
  barscale = 2,
  lithology = TRUE,
  bed.number = TRUE,
  file.name = "SDAR_graphical_output",
  GR.log = NULL,
  ncore = NULL,
  samples = NULL,
  oil.stain = NULL,
  sed.structures = NULL,
  fossils = NULL, 
  other.sym = NULL,
  bioturbation = NULL,
  lithostrat = NULL,
  chronostrat = NULL,
  legend = TRUE,
  metadata = NULL,
  main = NULL,
  sub = NULL,
  subset.base = NULL,
  subset.top = NULL,
  plot.order = NULL,  
  color = 0,
  fill.pattern = TRUE,
  bar.type = 0,
  w.litho = 2,
  w.litho.unit = "inches",
  symbols.size = 1,
  xlim.GR=c(0, 300),
  family = "serif",   
  fontsize = 10,
  cex.main = 1.5,
  suppress.ref = FALSE,
  \dots)
}
\arguments{
  \item{x}{an object of \link{strata} class to be plotted, which holds the minimum required data to draw a lithological
    profile in SDAR. See \link{strata} class for more details about the requirements of this 'x' object.}
    \item{datum}{string; either 'base' or 'top', case sensitive. 'base' is the case when thickness is measured up from the bottom of, e.g., an outcrop section; 'top' is the case when depths are measured down from the surface, e.g., boreholes and cores.}
    \item{data.units}{string; either 'meters' or 'feet', case sensitive. This argument specifies the unit of measure of the stratigraphic thickness used for input data (thickness measured in field), default unit 'feet'.}
    \item{scale}{default 100, a numerical value giving the graphic vertical scaling. It defines the vertical scale on which the graphic log is drawn; recommended scales are 1:25, 1:50, 1:100, 1:250, 1:500, and 1:1000.}
    \item{barscale}{default 2, a numerical value indicating how often the horizontal tick marks will be displayed on the vertical scale (thickness). The bar scale is displayed using the same units as 'data.units' parameter, and it will be plotted at the left side of the graphic log.}
    \item{lithology}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether lithology should be displayed.}
    \item{bed.number}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether bed number should be displayed.}
    \item{file.name}{string; a file name without extension. Graphic logs generated by SDAR are exported as PDF files (editable with vector graphics software). The paper size is automatically updated when the vertical scale changes, or when different sets of attributes are displayed adjacent to the lithology column, default name "SDAR_graphical_output".}
    \item{GR.log}{a two-column data frame containing the depth, and the total natural radioactivity measured in API units; the columns must be named ('Depth' and 'GR'). See \link{read.LAS} function for details of how to parse geophysical "Log Ascii Standard" files (.las files) into R.}
    \item{ncore}{a data frame containing the depth interval and the core number. It is useful information about how the core is stored. This information should be stored into three columns named ('base', 'top', and 'core_number'), with the information of each core in a separate row containing the depth interval (base and top) and the core number.}
    \item{samples}{a data frame containing the sample dataset. This information should be stored in four columns named ('base', 'top', 'type', and 'label'); the columns 'base' and 'top' should be numeric. Each sample should be in a separate row containing the stratigraphic position (base and top) and the analysis type (e.g., palynology, petrography, rock sample). It is possible to add drawing parameters such as 'pch', and 'color', each one in a separate column in the input data.}
    \item{oil.stain}{a data frame containing the oil.stain dataset. This information should be stored in three columns named ('base', 'top', and 'intensity'); the columns 'base' and 'top' should be numeric. Each staining interval should be in a separate row containing the stratigraphic position (base and top) and the staining intensity (weak, moderate weak, moderate, moderate strong, strong).}
    \item{sed.structures}{a data frame containing the sedimentary structure dataset. This information should be stored in three columns named ('base', 'top', and 'sed_structure'); the columns 'base' and 'top' should be numeric. Each sedimentary structure occurrence should be in a separate row containing the stratigraphic position and the sedimentary structure type; overlapping between sedimentary structures intervals is allowed.}
    \item{fossils}{a data frame containing the fossil dataset. This information should be stored in three columns named ('base', 'top', and 'fossil'); the columns 'base' and 'top' should be numeric. Each fossil occurrence should be in a separate row containing the stratigraphic position and the fossil type. If the fossil occurrence covers a stratigraphic interval, define it using 'base' and 'top' (overlapping between fossil intervals is allowed); in a case where fossil occurrence corresponds to a specific stratigraphic position, fill variables 'base' and 'top' with the same stratigraphic position value.}
    \item{other.sym}{same dataset structure as fossil dataset. Three columns named ('base', 'top', and 'other_symbol'). This is the place to symbolize features as accessory minerals, organic matter, intraclast, etc.}
    \item{bioturbation}{a data frame containing the bioturbation dataset. This information should be stored in three columns named ('base', 'top', and 'index'); the columns 'base' and 'top' should be numeric. Each bioturbated interval should be in a separate row containing the stratigraphic interval (base and top) and the bioturbation index value. The index classifies on a scale of zero to six (Reineck, 1967; modified by Taylor and Goldring, 1993).}
    \item{lithostrat}{a data frame containing the lithostratigraphic data. This information should be stored in four columns named ('base', 'top', 'litho_unit_rank' and 'name'); the columns 'base' and 'top' should be numeric. Each lithostratigraphic interval should be in a separate row containing the stratigraphic interval (base and top), the lithostratigraphic unit rank (e.g., Group, Formation, Member, Informal Member), and the name of the lithostratigraphic unit.}
    \item{chronostrat}{a data frame containing the chronostratigraphic data. This information should be stored in four columns named ('base', 'top', 'chrono_unit' and 'name'); the columns 'base' and 'top' should be numeric. Each chronostratigraphic interval should be in a separate row containing the stratigraphic interval (base and top), the chronostratigraphic unit rank (e.g., system, series, stage), and the name of the chronostratigraphic unit.}
    \item{legend}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether automatic legend should be drawn. The legend function creates a legend based on the lithological, sedimentological, and paleontological data provided for the current stratigraphic section or well. SDAR uses the standard symbols suggested by the \href{http://www.fgdc.gov}{Federal Geographic Data Committee}. The legend will be displayed at the bottom of the PDF file.}
    \item{metadata}{an object of class \link{list}, including any or all of the following named values: locality_name, locality_id, locality_type, thickness_unit, ref_datum, latitude, longitude, elevation, country, author and/or reference. The objects in the list should be named using the previous names 'case sensitive', e.g., list(locality_name = "Saltarin", locality_type = "borehole core", thickness_unit = "meters", country = "Colombia", author ="Ortiz J. & Jaramillo C.").}
    \item{main}{an overall title for the graphic log, the main title (on top).}
    \item{sub}{an overall sub-title for the graphic log, the sub-title (on top).}
    \item{subset.base}{option to draw and analyse a specific interval for a given outcrop section or borehole log. This argument defines the lower limit of the stratigraphic interval of interest.}
    \item{subset.top}{This argument defines the upper limit of the stratigraphic interval of interest.}
    \item{plot.order}{a string vector. This parameter provides a user interaction to arrange (order) the layout scheme. If the user wants to change the default order, this string vector provides the desired order, e.g., c("samples", "ncore", "bed.number", "oil.stain", "lithostrat", "lithology", "fossils", \cr "sed.structures", "other.sym", "bioturbation").}
    \item{color}{numeric 0, 1, 2 or 3; '0 to draw SDAR default colors', '1 to draw using munsell color provided in beds template', '2 to draw using Rcolor provided in beds template' or '3 for black & white option'}
    \item{fill.pattern}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether lithologic pattern should be displayed.}
    \item{bar.type}{numeric 0 or 1; '0 to draw classical black & white bar', or '1 to draw a simple "line" with tick marks'}
    \item{w.litho}{default 2; a numerical value giving the width of the lithology track in the units specified at "w.litho.units" parameter.}
    \item{w.litho.unit}{string; either 'inches' or 'cm', case sensitive. This argument specifies the unit of measure of the "w.litho" parameter, default unit 'inches'.}
    \item{symbols.size}{numeric; a number indicating the amount by which plotting symbols (fossils, sedimentary structures, and other symbols) should be re-scaled relative to the default. 1 = default, 1.2 is 20\% larger, 0.8 is 20\% smaller, etc.}
    \item{xlim.GR}{a range to specify axis limits \emph{c(xmin, xmax)}, its define the minimum and maximun value of the x-axis to display Gamma Ray data.}
    \item{family}{the font family to be used. The default font family is serif, in windows, serif is mapped to "TT Times New Roman".}		
    \item{fontsize}{numeric; number indicating the amount by which plotting text should be scaled relative to default. 1=default, 1.5 is 50\% larger, 0.5 is 50\% smaller, etc.}
    \item{cex.main}{number indicating the amount by which main text should be scaled relative to the default. 1=default, 1.5 is 50\% larger, 0.5 is 50\% smaller, etc.}
    \item{suppress.ref}{default \code{\link[base:logical]{FALSE}}; a logical value indicating whether SDAR reference printed at the top should be suppressed.}
    \item{\dots}{ other arguments }
}
\note{
We have presented a summary of the specific types of data required by the SDAR package, 
along with the formats that should be followed when inputting data to be integrated using SDAR,
to get more details about the specific types of data required by SDAR, check "SDAR_data_model" vignette.
The SDAR project includes the development of a graphic user interface to connect this R package
with a database management system; for this reason the structure of the data and headers
(column names) should be followed in order to match the database structure.\cr
On the \href{https://doi.org/10.25573/data.13118426.v2}{SDAR repository} there are two excel spreadsheet with the suggested format by SDAR, one to store thickness, and texture description of rock layers \emph{SDAR_v0.95_beds_template.xlsx}, and other \emph{SDAR_v0.95_intervals_template.xlsx} to store "intervals information" (e.g., metadata, samples, oil stain, bioturbation, sedimentary structures, fossil content).\cr
If you see problems with the PDF output, remember that the problem is much more likely to be in your viewer
than in R. Try another viewer if possible, browsers as Mozilla Firefox and Google Chrome provide an excellent
rendering engine for PDF files.
}
\author{
John Ortiz, \email{jrortizt@unal.edu.co}
}
\references{
Reineck, H.-E., 1967. \emph{Parameter von Schichtung und bioturbation. Geologischen Rundschau} \bold{56}, 420-438.

Taylor, A.M., Goldring, R., 1993. \emph{Description and analysis of bioturbation and ichnofabric. Journal of the Geological Society of London} \bold{150}, 141-148.
}

\examples{
# --------  OUTPUT  --------
#  The output of this function is a graphic log of outcrop section or borehole log 
#     in a PDF format (check the working directory for the PDF output file, default
#     name "SDAR_graphic_log.pdf").
# --------------------------

# ----------
# example 1: Graphic log of Saltarin 1A core using SDAR default options.  
# ----------

data(saltarin_beds)
saltarin_val <- strata(saltarin_beds)
plot(saltarin_val, data.units="meters", subset.base=50, subset.top=0,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:200", scale=200, 
  file.name=tempfile("saltarin_well_SDAR_demo"))

# ----------
# example 2:   
# ----------

data(saltarin_log)
data(saltarin_beds)
saltarin_val <- strata(saltarin_beds)
saltarin_GR <- saltarin_log[,c("DEPT","GRTH")]
names(saltarin_GR) <- c("Depth", "GR")
plot(saltarin_val, data.units="meters", scale=250, barscale=5,
  subset.base=500, subset.top=450,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:250",
  GR.log=saltarin_GR, file.name=tempfile("log_demo_SDAR"))

# ----------
# example 3:  Read beds information from "SDAR excel spreadsheet format"  
# ----------

library(readxl)
fpath <- system.file("extdata", 
  "SDAR_v0.95_beds_saltarin.xlsx", package = "SDAR")
beds_saltarin <- read_excel(fpath, trim_ws=FALSE)
beds_val <- strata(beds_saltarin, datum="top")
plot(beds_val, data.units="meters", scale=200, 
  subset.base=300, subset.top=250,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:200",
  file.name=tempfile("saltarin_well"))
}
