#' Economic Cycles Dataset
#'
#' A simulated time series representing an economic indicator with regime changes
#' corresponding to expansion, recession, recovery, and stable growth periods.
#'
#' @format A time series object of length 500 with attributes:
#' \describe{
#'   \item{true_changepoints}{Vector of true changepoint locations: c(120, 250, 380)}
#'   \item{description}{Description of the dataset}
#'   \item{regimes}{Names of the regimes: Expansion, Recession, Recovery, Stable Growth}
#' }
#'
#' @details
#' The data simulates monthly economic data from 2000-2041 with four distinct regimes:
#' \itemize{
#'   \item Regime 1 (1-120): Expansion - positive trend, low volatility
#'   \item Regime 2 (121-250): Recession - negative trend, high volatility
#'   \item Regime 3 (251-380): Recovery - positive trend, medium volatility
#'   \item Regime 4 (381-500): Stable Growth - low trend, low volatility
#' }
#'
#' @source Simulated data for package examples
#'
#' @examples
#' \donttest{
#' data(economic_cycles)
#' result <- detect_regimes(economic_cycles, method = "pelt")
#' plot(result)
#'
#' # Compare with true changepoints
#' true_cps <- attr(economic_cycles, "true_changepoints")
#' evaluate(result, true_changepoints = true_cps)
#' }
"economic_cycles"

#' Industrial Sensor Dataset
#'
#' A simulated time series from a manufacturing process with abrupt changes
#' representing process shifts and equipment states.
#'
#' @format A time series object of length 600 with attributes:
#' \describe{
#'   \item{true_changepoints}{Vector of true changepoint locations: c(150, 300, 450)}
#'   \item{description}{Description of the dataset}
#'   \item{regimes}{Names of the regimes: Normal, Drift, Malfunction, Corrected}
#' }
#'
#' @details
#' The data simulates per-minute sensor measurements with four distinct regimes:
#' \itemize{
#'   \item Regime 1 (1-150): Normal operation - mean=50, sd=2
#'   \item Regime 2 (151-300): Process drift - mean=55, sd=2.5
#'   \item Regime 3 (301-450): Equipment malfunction - mean=45, sd=5
#'   \item Regime 4 (451-600): Corrected operation - mean=50, sd=1.5
#' }
#'
#' The data includes autocorrelation typical of industrial processes.
#'
#' @source Simulated data for package examples
#'
#' @examples
#' \donttest{
#' data(industrial_sensor)
#' result <- detect_regimes(industrial_sensor, method = "bocpd")
#' plot(result, type = "segments")
#' }
"industrial_sensor"

#' Simulated Changepoints Benchmark Dataset
#'
#' A collection of simulated datasets with known changepoints for benchmarking
#' changepoint detection methods.
#'
#' @format A list with multiple scenarios:
#' \describe{
#'   \item{single_mean}{Single mean change from 0 to 3}
#'   \item{single_variance}{Single variance change from 1 to 9}
#'   \item{multiple_mean}{Three mean changes}
#'   \item{gradual}{Gradual trend change (challenging)}
#'   \item{small_change}{Small mean change of 0.5 SD (challenging)}
#'   \item{close_changepoints}{Three closely spaced changepoints (challenging)}
#'   \item{heavy_tailed}{Mean change with t-distributed noise}
#'   \item{multivariate}{Bivariate mean and covariance change}
#' }
#'
#' Each scenario contains:
#' \itemize{
#'   \item \code{data}: The time series data
#'   \item \code{true_changepoints}: Vector of true changepoint locations
#'   \item \code{type}: Type of change (mean, variance, etc.)
#'   \item \code{description}: Description of the scenario
#' }
#'
#' @source Simulated data for package benchmarking
#'
#' @examples
#' \donttest{
#' data(simulated_changepoints)
#'
#' # Run benchmark on single mean scenario
#' scenario <- simulated_changepoints$single_mean
#' result <- detect_regimes(scenario$data, method = "pelt")
#' evaluate(result, true_changepoints = scenario$true_changepoints)
#'
#' # Compare multiple methods
#' comparison <- compare_methods(
#'   data = scenario$data,
#'   methods = c("pelt", "bocpd", "binseg"),
#'   true_changepoints = scenario$true_changepoints
#' )
#' print(comparison)
#' }
"simulated_changepoints"

#' Well Log Dataset
#'
#' A simulated well-log porosity dataset with abrupt lithology changes
#' typical of geological formations.
#'
#' @format A numeric vector of length 1000 with attributes:
#' \describe{
#'   \item{true_changepoints}{Vector of true changepoint locations: c(200, 350, 500, 700, 850)}
#'   \item{description}{Description of the dataset}
#'   \item{lithologies}{Names of lithology units}
#' }
#'
#' @details
#' The data simulates porosity measurements from a well log with six distinct
#' lithological units:
#' \itemize{
#'   \item Unit 1 (1-200): Sandstone - porosity ~15%
#'   \item Unit 2 (201-350): Shale - porosity ~8%
#'   \item Unit 3 (351-500): Limestone - porosity ~20%
#'   \item Unit 4 (501-700): Sandstone - porosity ~12%
#'   \item Unit 5 (701-850): Dense Shale - porosity ~5%
#'   \item Unit 6 (851-1000): Sandstone - porosity ~18%
#' }
#'
#' @source Simulated data based on typical well-log characteristics
#'
#' @examples
#' \donttest{
#' data(well_log)
#' result <- detect_regimes(well_log, method = "pelt", min_segment = 50)
#' plot(result, type = "segments")
#'
#' # Compare with true lithology boundaries
#' true_cps <- attr(well_log, "true_changepoints")
#' evaluate(result, true_changepoints = true_cps)
#' }
"well_log"