% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regu-est-c.r
\name{glm.regu}
\alias{glm.regu}
\title{Regularied M-estimation for fitting generalized linear models with a fixed tuning parameter}
\usage{
glm.regu(y, x, iw = NULL, loss = "cal", init = NULL, rhos, test = NULL,
  offs = NULL, id = NULL, Wmat = NULL, Rmat = NULL, zzs = NULL,
  xxs = NULL, n.iter = 100, eps = 1e-06, bt.lim = 3, nz.lab = NULL,
  pos = 10000)
}
\arguments{
\item{y}{An \eqn{n} x \eqn{1} response vector.}

\item{x}{An \eqn{n} x \eqn{p} matix of covariates, excluding a constant.}

\item{iw}{An \eqn{n} x \eqn{1} weight vector.}

\item{loss}{A loss function, which can be specified as "gaus" for continuous responses, or "ml" or "cal" for binary respones.}

\item{init}{A \eqn{(p+1)} x \eqn{1} vector of initial values (the intercept and coefficients).}

\item{rhos}{A \eqn{p} x \eqn{1} vector of Lasso tuning parameters, usually a constant vector, associated with the \eqn{p} coefficients.}

\item{test}{A vector giving the indices of observations between 1 and \eqn{n} which are included in the test set.}

\item{offs}{An \eqn{n} x \eqn{1} vector of offset values, similarly as in \code{glm}.}

\item{id}{An argument which can be used to speed up computation.}

\item{Wmat}{An argument which can be used to speed up computation.}

\item{Rmat}{An argument which can be used to speed up computation.}

\item{zzs}{An argument which can be used to speed up computation.}

\item{xxs}{An argument which can be used to speed up computation.}

\item{n.iter}{The maximum number of iterations allowed. An iteration is defined by computing an quadratic approximation and solving a least-squares Lasso problem.}

\item{eps}{The tolerance at which the difference in the objective (loss plus penalty) values is considered close enough to 0 to declare convergence.}

\item{bt.lim}{The maximum number of backtracking steps allowed.}

\item{nz.lab}{A \eqn{p} x \eqn{1} logical vector (useful for simulations), indicating which covariates are included when calculating the number of nonzero coefficients. 
If \code{nz.lab=NULL}, then \code{nz.lab} is reset to a vector of 0s.}

\item{pos}{A value which can be used to facilitate recording the numbers of nonzero coefficients with or without the restriction by \code{nz.lab}. 
If \code{nz.lab=NULL}, then \code{pos} is reset to 1.}
}
\value{
\item{iter}{The number of iterations performed up to \code{n.iter}.}
\item{conv}{1 if convergence is obtained, 0 if exceeding the maximum number of iterations, or -1 if exceeding maximum number of backtracking steps.}
\item{nz}{A value defined as (nz0 * \code{pos} + nz1) to record the numbers of nonzero coefficients without or with the restriction 
(denoted as nz0 and nz1) by \code{nz.lab}.
If \code{nz.lab=NULL}, then nz1 is 0, \code{pos} is 1, and hence \code{nz} is nz0.}
\item{inter}{The estimated intercept.}
\item{bet}{The \eqn{p} x \eqn{1} vector of estimated coefficients, excluding the intercept.}
\item{fit}{The vector of fitted values in the training set.}
\item{eta}{The vector of linear predictors in the training set.}
\item{tau}{The \eqn{p} x \eqn{1} vector of generalized signs, which should be -1 or 1 for a negative or positive estimate and between -1 and 1 for a zero estimate.}
\item{obj.train}{The average loss in the training set.}
\item{pen}{The Lasso penalty of the estimates.}
\item{obj}{The average loss plus the Lasso penalty.}
\item{fit.test}{The vector of fitted values in the test set.}
\item{eta.test}{The vector of linear predictors in the test set.}
\item{obj.test}{The average loss in the test set.}
\item{id}{This can be re-used to speed up computation.}
\item{Wmat}{This can be re-used to speed up computation.}
\item{Rmat}{This can be re-used to speed up computation.}
\item{zzs}{This can be re-used to speed up computation.}
\item{xxs}{This can be re-used to speed up computation.}
}
\description{
This function implements regularized M-estimation for fitting generalized linear models with continuous or binary responses 
for a fixed choice of tuning parameters.
}
\details{
For continuous responses, this function uses an active-set descent algorithm (Osborne et al. 2000; Yang and Tan 2018) to solve the least-squares Lasso problem. 
For binary responses, regularized calibrated estimation is implemented using the Fisher scoring descent algorithm in Tan (2020), whereas
regularized maximum likelihood estimation is implemented in a similar manner based on quadratic approximation as in the R package \pkg{glmnet}.
}
\examples{
data(simu.data)
n <- dim(simu.data)[1]
p <- dim(simu.data)[2]-2

y <- simu.data[,1]
tr <- simu.data[,2]
x <- simu.data[,2+1:p]
x <- scale(x)

### Example 1: linear regression
# rhos should be a vector of length p, even though a constant vector
out.rgaus <- glm.regu(y[tr==1], x[tr==1,], rhos=rep(.05,p), loss="gaus")

# the intercept
out.rgaus$inter

# the estimated coefficients and generalized signs; the first 10 are shown
cbind(out.rgaus$bet, out.rgaus$tau)[1:10,]

# the number of nonzero coefficients 
out.rgaus$nz

### Example 2: logistic regression using likelihood loss
out.rml <- glm.regu(tr, x, rhos=rep(.01,p), loss="ml")
out.rml$inter
cbind(out.rml$bet, out.rml$tau)[1:10,]
out.rml$nz

### Example 3: logistic regression using calibration loss
out.rcal <- glm.regu(tr, x, rhos=rep(.05,p), loss="cal")
out.rcal$inter
cbind(out.rcal$bet, out.rcal$tau)[1:10,]
out.rcal$nz

}
\references{
Osborne, M., Presnell, B., and Turlach, B. (2000) A new approach to variable selection in least squares problems, \emph{IMA Journal of Numerical Analysis}, 20, 389-404.

Yang, T. and Tan, Z. (2018) Backfitting algorithms for total-variation and empirical-norm penalized additive modeling with high-dimensional data, \emph{Stat}, 7, e198.

Tibshirani, R. (1996) Regression shrinkage and selection via the Lasso, \emph{Journal of the Royal Statistical Society}, Ser. B, 58, 267-288.

Tan, Z. (2020) Regularized calibrated estimation of propensity scores with model misspecification and high-dimensional data, \emph{Biometrika}, 107, 137–158.
}
