% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postmix.R
\name{postmix}
\alias{postmix}
\alias{postmix.betaMix}
\alias{postmix.normMix}
\alias{postmix.gammaMix}
\title{Conjugate Posterior Analysis}
\usage{
postmix(priormix, data, ...)

\method{postmix}{betaMix}(priormix, data, n, r, ...)

\method{postmix}{normMix}(priormix, data, n, m, se, ...)

\method{postmix}{gammaMix}(priormix, data, n, m, ...)
}
\arguments{
\item{priormix}{prior (mixture of conjugate distributions).}

\item{data}{individual data. If the individual data is not given, then
summary data has to be provided (see below).}

\item{...}{includes arguments which depend on the specific case, see description below.}

\item{n}{sample size.}

\item{r}{Number of successes.}

\item{m}{Sample mean.}

\item{se}{Sample standard error.}
}
\description{
Calculates the posterior distribution for data \code{data} given a prior
\code{priormix}, where the prior is a mixture of conjugate distributions.
The posterior is then also a mixture of conjugate distributions.
}
\details{
A conjugate prior-likelihood pair has the convenient
property that the posterior is in the same distributional class as
the prior. This property also applies to mixtures of conjugate
priors. Let

\deqn{p(\theta;\mathbf{w},\mathbf{a},\mathbf{b})}{p(\theta;w,a,b)}

denote a conjugate mixture prior density for data

\deqn{y|\theta \sim f(y|\theta),}{y|\theta ~ f(y|\theta),}

where \eqn{f(y|\theta)} is the likelihood. Then the posterior is
again a mixture with each component \eqn{k} equal to the respective
posterior of the \eqn{k}th prior component and updated weights
\eqn{w'_k},

\deqn{p(\theta;\mathbf{w'},\mathbf{a'},\mathbf{b'}|y) = \sum_{k=1}^K w'_k \, p_k(\theta;a'_k,b'_k|y).}{p(\theta;w',a',b'|y) = \sum_{k=1}^K w'_k * p(\theta;a'_k,b'_k|y).}

The weight \eqn{w'_k} for \eqn{k}th component is determined by the
marginal likelihood of the new data \eqn{y} under the \eqn{k}th prior
distribution which is given by the predictive distribution of the
\eqn{k}th component,

\deqn{w'_k \propto w_k \, \int p_k(\theta;a_k,b_k) \, f(y|\theta) \, d\theta \equiv w^\ast_k .}{w'_k \propto w_k \int p_k(u;a_k,b_k) f(y|u) du = w^*_k .}

The final weight \eqn{w'_k} is then given by appropriate
normalization, \eqn{w'_k = w^\ast_k / \sum_{k=1}^K w^\ast_k}{w'_k =
w^*_k / \sum_{k=1}^K w^*_k}. In other words, the weight of
component \eqn{k} is proportional to the likelihood that data
\eqn{y} is generated from the respective component, i.e. the
marginal probability; for details, see for example \emph{Schmidli
et al., 2015}.

\emph{Note:} The prior weights \eqn{w_k} are fixed, but the
posterior weights \eqn{w'_k \neq w_k} still change due to the
changing normalization.

The data \eqn{y} can either be given as individual data or as
summary data (sufficient statistics). See below for details for the
implemented conjugate mixture prior densities.
}
\section{Methods (by class)}{
\itemize{
\item \code{postmix(betaMix)}: Calculates the posterior beta mixture
distribution. The individual data vector is expected to be a vector
of 0 and 1, i.e. a series of Bernoulli experiments. Alternatively,
the sufficient statistics \code{n} and \code{r} can be given,
i.e. number of trials and successes, respectively.

\item \code{postmix(normMix)}: Calculates the posterior normal mixture
distribution with the sampling likelihood being a normal with fixed
standard deviation. Either an individual data vector \code{data}
can be given or the sufficient statistics which are the standard
error \code{se} and sample mean \code{m}. If the sample size
\code{n} is used instead of the sample standard error, then the
reference scale of the prior is used to calculate the standard
error. Should standard error \code{se} and sample size \code{n} be
given, then the reference scale of the prior is updated; however it
is recommended to use the command \code{\link[=sigma]{sigma()}} to set the
reference standard deviation.

\item \code{postmix(gammaMix)}: Calculates the posterior gamma mixture
distribution for Poisson and exponential likelihoods. Only the
Poisson case is supported in this version.

}}
\section{Supported Conjugate Prior-Likelihood Pairs}{


\tabular{lccc}{
\strong{Prior/Posterior} \tab \strong{Likelihood} \tab \strong{Predictive}
\tab \strong{Summaries} \cr
Beta \tab Binomial \tab Beta-Binomial \tab \code{n}, \code{r} \cr
Normal \tab Normal (\emph{fixed \eqn{\sigma}}) \tab Normal \tab \code{n}, \code{m}, \code{se}  \cr
Gamma \tab Poisson \tab Gamma-Poisson \tab  \code{n}, \code{m} \cr
Gamma \tab Exponential \tab Gamma-Exp (\emph{not supported}) \tab \code{n}, \code{m}
}
}

\examples{

# binary example with individual data (1=event,0=no event), uniform prior
prior.unif <- mixbeta(c(1, 1, 1))
data.indiv <- c(1, 0, 1, 1, 0, 1)
posterior.indiv <- postmix(prior.unif, data.indiv)
print(posterior.indiv)
# or with summary data (number of events and number of patients)
r <- sum(data.indiv)
n <- length(data.indiv)
posterior.sum <- postmix(prior.unif, n = n, r = r)
print(posterior.sum)

# binary example with robust informative prior and conflicting data
prior.rob <- mixbeta(c(0.5, 4, 10), c(0.5, 1, 1))
posterior.rob <- postmix(prior.rob, n = 20, r = 18)
print(posterior.rob)

# normal example with individual data
sigma <- 88
prior.mean <- -49
prior.se <- sigma / sqrt(20)
prior <- mixnorm(c(1, prior.mean, prior.se), sigma = sigma)
data.indiv <- c(-46, -227, 41, -65, -103, -22, 7, -169, -69, 90)
posterior.indiv <- postmix(prior, data.indiv)
# or with summary data (mean and number of patients)
mn <- mean(data.indiv)
n <- length(data.indiv)
posterior.sum <- postmix(prior, m = mn, n = n)
print(posterior.sum)

}
\references{
Schmidli H, Gsteiger S, Roychoudhury S, O'Hagan A, Spiegelhalter D, Neuenschwander B.
Robust meta-analytic-predictive priors in clinical trials with historical control information.
\emph{Biometrics} 2014;70(4):1023-1032.
}
