test_that("summariseConceptIdCount works", {
  skip_on_cran()

  cdm <- cdmEunomia()

  expect_true(inherits(summariseConceptIdCounts(cdm, "drug_exposure"), "summarised_result"))
  expect_error(summariseConceptIdCounts(cdm, "observation_period"))
  expect_no_error(x <- summariseConceptIdCounts(cdm, "visit_occurrence"))
  checkResultType(x, "summarise_concept_id_counts")
  expect_no_error(summariseConceptIdCounts(cdm, "condition_occurrence", countBy = c("record", "person")))
  expect_no_error(summariseConceptIdCounts(cdm, "drug_exposure"))
  expect_no_error(summariseConceptIdCounts(cdm, "procedure_occurrence", countBy = "person"))
  expect_warning(summariseConceptIdCounts(cdm, "device_exposure"))
  expect_no_error(y <- summariseConceptIdCounts(cdm, "measurement"))
  expect_no_error(summariseConceptIdCounts(cdm, "observation", interval = "quarters"))
  expect_warning(p <- summariseConceptIdCounts(cdm, "death"))

  expect_no_error(all <- summariseConceptIdCounts(cdm, c("visit_occurrence", "measurement")))
  expect_equal(all |> sortTibble(), x |> dplyr::bind_rows(y) |> sortTibble())
  expect_equal(
    summariseConceptIdCounts(cdm, "procedure_occurrence", countBy = "record") |>
      sortTibble(),
    summariseConceptIdCounts(cdm, "procedure_occurrence") |>
      sortTibble()
  )

  expect_equal(
    summariseConceptIdCounts(cdm, "procedure_occurrence", countBy = "record", interval = "overall") |>
      sortTibble(),
    summariseConceptIdCounts(cdm, "procedure_occurrence", countBy = "record", interval = "months") |>
      omopgenerics::splitAdditional() |>
      dplyr::filter(.data$time_interval == "overall") |>
      omopgenerics::uniteAdditional(cols = c("time_interval", "source_concept_id", "source_concept_name")) |>
      sortTibble(),
    ignore_attr = TRUE
  )

  expect_error(summariseConceptIdCounts(cdm, omopTableName = ""))
  expect_error(summariseConceptIdCounts(cdm, omopTableName = "visit_occurrence", countBy = "dd"))
  expect_equal(settings(y)$result_type, settings(p)$result_type)
  expect_true(summariseConceptIdCounts(cdm, "procedure_occurrence", sex = TRUE, ageGroup = list(c(0, 50), c(51, Inf))) |>
    dplyr::distinct(.data$strata_level) |>
    dplyr::tally() |>
    dplyr::pull() == 9)

  expect_true(summariseConceptIdCounts(cdm, "procedure_occurrence", ageGroup = list(c(0, 50))) |>
    dplyr::distinct(.data$strata_level) |>
    dplyr::tally() |>
    dplyr::pull() == 3)

  s <- summariseConceptIdCounts(cdm, "procedure_occurrence") |>
    sortTibble()
  z <- summariseConceptIdCounts(cdm, "procedure_occurrence", sex = TRUE, interval = "years", ageGroup = list(c(0, 50), c(51, Inf))) |>
    sortTibble()

  x <- z |>
    omopgenerics::splitAdditional() |>
    dplyr::filter(strata_level == "overall" & time_interval == "overall") |>
    dplyr::select(variable_level, estimate_value)
  s <- s |>
    dplyr::select(variable_level, estimate_value)
  expect_equal(x, s, ignore_attr = TRUE)

  x <- z |>
    omopgenerics::splitAdditional() |>
    dplyr::filter(strata_name == "age_group" & time_interval == "overall") |>
    dplyr::group_by(variable_level) |>
    dplyr::summarise(estimate_value = sum(as.numeric(estimate_value), na.rm = TRUE), .groups = "drop") |>
    dplyr::mutate(estimate_value = as.character(estimate_value))

  p <- s |>
    dplyr::select(variable_level, estimate_value)

  expect_true(all.equal(
    as.data.frame(x) |> dplyr::arrange(variable_level),
    as.data.frame(p) |> dplyr::arrange(variable_level),
    check.attributes = FALSE
  ))

  dropCreatedTables(cdm = cdm)
})

test_that("dateRange argument works", {
  skip_on_cran()
  # Load mock database ----
  cdm <- cdmEunomia()
  expect_no_error(summariseConceptIdCounts(cdm, "condition_occurrence", dateRange = as.Date(c("2012-01-01", "2018-01-01"))))
  expect_message(x <- summariseConceptIdCounts(cdm, "drug_exposure", dateRange = as.Date(c("2012-01-01", "2025-01-01"))))
  observationRange <- cdm$observation_period |>
    dplyr::summarise(
      minobs = min(.data$observation_period_start_date, na.rm = TRUE),
      maxobs = max(.data$observation_period_end_date, na.rm = TRUE)
    )
  expect_no_error(y <- summariseConceptIdCounts(cdm, "drug_exposure", dateRange = as.Date(c("2012-01-01", observationRange |> dplyr::pull("maxobs")))))
  expect_equal(x |> sortTibble(), y |> sortTibble(), ignore_attr = TRUE)
  expect_false(settings(x)$study_period_end == settings(y)$study_period_end)
  expect_error(summariseConceptIdCounts(cdm, "drug_exposure", dateRange = as.Date(c("2015-01-01", "2014-01-01"))))
  expect_warning(y <- summariseConceptIdCounts(cdm, "drug_exposure", dateRange = as.Date(c("2020-01-01", "2021-01-01"))))
  expect_equal(y, omopgenerics::emptySummarisedResult(), ignore_attr = TRUE)
  expect_equal(settings(y)$result_type, settings(x)$result_type)
  expect_equal(colnames(settings(y)), colnames(settings(x)))

  dropCreatedTables(cdm = cdm)
})

test_that("tableConceptIdCounts() works", {
  skip_on_cran()
  # Load mock database ----
  cdm <- cdmEunomia()

  # Check that works ----
  expect_no_error(x <- tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence")))
  expect_true(inherits(x, "reactable"))
  expect_no_error(y <- tableConceptIdCounts(summariseConceptIdCounts(cdm, c(
    "drug_exposure",
    "measurement"
  ))))
  expect_true(inherits(y, "reactable"))
  expect_warning(t <- summariseConceptIdCounts(cdm, "death"))
  expect_warning(inherits(tableConceptIdCounts(t), "reactable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "overall", type = "datatable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "standard", type = "datatable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "source", type = "datatable"))
  expect_warning(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "missing source", type = "datatable"))
  expect_warning(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "missing standard", type = "datatable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "overall", type = "reactable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "standard", type = "reactable"))
  expect_no_error(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "source", type = "reactable"))
  expect_warning(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "missing source", type = "reactable"))
  expect_warning(tableConceptIdCounts(summariseConceptIdCounts(cdm, "condition_occurrence"), display = "missing standard", type = "reactable"))
  cdm$drug_exposure <- cdm$drug_exposure |>
    dplyr::mutate(drug_source_concept_id = dplyr::if_else(.data$drug_exposure_id %%2 == 0, 0L, .data$drug_source_concept_id ))
  res <- summariseConceptIdCounts(cdm, "drug_exposure", countBy = c("record", "person"))
  expect_no_error(tableConceptIdCounts(res, display = "source"))
  expect_no_error(tableConceptIdCounts(res, display = "standard"))
  expect_warning(tableConceptIdCounts(omopgenerics::emptySummarisedResult()))

  dropCreatedTables(cdm = cdm)
})

test_that("interval argument works", {
  skip_on_cran()
  # Load mock database ----
  cdm <- cdmEunomia()
  expect_no_error(y <- summariseConceptIdCounts(
    cdm = cdm,
    omopTableName = "drug_exposure",
    interval = "years"
  ))

  expect_no_error(o <- summariseConceptIdCounts(
    omopTableName = "drug_exposure",
    cdm = cdm,
    interval = "overall"
  ))
  expect_no_error(q <- summariseConceptIdCounts(
    omopTableName = "drug_exposure",
    cdm = cdm,
    interval = "quarters"
  ))
  expect_no_error(m <- summariseConceptIdCounts(
    omopTableName = "drug_exposure",
    cdm = cdm,
    interval = "months"
  ))

  m_quarters <- m |>
    omopgenerics::splitAdditional() |>
    omopgenerics::pivotEstimates() |>
    dplyr::filter(time_interval != "overall") |>
    dplyr::mutate(
      start_date = as.Date(sub(" to .*", "", time_interval)),
      quarter_start = lubridate::quarter(start_date, type = "date_first"),
      quarter_end = lubridate::quarter(start_date, type = "date_last"),
      quarter = paste(quarter_start, "to", quarter_end)
    ) |>
    dplyr::select(!c("time_interval", "start_date", "quarter_start", "quarter_end")) |>
    dplyr::group_by(quarter, variable_level) |>
    dplyr::summarise(count_records = sum(count_records), .groups = "drop") |>
    dplyr::rename("time_interval" = quarter) |>
    dplyr::arrange(time_interval)

  q_quarters <- q |>
    omopgenerics::splitAdditional() |>
    omopgenerics::pivotEstimates() |>
    dplyr::filter(time_interval != "overall") |>
    dplyr::select(time_interval, variable_level, count_records) |>
    dplyr::arrange(time_interval)

  expect_equal(m_quarters |>
    sortTibble(), q_quarters |> sortTibble())

  m_year <- m |>
    omopgenerics::splitAdditional() |>
    dplyr::filter(time_interval != "overall") |>
    dplyr::mutate(
      # Extract the start date
      start_date = clock::date_parse(stringr::str_extract(time_interval, "^\\d{4}-\\d{2}-\\d{2}")),
      # Convert start_date to a year-month-day object and extract the year
      year = clock::get_year(clock::as_year_month_day(start_date))
    ) |>
    omopgenerics::pivotEstimates() |>
    dplyr::group_by(year, variable_level) |>
    dplyr::summarise(
      count_records = sum(count_records),
      .groups = "drop"
    ) |>
    dplyr::arrange(year)
  y_year <- y |>
    omopgenerics::splitAdditional() |>
    dplyr::filter(time_interval != "overall") |>
    dplyr::mutate(
      # Extract the start date
      start_date = clock::date_parse(stringr::str_extract(time_interval, "^\\d{4}-\\d{2}-\\d{2}")),
      # Convert start_date to a year-month-day object and extract the year
      year = clock::get_year(clock::as_year_month_day(start_date))
    ) |>
    omopgenerics::pivotEstimates() |>
    dplyr::select(year, variable_level, count_records) |>
    dplyr::arrange(year)

  expect_equal(m_year |> sortTibble(), y_year |> sortTibble())

  o <- o |>
    omopgenerics::splitAdditional() |>
    omopgenerics::pivotEstimates() |>
    dplyr::select(variable_level, count_records)

  expect_equal(y_year |> dplyr::group_by(variable_level) |> dplyr::summarise(count_records = sum(count_records), .groups = "drop") |> sortTibble(), o |> sortTibble())

  q_year <- q |>
    omopgenerics::splitAdditional() |>
    dplyr::filter(time_interval != "overall") |>
    dplyr::mutate(
      # Extract the start date
      start_date = clock::date_parse(stringr::str_extract(time_interval, "^\\d{4}-\\d{2}-\\d{2}")),
      # Convert start_date to a year-month-day object and extract the year
      year = clock::get_year(clock::as_year_month_day(start_date))
    ) |>
    omopgenerics::pivotEstimates() |>
    dplyr::group_by(year, variable_level) |>
    dplyr::summarise(
      count_records = sum(count_records),
      .groups = "drop"
    ) |>
    dplyr::arrange(year)

  expect_equal(q_year |> sortTibble(), y_year |> sortTibble())

  dropCreatedTables(cdm = cdm)
})

test_that("tableTopConceptCounts works", {
  skip_on_cran()
  cdm <- cdmEunomia()

  expect_no_error(result <- summariseConceptIdCounts(cdm, "drug_exposure", sex = TRUE, ageGroup = list(c(0, 50))))
  expect_no_error(tableTopConceptCounts(result))
  expect_no_error(tableTopConceptCounts(result, top = 5))
  expect_error(tableTopConceptCounts(result, top = 0.5))
  expect_no_error(tableTopConceptCounts(result, top = 10000000))

  result <- summariseConceptIdCounts(cdm, "drug_exposure")
  result <- bind(result, result |> dplyr::mutate(cdm_name = "mock"))

  expect_no_error(tableTopConceptCounts(result))
  expect_no_error(tableTopConceptCounts(result, type = "datatable"))
  expect_no_error(tableTopConceptCounts(result, type = "flextable"))
  expect_no_error(result <- summariseConceptIdCounts(cdm, "drug_exposure", interval = "months"))
  expect_no_error(tableTopConceptCounts(result))

  expect_no_error(result <- summariseConceptIdCounts(cdm, "drug_exposure", countBy = c("record", "person")))
  expect_no_error(tableTopConceptCounts(result, countBy = "record"))
  set <- omopgenerics::settings(result) |>
    dplyr::mutate(test = "test")
  result <- omopgenerics::newSummarisedResult(result, settings = set)
  expect_no_error(tableTopConceptCounts(result, countBy = "record"))
  expect_no_error(tableTopConceptCounts(result, countBy = "record", type = "reactable"))
  expect_warning(tableTopConceptCounts(omopgenerics::emptySummarisedResult()))

  dropCreatedTables(cdm = cdm)
})

test_that("inObservation argument works", {
  skip_on_cran()
  cdm <- cdmEunomia()

  expect_no_error(result <- summariseConceptIdCounts(cdm, "drug_exposure", inObservation = TRUE))
  x <- result |>
    omopgenerics::filterStrata(.data$in_observation == "TRUE") |>
    dplyr::select(!c("strata_name", "strata_level"))

  cdm$drug_exposure <- cdm$drug_exposure |>
    dplyr::inner_join(cdm[["observation_period"]] |>
      dplyr::select(
        "obs_start" = "observation_period_start_date",
        "obs_end" = "observation_period_end_date",
        "person_id"
      ), by = "person_id") |>
    dplyr::filter(.data$drug_exposure_start_date >= .data$obs_start & .data$drug_exposure_start_date <= .data$obs_end)

  expect_no_error(resultInObs <- summariseConceptIdCounts(cdm, "drug_exposure"))
  expect_equal(
    x |>
      dplyr::arrange(.data$variable_level),
    resultInObs |>
      dplyr::select(!c("strata_name", "strata_level")) |>
      dplyr::arrange(.data$variable_level),
    ignore_attr = TRUE
  )

  expect_no_error(summariseConceptIdCounts(cdm, "drug_exposure", inObservation = TRUE, sex = TRUE, ageGroup = list(c(0, 70))))

  dropCreatedTables(cdm = cdm)
})

test_that("sample argument works", {
  skip_on_cran()
  # Load mock database ----
  cdm <- cdmEunomia()

  n_person <- cdm$person |> dplyr::tally() |> dplyr::pull("n")
  expect_no_error(summariseConceptIdCounts(cdm, "drug_exposure", sample =  n_person - 1))
  expect_message(summariseConceptIdCounts(cdm, "drug_exposure", sample = n_person))
  expect_message(summariseConceptIdCounts(cdm, "drug_exposure", sample = n_person+1))
  expect_message(summariseConceptIdCounts(cdm, "drug_exposure", sample = "pajfn"))
  cdm[["pharyngitis"]] <- CohortConstructor::conceptCohort(cdm, conceptSet = list("pharyngitis" = 4112343L), name = "pharyngitis")
  expect_no_error(x <- summariseConceptIdCounts(cdm, "drug_exposure", sample = "pharyngitis"))

  expect_no_error(x <- summariseConceptIdCounts(cdm, "condition_occurrence", sample =  n_person - 1, countBy = "person"))
  expect_true(all(x |> omopgenerics::tidy() |> dplyr::pull("count_subjects") <= n_person - 1))

  expect_no_error(x <- summariseConceptIdCounts(cdm, "condition_occurrence", sample = "pharyngitis", countBy = "record"))
  expect_equal(x |> omopgenerics::tidy() |> dplyr::filter(.data$variable_level == "4112343") |> dplyr::pull("count_records"),  omopgenerics::numberRecords(cdm$pharyngitis))

  dropCreatedTables(cdm = cdm)
})
