#
#  rmutil : A Library of Special Functions for Repeated Measurements
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     covind(z, ...)
#     wr(formula)
#     det(x)
#     x%**%y
#     orth(x,order=2)
#     collapse(x,index,fcn=sum)
#     mexp(x, t=1, n=20, k=3)
#
#  DESCRIPTION
#
#    Utility functions for repeated measurements

covind <- function(z, ...)
	UseMethod("covind")

covind.response <- function(z)
	rep(1:length(z$nobs),z$nobs)

covind.default <- function(z){
	if(!is.null(z$response$nobs))
		rep(1:length(z$response$nobs),z$response$nobs)
	else NULL}

wr <- function(formula)
{
	mt <- terms(formula)
	data <- sys.frame(sys.parent())
	mf <- model.frame(mt, data, na.action=na.fail)
	x <- model.matrix(mt, mf)
	y <- model.response(mf, "numeric")
	z <- list(response=y, design=x)
	z
}

# det <- function(x) abs(Re(prod(eigen(x,only.values=T)$values)))
det <- function(x) abs(prod(diag(qr(x)$qr)))

# kronecker product
"%**%" <- function(x,y) {
	if(prod(dim(x))<prod(dim(y))){
		z <- NULL
		for(i in 1:nrow(x)){
			tmp <- NULL
			for(j in 1:ncol(x))tmp <- cbind(tmp,x[i,j]*y)
			z <- rbind(z,tmp)}}
	else {
#		nc <- ncol(x)*ncol(y)
#		nr <- nrow(x)*nrow(y)
#		z <- matrix(0,ncol=nc,nrow=nr)
		z <- matrix(0,ncol=ncol(x)*ncol(y),nrow=nrow(x)*nrow(y))
		for(i in 1:nrow(y))for(j in 1:ncol(y)){
#			z[seq(i,nr,by=nrow(y)),seq(j,nc,by=ncol(y))] <- x*y[i,j]}}
			z[(0:(nrow(x)-1))*nrow(y)+i,(0:(ncol(x)-1))*ncol(y)+j] <- x*y[i,j]}}
	z}

orth <- function(x,order=2){
	if(order>length(unique(x))-1)order <- length(unique(x))-1
	z3 <- length(x)
	z <- rep(1,z3)
	tt <- x-sum(x)/z3
	z <- rbind(z,tt/sqrt(sum(tt*tt)))
	if(order<2)return(z)
	for(i in 2:order){
		xx <-x^i
		zz <- xx-sum(xx)/z3
		for(j in 2:i)zz <- zz-z[j,]*sum(xx*z[j,])
		z <- rbind(z,zz/sqrt(sum(zz*zz)))}
	z}

collapse <- function(x,index,fcn=sum){
	ans <- NULL
	for(i in split(x,index))ans <- c(ans,fcn(i))
	ans}

mexp <- function(x, type="spectral decomposition", t=1, n=20, k=3){
	if(!is.matrix(x))stop("x must be a matrix")
	if(length(dim(x))!=2)stop("x must be a two dimensional matrix")
	if(dim(x)[1]!=dim(x)[2])stop("x must be a square matrix")
	type <- match(type,c("spectral decomposition","series approximation"))
	if(type=="spectral decomposition"){
		z <- eigen(t*x,sym=F)
		p <- z$vectors%*%diag(exp(z$values))%*%solve(z$vectors)}
	else {
		xx <- x*t/2^k
		p <- diag(dim(x)[1])
		q <- p
		for(r in 1:n){
			q <- xx%*%q/r
			p <- p+q}
		for(i in 1:k) p <- p%*%p}
	p}
