#
#  event : A Library of Special Functions for Event Histories
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#  kalsurv(response, intensity="exponential", dist="Pareto",
#	depend="independence", update="Markov", mu=NULL, shape=NULL,
#	renewal=T, density=F, censor=NULL, ccov=NULL, tvcov=NULL,
#	pinitial=1, pdepend=NULL, pshape=NULL, preg=NULL, pbirth=NULL,
#	ptvc=NULL, pintercept=NULL, print.level=0, ndigit=10,
#	gradtol=0.00001, steptol=0.00001, iterlim=100, fscale=1,
#	typsiz=abs(p), stepmax=10*sqrt(p%*%p))
#
#  DESCRIPTION
#
#    Function to fit various distributions inserted in a Pareto, gamma, or
# Weibull distribution with serial dependence or gamma frailties using
# Kalman-type update for event histories.

kalsurv <- function(response, intensity="exponential", dist="Pareto",
	depend="independence", update="Markov", mu=NULL, shape=NULL,
	renewal=T, density=F, censor=NULL, ccov=NULL, tvcov=NULL,
	pinitial=1, pdepend=NULL, pshape=NULL, preg=NULL, pbirth=NULL,
	ptvc=NULL, pintercept=NULL, print.level=0, ndigit=10,
	gradtol=0.00001, steptol=0.00001, iterlim=100, fscale=1,
	typsiz=abs(p), stepmax=10*sqrt(p%*%p)){
ksurvb <- function(p){
	if(rf)b <- mu(p)
	if(sf)v <- shape(p[nps1:np])
	z <- .C("ksurvb",
		p=as.double(p),
		y=as.double(zna$response$y),
		x=as.double(zna$ccov$ccov),
		cens=as.integer(zna$response$censor),
		nind=as.integer(nind),
		nobs=as.integer(zna$response$nobs),
		nbs=as.integer(length(zna$response$y)),
		nccov=as.integer(nccov),
		model=as.integer(mdl),
		density=as.integer(density),
		dep=as.integer(dep),
		birth=as.integer(birth),
		tvc=as.integer(tvc),
		tvcov=as.double(zna$tvcov$tvcov),
		fit=as.integer(0),
		pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		renewal=as.integer(renewal),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))
	z$like}
ksurvg <- function(p){
	if(rf)b <- mu(p)
	if(sf)v <- shape(p[nps1:np])
	z <- .C("ksurvg",
		p=as.double(p),
		y=as.double(zna$response$y),
		x=as.double(zna$ccov$ccov),
		cens=as.integer(zna$response$censor),
		nind=as.integer(nind),
		nobs=as.integer(zna$response$nobs),
		nbs=as.integer(length(zna$response$y)),
		nccov=as.integer(nccov),
		model=as.integer(mdl),
		dist=as.integer(dst),
		density=as.integer(density),
		dep=as.integer(dep),
		birth=as.integer(birth),
		tvc=as.integer(tvc),
		tvcov=as.double(zna$tvcov$tvcov),
		fit=as.integer(0),
		pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		renewal=as.integer(renewal),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))
	z$like}
frailb <- function(p){
	if(rf)b <- mu(p)
	if(sf)v <- shape(p[nps1:np])
	z <- .C("frailb",
		p=as.double(p),
		y=as.double(zna$response$y),
		x=as.double(zna$ccov$ccov),
		cens=as.integer(zna$response$censor),
		nind=as.integer(nind),
		nobs=as.integer(zna$response$nobs),
		nbs=as.integer(length(zna$response$y)),
		nccov=as.integer(nccov),
		model=as.integer(mdl),
		density=as.integer(density),
		dep=as.integer(dep),
		birth=as.integer(birth),
		tvc=as.integer(tvc),
		tvcov=as.double(zna$tvcov$tvcov),
		fit=as.integer(0),
		pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))
	z$like}
call <- sys.call()
tmp <- c("exponential","Weibull","gamma","gen logistic",
	"log normal","log logistic","log Cauchy","log Laplace")
mdl <- match(intensity <- match.arg(intensity,tmp),tmp)
tmp <- c("Pareto","gamma","Weibull")
dst <- match(dist <- match.arg(dist,tmp),tmp)
depend <- match.arg(depend,c("independence","serial","frailty"))
tmp <- c("elapsed Markov","serial","Markov","event","cumulated","count","kalman","time")
dep <- match(update <- match.arg(update,tmp),tmp)
v <- b <- NULL
rf <- !missing(mu)
sf <- !missing(shape)
if(rf&&!is.function(mu))stop("mu must be a function")
if(sf&&!is.function(shape))stop("shape must be a function")
npreg <- length(preg)
birth <- !missing(pbirth)
tvc <- length(ptvc)
if(rf&&birth)stop("Birth models cannot be fitted with a mean function")
if(intensity=="exponential"){
	sf <- F
	pshape <- NULL}
else {
	if(missing(pshape))
		stop("Initial value of the shape parameter must be supplied")
	if(!sf){
		if(pshape<=0)stop("Shape must be positive")
		else pshape <- log(pshape)}}
if(intensity=="gen logistic"){
	if(is.null(pintercept))stop("Initial value of the intercept parameter must be supplied")}
else pintercept <- NULL
nps <- length(pshape)
np <- 1+npreg+(depend=="serial")+birth+nps+!is.null(pintercept)
if(pinitial<=0)stop("Estimate of initial parameter must be > 0")
else pinitial <- log(pinitial)
if(depend=="independence"){
	pdepend <- NULL
	dep <- 0}
else if(depend=="serial"){
	if(update=="time")stop("time update can only be used with frailty")
	if(missing(pdepend))
		stop("An estimate of the dependence parameter must be supplied")
	else if(pdepend<=0|pdepend>=1)
		stop("Dependence parameter must be between 0 and 1")
	else pdepend <- log(pdepend/(1-pdepend))}
else if(depend=="frailty"){
	if(update=="time")dep <- 1
	else {
		dep <- 0
		update <- "no"}
	if(!missing(pdepend))pdepend <- NULL}
if(missing(ccov)){
	nccov <- 0
	zc <- NULL}
else {
	if(is.vector(ccov,mode="double")||(is.matrix(ccov)&&is.null(colnames(ccov))))
		ccname <- paste(deparse(substitute(ccov)))
	else ccname <- NULL
	zc <- tcctomat(ccov,names=ccname)
	nccov <- ncol(zc$ccov)}
if(rf&&npreg>0)nccov <- npreg-1
if(!rf&&nccov+1!=npreg)stop(paste(nccov+1,"regression estimates must be supplied"))
if(inherits(response,"response"))zr <- response
else {
	if(missing(censor)){
		if(is.matrix(response)||is.data.frame(response))
			censor <- rep(1,nrow(response))
		else if(is.list(response))censor <- rep(1,length(response))}
	zr <- restovec(response,censor=censor)}
if(!missing(tvcov)){
	if(inherits(tvcov,"tvcov"))zt <- tvcov
	else {
		if(is.matrix(tvcov)||is.data.frame(tvcov)||
			(is.list(tvcov)&&is.null(colnames(tvcov[[1]]))))
			tvcname <- paste(deparse(substitute(tvcov)))
		else tvcname <- NULL
		zt <- tvctomat(tvcov, names=tvcname)}
	ttvc <- ncol(zt$tvcov)}
else {
	ttvc <- 0
	zt <- NULL}
zna <- rmna(response=zr, tvcov=zt, ccov=zc)
rm(zr,zt,zc)
nind <- length(zna$response$nobs)
if(any(zna$response$y<0))stop("All times must be positive")
if(ttvc>0){
	np <- np+tvc
	if(tvc!=ttvc)stop(paste(ttvc,"initial estimates of coefficients for time-varying covariates must be supplied"))}
if(rf){
	if(tvc>0&&nccov>0)stop("With a mean function, initial estimates must be supplied either in preg or in ptvc")
	if(tvc>0){
		if(length(mu(ptvc))!=length(zna$response$y))stop("The mu function must provide an estimate for each observation")
		tvc <- tvc-1
		np <- np+tvc}
	else if(length(mu(preg))==1){
		if(nccov==0)mu <- function(p) rep(p[1],length(zna$response$y))
		else stop("Number of estimates does not correspond to mu function")}
	else if(length(mu(preg))!=nind)stop("The mu function must provide an estimate for each individual")}
if(sf&&length(shape(pshape))!=length(zna$response$y))stop("The shape function must provide an estimate for each observation")
nps1 <- np-nps-!is.null(pintercept)+1
p <- c(preg,pbirth,ptvc,pinitial,pdepend,pshape,pintercept)
if(dist=="Pareto"){
	if(depend=="frailty")surv <- frailb
	else surv <- ksurvb}
else if(dist=="gamma"||dist=="Weibull"){
	if(depend=="frailty")surv <- frailg
	else surv <- ksurvg}
if(fscale==1)fscale <- surv(p)
z0 <- nlm(surv, p=p, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
p <- z0$estimate
like <- z0$minimum
if(any(is.na(z0$hessian)))a <- 0
else a <- qr(z0$hessian)$rank
if(a==np)cov <- solve(z0$hessian)
else cov <- matrix(NA,ncol=np,nrow=np)
se <- sqrt(diag(cov))
corr <- cov/(se%o%se)
dimnames(corr) <- list(1:np,1:np)
if(mdl==4)z <- list()
else {
	z <- if(depend=="frailty"){
		if(rf)b <- mu(p)
		if(sf)v <- shape(p[nps1:np])
		z <- .C("frailb",
			p=as.double(p),
			y=as.double(zna$response$y),
			x=as.double(zna$ccov$ccov),
			cens=as.integer(zna$response$censor),
			nind=as.integer(nind),
			nobs=as.integer(zna$response$nobs),
			nbs=as.integer(length(zna$response$y)),
			nccov=as.integer(nccov),
			model=as.integer(mdl),
			density=as.integer(density),
			dep=as.integer(dep),
			birth=as.integer(birth),
			tvc=as.integer(tvc),
			tvcov=as.double(zna$tvcov$tvcov),
			fit=as.integer(1),
			pred=double(length(zna$response$y)),
			rpred=double(length(zna$response$y)),
			rf=as.integer(rf),
			bb=as.double(b),
			sf=as.integer(sf),
			vv=as.double(v),
			like=double(1))}
	else if(dist=="Pareto"){
		if(rf)b <- mu(p)
		if(sf)v <- shape(p[nps1:np])
		z <- .C("ksurvb",
			p=as.double(p),
			y=as.double(zna$response$y),
			x=as.double(zna$ccov$ccov),
			cens=as.integer(zna$response$censor),
			nind=as.integer(nind),
			nobs=as.integer(zna$response$nobs),
			nbs=as.integer(length(zna$response$y)),
			nccov=as.integer(nccov),
			model=as.integer(mdl),
			density=as.integer(density),
			dep=as.integer(dep),
			birth=as.integer(birth),
			tvc=as.integer(tvc),
			tvcov=as.double(zna$tvcov$tvcov),
			fit=as.integer(1),
			pred=double(length(zna$response$y)),
			rpred=double(length(zna$response$y)),
			renewal=as.integer(renewal),
			rf=as.integer(rf),
			bb=as.double(b),
			sf=as.integer(sf),
			vv=as.double(v),
			like=double(1))}
	else {
		if(rf)b <- mu(p)
		if(sf)v <- shape(p[nps1:np])
		z <- .C("ksurvg",
			p=as.double(p),
			y=as.double(zna$response$y),
			x=as.double(zna$ccov$ccov),
			cens=as.integer(zna$response$censor),
			nind=as.integer(nind),
			nobs=as.integer(zna$response$nobs),
			nbs=as.integer(length(zna$response$y)),
			nccov=as.integer(nccov),
			model=as.integer(mdl),
			dist=as.integer(dst),
			density=as.integer(density),
			dep=as.integer(dep),
			birth=as.integer(birth),
			tvc=as.integer(tvc),
			tvcov=as.double(zna$tvcov$tvcov),
			fit=as.integer(1),
			pred=double(length(zna$response$y)),
			rpred=double(length(zna$response$y)),
			renewal=as.integer(renewal),
			rf=as.integer(rf),
			bb=as.double(b),
			sf=as.integer(sf),
			vv=as.double(v),
			like=double(1))}
	if(mdl>4){
		z$pred <- exp(z$pred)
		z$rpred <- exp(z$rpred)}
	for(i in 1:length(zna$response$y))if(zna$response$y[i]==0){
		z$pred[i] <- z$pred[i-1]
		z$rpred[i] <- z$rpred[i-1]}}
j <- 1
for(i in 1:length(zna$resp$nobs)){
	zna$response$times <- c(zna$response$times,
		cumsum(zna$response$y[j:(j+zna$response$nobs[i]-1)]))
	j <- j+zna$response$nobs[i]}
z <- list(
	call=call,
	intensity=intensity,
	dist=dist,
	mu=mu,
	npr=1+nccov+tvc+birth,
	shape=shape,
	nps=np-nps,
	density=density,
	depend=depend,
	update=update,
	birth=birth,
	renewal=renewal,
	response=zna$response,
	pred=z$pred,
	rpred=z$rpred,
	ccov=zna$ccov,
	tvcov=zna$tvcov,
	maxlike=like,
	aic=like+np,
	df=length(zna$response$y)-np,
	npt=np,
	npv=npreg,
	coefficients=p,
	se=se,
	cov=cov,
	corr=corr,
	grad=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z) <- if(mdl==4)"kalsurv" else c("kalsurv","recursive")
return(z)}

coefficients.kalsurv <- function(z) z$coefficients
deviance.kalsurv <- function(z) 2*z$maxlike
fitted.kalsurv <- function(z, recursive=TRUE)
	if(recursive) z$rpred else z$pred
residuals.kalsurv <- function(z, recursive=TRUE)
	if(recursive) z$response$y-z$rpred else z$response$y-z$pred

print.kalsurv <- function(z, digits = max(3, .Options$digits - 3)) {
	tvc <- !is.null(z$tvcov)
	expm <- z$intensity!="exponential"&&!is.function(z$shape)
	glm <- z$intensity=="gen logistic"
	nps <- if(is.function(z$shape)) z$nps else z$npt
	deppar <- (z$depend!="independence")&&(z$depend!="frailty")
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	if(z$code>2)cat("Warning: no convergence - error",z$code,"\n\n")
	cat("Number of subjects    ",length(z$resp$nobs),"\n")
	cat("Number of observations",length(z$response$y),"\n")
	cat(z$dist,"distribution ")
	if(z$renewal)cat("with renewal process\n")
	else cat("with zero origin\n")
	if(z$density)cat(z$intensity," density",sep="")
	else cat(z$intensity," intensity",sep="")
	if(z$depend=="independence")cat(" with independence\n")
	else if(z$depend=="frailty")
		cat(" with",z$depend,"dependence and",z$update,"weight\n")
	else cat(" with ",z$update," update\n",sep="")
	if(is.function(z$mu)){
		t <- deparse(z$mu)
		cat("Location function:",t[2:length(t)],sep="\n")}
	cat("\n-Log likelihood   ",z$maxlike,"\n")
	cat("Degrees of freedom",z$df,"\n")
	cat("AIC               ",z$aic,"\n")
	cat("Iterations        ",z$iterations,"\n\n")
	cat("Location parameters\n")
	if(is.function(z$mu)){
		t <- deparse(z$mu)
		cat("Function:",t[2:length(t)],sep="\n")}
	coef.table <- cbind(z$coef[1:z$npr],z$se[1:z$npr])
	if(!is.function(z$mu)){
		cname <- "(Intercept)"
		if(z$npv>0)cname <- c(cname,colnames(z$ccov$ccov))
		if(z$birth)cname <- c(cname,"birth")
		if(tvc)cname <- c(cname,colnames(z$tvcov$tvcov))}
	else {
		cname <- NULL
		for(i in 1:nrow(coef.table))cname <- c(cname,paste("p",i,sep=""))}
	dimnames(coef.table) <- list(cname, c("estimate","se"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(is.function(z$shape))cat("\nDependence parameters\n")
	else cat("\nNonlinear parameters\n")
	coef <- exp(z$coef[(nps-deppar-expm-glm):nps])
	cname <- "initial"
	if(deppar){
		coef[2] <- coef[2]/(1+coef[2])
		cname <- c(cname,"depend")}
	if(glm){
		cname <- c(cname,"asymptote","intercept")
		coef[length(coef)-1] <- 1/coef[length(coef)-1]
		coef[length(coef)] <- NA}
	else if(expm)cname <- c(cname,"shape")
	coef.table <- cbind(z$coef[(nps-deppar-expm-glm):nps],z$se[(nps-deppar-expm-glm):nps],coef)
	dimnames(coef.table) <- list(cname, c("estimate","se","parameter"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(z$depend=="frailty"){
		tmp <- trigamma(exp(-z$coef[nps-deppar-expm]))
		cat("Correlation =",tmp/(tmp+trigamma(1)),"\n")}
	if(is.function(z$shape)){
		cat("\nShape parameters\n")
		t <- deparse(z$shape)
		cat("Function:",t[2:length(t)],sep="\n")
		coef.table <- cbind(z$coef[(z$nps+1):z$npt],
			z$se[(z$nps+1):z$npt])
		cname <- NULL
		for(i in 1:nrow(coef.table))
			cname <- c(cname,paste("p",i,sep=""))
		dimnames(coef.table) <- list(cname, c("estimate","se"))
		print.default(coef.table, digits=digits, print.gap=2)}
	cat("\nCorrelation matrix\n")
	print.default(z$corr, digits=digits)}

plot.profile.kalsurv <- function(z, times=NULL, nind=1, mu=NULL, intensity=F,
	add=F, ylim=NULL, lty=NULL, ylab=NULL, xlab="Chronological time", ...){
	for(i in 1:length(z$response$y))if(z$response$y[i]==0)
		z$response$y[i] <- z$response$y[i-1]
	if(intensity){
		z$pred <- 1/z$pred
		if(is.null(ylab))ylab <- "Mean intensity"}
	else if(is.null(ylab))ylab <- "Time between events"
	plot.profile.default(z, times=times, nind=nind, mu=mu, add=add,
		ylim=ylim, lty=lty, ylab=ylab, xlab=xlab, ...)}

plot.iprofile.kalsurv <- function(z, nind=1, obs=T, add=F, intensity=F,
	plotsd=T, lty=NULL, pch=NULL, ylab=NULL, xlab="Chronological time",
	main=NULL, ylim=NULL, ...){
	for(i in 1:length(z$response$y))if(z$response$y[i]==0)
		z$response$y[i] <- z$response$y[i-1]
	if(intensity){
		z$rpred <- 1/z$rpred
		z$response$y <- 1/z$response$y
		if(is.null(ylab))ylab <- "Mean intensity"}
	else if(is.null(ylab))ylab <- "Time between events"
	plot.iprofile.default(z, nind=nind, obs=obs, add=add, plotsd=plotsd,
		lty=lty, pch=pch, ylab=ylab, xlab=xlab, main=main,
		ylim=ylim, ...)}

plot.residuals.kalsurv <- function(z, x=NULL, subset=NULL, ccov=NULL,
	nind=NULL, recursive=T, pch=20, ylab="Residual",
	xlab=NULL, main=NULL, ...){
	for(i in 1:length(z$response$y))if(z$response$y[i]==0)
		z$response$y[i] <- z$response$y[i-1]
	plot.residuals.default(z, x=x, subset=subset, ccov=ccov,
		nind=nind, recursive=recursive, pch=pch, ylab=ylab,
		xlab=xlab, main=main, ...)}
