#' Calculate All NUE Indicators (Batch Processing)
#'
#' Automatically detects available data columns and calculates all 23 NUE indicators.
#'
#' @param data A dataframe containing experimental data.
#' @param yield_f Column name for Fertilized Yield. Default: "YieldF"
#' @param yield_0 Column name for Control Yield. Default: "Yield0"
#' @param fert_n Column name for Fertilizer N input. Default: "FertN"
#' @param plant_n_f Column name for Plant N (fertilized). Default: "PlantNf"
#' @param plant_n_0 Column name for Plant N (control). Default: "PlantN0"
#' @param yield_n Column name for N removed as yield. Default: "YieldN"
#' @param soil_n Column name for Soil N. Default: "SoilN"
#' @param plant_bm Column name for Plant Biomass. Default: "PlantBM"
#' @param plant_15n Column name for Plant 15N excess. Default: "Plant15N"
#' @param fert_15n Column name for Fertilizer 15N excess. Default: "Fert15N"
#' @param gr Column name for Plant Growth Rate. Default: "GR"
#' @param n_con Column name for N available for consumption. Default: "Ncon"
#' @param n_rec Column name for N recycled. Default: "Nrec"
#' @param n_loss Column name for N lost from system. Default: "Nloss"
#' @param n_in Column name for Total N Input (System). Default: "Ni"
#' @param n_out Column name for Total N Output (System). Default: "No"
#' @param delta_soil_n Column name for Change in Soil N. Default: "delSoilN"
#' @param mrt Column name for Mean Residency Time. Default: "MRT"
#' @return A dataframe with new columns appended for every calculable metric.
#' @export
#' @examples
#' # Create a simple dataset
#' df <- data.frame(
#'   YieldF = c(3.5, 4.2, 5.0),
#'   Yield0 = c(2.0, 2.5, 3.0),
#'   FertN = c(10, 10, 10)
#' )
#'
#' # Run the batch calculator
#' result <- calculate_nue_batch(df)
#'
#' # View calculated AE column
#' print(result$AE)
calculate_nue_batch <- function(data,
                                yield_f = "YieldF",
                                yield_0 = "Yield0",
                                fert_n = "FertN",
                                plant_n_f = "PlantNf",
                                plant_n_0 = "PlantN0",
                                yield_n = "YieldN",
                                soil_n = "SoilN",
                                plant_bm = "PlantBM",
                                plant_15n = "Plant15N",
                                fert_15n = "Fert15N",
                                gr = "GR",
                                n_con = "Ncon",
                                n_rec = "Nrec",
                                n_loss = "Nloss",
                                n_in = "Ni",
                                n_out = "No",
                                delta_soil_n = "delSoilN",
                                mrt = "MRT") {
  
  # Helper to check if a list of columns exists in the data
  has_cols <- function(...) {
    cols <- c(...)
    all(cols %in% names(data))
  }
  
  # --- GROUP 1: FERTILIZER BASED INDICATORS ---
  
  # 1. AE (Agronomic Efficiency)
  if (has_cols(yield_f, yield_0, fert_n)) {
    data$AE <- AE(YieldF = data[[yield_f]], Yield0 = data[[yield_0]], FertN = data[[fert_n]])
  }
  
  # 2. PFP (Partial Factor Productivity)
  if (has_cols(yield_f, fert_n)) {
    data$PFP <- PFP(YieldF = data[[yield_f]], FertN = data[[fert_n]])
  }
  
  # 3. NRE (Recovery Efficiency)
  if (has_cols(plant_n_f, plant_n_0, fert_n)) {
    data$NRE <- NRE(PlantNf = data[[plant_n_f]], PlantN0 = data[[plant_n_0]], FertN = data[[fert_n]])
  }
  
  # 4. NUEcrop
  if (has_cols(yield_n, fert_n)) {
    data$NUEcrop <- NUEcrop(YieldN = data[[yield_n]], FertN = data[[fert_n]])
  }
  
  # 5. PNB (Partial N Balance)
  if (has_cols(plant_n_f, fert_n)) {
    data$PNB <- PNB(PlantNf = data[[plant_n_f]], FertN = data[[fert_n]])
  }
  
  # 6. NBI (N Balance Intensity)
  if (has_cols(yield_n, fert_n)) {
    data$NBI <- NBI(YieldN = data[[yield_n]], FertN = data[[fert_n]])
  }
  
  # --- GROUP 2: PLANT BASED INDICATORS ---
  
  # 7. IE (Internal Efficiency)
  if (has_cols(yield_n, plant_n_f)) {
    # Note: IE defines yield as 'YieldNF' (Yield N in fertilized conditions)
    data$IE <- IE(YieldNF = data[[yield_n]], PlantNf = data[[plant_n_f]])
  }
  
  # 8. NHI (N Harvest Index)
  if (has_cols(yield_f, plant_n_f)) {
    data$NHI <- NHI(YieldF = data[[yield_f]], PlantNf = data[[plant_n_f]])
  }
  
  # 9. NP (Nitrogen Productivity)
  if (has_cols(gr, plant_n_f)) {
    data$NP <- NP(GR = data[[gr]], PlantN = data[[plant_n_f]])
  }
  
  # 10. NUtE (N Utilization Efficiency)
  if (has_cols(yield_f, plant_n_f)) {
    data$NUtE <- NUtE(Yield = data[[yield_f]], PlantN = data[[plant_n_f]])
  }
  
  # 11. PE (Physiological Efficiency)
  if (has_cols(yield_f, yield_0, plant_n_f, plant_n_0)) {
    data$PE <- PE(YieldF = data[[yield_f]], Yield0 = data[[yield_0]],
                  PlantNf = data[[plant_n_f]], PlantN0 = data[[plant_n_0]])
  }
  
  # 12. NUpE (N Uptake Efficiency)
  if (has_cols(plant_n_f, soil_n, fert_n)) {
    data$NUpE <- NUpE(PlantN = data[[plant_n_f]], SoilN = data[[soil_n]], FertN = data[[fert_n]])
  }
  
  # --- GROUP 3: SOIL BASED INDICATORS ---
  
  # 13. NUEsoil
  if (has_cols(plant_bm, soil_n, fert_n)) {
    data$NUEsoil <- NUEsoil(PlantBM = data[[plant_bm]], SoilN = data[[soil_n]], FertN = data[[fert_n]])
  }
  
  # 14. sNBI (System N Balance Index)
  if (has_cols(n_in, n_out, delta_soil_n)) {
    data$sNBI <- sNBI(Ni = data[[n_in]], No = data[[n_out]], delSoilN = data[[delta_soil_n]])
  }
  
  # --- GROUP 4: ISOTOPE INDICATORS ---
  
  # 15. NdfF (N Derived from Fertilizer)
  if (has_cols(plant_15n, fert_15n)) {
    data$NdfF <- NdfF(Plant15N = data[[plant_15n]], Fert15N = data[[fert_15n]])
  }
  
  # 16. TNdfF (Total N Derived from Fertilizer)
  # Requires NdfF to be calculated first.
  if ("NdfF" %in% names(data)) {
    if (has_cols(plant_n_f)) {
      data$TNdfF <- TNdfF(NdfF = data$NdfF, PlantN = data[[plant_n_f]])
    } else if (has_cols(soil_n)) {
      data$TNdfF <- TNdfF(NdfF = data$NdfF, SoilN = data[[soil_n]])
    }
  }
  
  # 17. NRE15 (Isotope Recovery Efficiency)
  # Requires TNdfF to be calculated first.
  if ("TNdfF" %in% names(data) && has_cols(fert_n)) {
    data$NRE15 <- NRE15(TNdfF = data$TNdfF, FertN = data[[fert_n]])
  }
  
  # --- GROUP 5: ECOLOGY & SYSTEM INDICATORS ---
  
  # 18. NUEyield (Requires NUpE and NUtE)
  if ("NUpE" %in% names(data) && "NUtE" %in% names(data)) {
    data$NUEyield <- NUEyield(NUpE = data$NUpE, NUtE = data$NUtE)
  }
  
  # 19. ecoNUE (Requires NP)
  if ("NP" %in% names(data) && has_cols(mrt)) {
    data$ecoNUE <- ecoNUE(NP = data$NP, MRT = data[[mrt]])
  }
  
  # 20. NUEFC (Food Chain)
  if (has_cols(n_con, n_in)) {
    data$NUEFC <- NUEFC(Ncon = data[[n_con]], Ni = data[[n_in]])
  }
  
  # 21. NUEbal (Balance)
  if (has_cols(n_out, n_in)) {
    data$NUEbal <- NUEbal(No = data[[n_out]], Ni = data[[n_in]])
  }
  
  # 22. VNF (Virtual N Factor)
  if (has_cols(n_rec, n_con)) {
    data$VNF <- VNF(Nrec = data[[n_rec]], Ncon = data[[n_con]])
  }
  
  # 23. sNUE (System NUE)
  if (has_cols(yield_n, n_loss)) {
    data$sNUE <- sNUE(YieldN = data[[yield_n]], Nloss = data[[n_loss]])
  }
  
  return(data)
}