#===============================================================================
#                      Plotting Functions
#===============================================================================

#' @title Trend Tests Plots
#'
#' @description Informative plots on the given data series in regards to univariate
#' and multivariate trend testing.
#'
#' @param data Numeric matrix, a two dimensional data set where each column is a
#'  variable.
#' @param width Integer, the width to be used for the *moving window* algorithm.
#'  Should be bigger than 1 and smaller than the number of observations.
#' @param covar Numeric vector, optional covariate vector, must be the same length as `data`.
#'  Default is NULL.
#' @param graph Character vector, what will be plotted. Options are:
#'
#'  - "summary", creates line plots of each variable, Kendall tau with moving window, and the results from the tests.
#'  - "variable1", creates a line plot of the first variable.
#'  - "variable2", creates a line plot of the second variable.
#'  - "window", creates scatter plot of Kendall tau using moving window.
#'
#'  Default is "summary".
#'
#' @param color Character vector, plot color theme. Options are: "blue", "dark", "green", and "warm".
#'
#' @return The specified plot.
#'
#' @details
#' The function is able to do three plots: "window" returns the scatter plot of
#' Kendall \eqn{\tau} with the moving window technique, "variable1" and "variable2"
#' return the line plots for each variable against the covariate or "Time", finally
#' "summary" returns the previous plot combined in the same display.
#' Four color profiles are available .
#'
#'
#' @references
#' Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
#' Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
#'
#' @seealso
#' \itemize{
#'  \item `kendall.tau` : Function from the package `VGAM` used for computing the bivariate Kendall's \eqn{\tau}.
#'  \item `rollapply` : Function from the package `zoo` used to apply the moving window technique.
#' }
#'
#' @examples
#'
#' \donttest{
#' # Sample data:
#' dataDependenceStructure <- generate_data("dependenceStructure", 50)
#' dataMarginalTrend <- generate_data("marginalTrend", 50)
#' dataNoTrend <- generate_data("noTrend", 50)
#'
#' # Plot Trend summary:
#' plotTrend(dataDependenceStructure, covar = NULL, width = 10, graph = "summary", color = "blue")
#'
#' plotTrend(dataMarginalTrend, covar = NULL, width = 10, graph = "summary", color = "green")
#'
#' plotTrend(dataNoTrend, covar = NULL, width = 10, graph = "summary", color = "warm")
#'
#' # Plot a variable
#' plotTrend(dataMarginalTrend, width = 10, graph = "variable1", color = "green")
#'
#' # Plot the evolution of Kendall tau
#' plotTrend(dataDependenceStructure, width = 10, graph = "window", color = "warm")
#' }
#'
#'
#' @importFrom zoo rollapply
#' @importFrom VGAM kendall.tau
#' @importFrom graphics grid legend lines par plot.new
#' @importFrom grDevices dev.off dev.cur
#'
#' @export

plotTrend <- function(data,
                      covar = NULL,
                      width,
                      graph = c("summary", "variable1", "variable2", "window"),
                      color = c("blue", "dark", "green", "warm")) {}

plotTrend <- function(data,
                      covar = NULL,
                      width,
                      graph = c("summary", "variable1", "variable2", "window"),
                      color = c("blue", "dark", "green", "warm")) {

  if (dev.cur() > 1) {dev.off()}
  old.par <- par(no.readlonly = TRUE)
  on.exit(par(old.par))

  data_ <- as.matrix(data)

  # Extract variables
  X <- data_[, 1]
  Y <- data_[, 2]

  if (!is.numeric(X) || !is.numeric(Y)) {
    stop("Data must be numeric.")
  }

  n <- length(X)

  # Declaring covar in case of NULL
  if (is.null(covar) || length(covar) != n) {
    covar = seq(1:n)
  }

  if (!is.numeric(width)) {
    stop("Parameter 'width' must be numeric")
  }

  if (width <= 1 || width > n) {
    stop("Parameter 'width' must be bigger than 1 and smaller than number of observations")
  } else if (width == n) {
    warning("Inadequate value for 'width' parameter, must be bigger than 1 and smaller than number of observations")
  }

  color <- match.arg(color)

  switch(color,
         "blue" = {col1 <- "#0033FF"; col2 <- "#6699FF"; col3 = "#6688ee"},
         "dark" = {col1 <- "#222222"; col2 <- "#999999"; col3 = "#777777"},
         "green" = {col1 <- "#009911"; col2 <- "#11CC44"; col3 = "#22bb44"},
         "warm" = {col1 <- "#FF6600"; col2 <- "#FFBB00"; col3 = "#FFaa00"})

  graph <- match.arg(graph)

  switch(graph,
         "summary" = {
           # Defining 'kendall.tau' as a function
           kend <- function(x){kendall.tau(x[, 1], x[, 2])}

           # Getting correlation coefficients between 'x' and 'y' in different subsets,
           # moving or rolling window
           mat = cbind(X, Y)

           roll = rollapply(mat, width = width, FUN = kend,
                            by.column = FALSE, align = "right")

           t <- n - width + 1
           T2 <- rollapply(covar, width =width, FUN = function(x){mean(x)}, by.column = FALSE, align = "right")

           # Tests' p-values
           mdt <- MDT_test(data, width = width)
           mot <- MOT_test(data, covar = covar, width = width)
           var1 <- mkUnivariate(X)
           var2 <- mkUnivariate(Y)

           mdt <- pvalue_round(mdt$p.value)
           mot <- pvalue_round(mot$p.value)
           var1 <- pvalue_round(var1$p.value)
           var2 <- pvalue_round(var2$p.value)

           # Plotting

           par(cex=0.7, mai=c(0.8, 0.8, 0.2, 0.1))
           par(fig = c(0.02, 0.65, 0.02, 0.61))
           plot(x = T2, y = roll,
                main = "Kendall tau with Moving Window",
                xlab = "Time",
                ylab = "Kendall Tau",
                col = col1)
           legend("bottomright", ncol = 2L, title = "Mult. Trend Tests",
                  legend = c("MOT", "MDT",
                             mdt, mot))
           grid()
           par(fig = c(0.02, 0.65, 0.62, 0.98), cex=0.7,
               mai=c(0.8, 0.8, 0.1, 0.1), new =TRUE, pch = 20)
           plot(x = covar, y = X,
                main = paste("MK Test: ", as.character(var1)),
                xlab = "Time",
                ylab = colnames(data)[1],
                col = col3)
           lines(covar, X, col = col2)
           grid(nx = NA, ny = NULL, lty = 1)
           par(fig = c(0.67, 0.98, 0.02, 0.6), cex=0.7,
               mai=c(0.8, 0.8, 0.1, 0.1), new =TRUE, pch = 20)
           plot(Y, covar,
                main = paste("MK Test: ", as.character(var2)),
                xlab = colnames(data)[2],
                ylab = "Time",
                col = col3)
           lines(Y, covar, col = col2)
           grid(nx = NULL, ny = NA, lty = 1)},

         "variable1" = {

           # Mann-Kendall test
           var1 <- mkUnivariate(X)
           var1 <- pvalue_round(var1$p.value)

           # Plot
           plot(covar, X,
                xlab = "Time",
                ylab = colnames(data[, 1]),
                main = paste("Time and First Variable | MK Test: ", as.character(var1)),
                col = col3)
           lines(covar, X, col = col1)
           grid()},

         "variable2" = {

           # Mann-Kendall test
           var2 <- mkUnivariate(X)
           var2 <- pvalue_round(var2$p.value)

           # Plot
           plot(covar, Y,
                xlab = "Time",
                ylab = colnames(data[, 2]),
                main = paste("Time and Second Variable | MK Test: ", as.character(var2)),
                col = col3)
           lines(covar, Y, col = col1)
           grid()},

         "window" = {

           # Trend tests
           mdt <- MDT_test(data, width = width)
           mot <- MOT_test(data, covar = covar, width = width)
           mdt <- pvalue_round(mdt$p.value)
           mot <- pvalue_round(mot$p.value)

           # Defining 'kendall.tau' as a function
           kend <- function(x){kendall.tau(x[, 1], x[, 2])}

           # Getting correlation coefficients between 'x' and 'y' in different subsets,
           # moving or rolling window
           mat = cbind(X, Y)

           roll = rollapply(mat, width = width, FUN = kend,
                            by.column = FALSE, align = "right")

           t <- n - width + 1
           T2 <- rollapply(covar, width =width, FUN = function(x){mean(x)}, by.column = FALSE, align = "right")



           plot(x = T2, y = roll,
                main = "Kendall tau with Moving Window",
                xlab = "Time",
                ylab = "KT",
                col = col1)
           legend("bottomright", ncol = 2L, title = "Mult. Trend Tests",
                  legend = c("MOT", "MDT",
                             mdt, mot))
           grid()},
         )
}

#===============================================================================

#' @title Trend Tests Summary
#'
#' @description Performs multivariate and univariate trend tests on the given data series
#'  and returns a data frame with the results.
#'
#' @param data Numeric matrix, a two dimensional data set where each column is a
#'  variable.
#' @param width Numeric value, the width to be used for the *moving window* algorithm.
#'  Should be bigger than 1 and smaller than the number of observations.
#' @param covar Numeric vector, time order of the data series. If none give \eqn{covar = 1, 2, ..., n}.
#'
#' @return The results dataframe, a column for the respective test statistic, a column
#' for the p-value, and four rows each for a test MDT, MOT, and MK for each variable.
#'
#' @details
#' The function performs the Multivariate Dependence Trend and Multivariate Overall
#' Trend tests as described by Goutali and Chebana (2024), as well as the univariate
#' Mann-Kendall (MK) test to each variable and returns a data frame with the results.
#'
#' This functions performs the test with the default values for `alpha = 0.05` and `Nbs = 1000`,
#' for a more precise testing you can use the functions described in **see also**.
#'
#' @references
#' Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
#' Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
#'
#' @seealso
#' \itemize{
#'  \item [MDT_test]: Function used to perform the MDT test.
#'  \item [MOT_test]: Function used to compute the MOT test.
#'  \item [mkUnivariate]: Function used to perform the univariate MK test.
#' }
#'
#' @examples
#'
#' \donttest{
#' # Summary for data with trend in the Dependence Structure:
#' DependenceStructure <- generate_data("dependenceStructure", 50)
#' summaryTrend(DependenceStructure, covar = NULL, width = 10)
#'
#' # Summary for data with trend in the Marginals:
#' MarginalTrend <- generate_data("marginalTrend", 50)
#' summaryTrend(MarginalTrend, covar = NULL, width = 10)
#'
#' # Summary for data without trend:
#' NoTrend <- generate_data("noTrend", 50)
#' summaryTrend(NoTrend, covar = NULL, width = 10)
#' }
#'
#' @export


summaryTrend <- function(data, width, covar = NULL) {}

summaryTrend <- function(data, width, covar = NULL) {

  data <- as.matrix(data)

  if (nrow(data) < 2) {
    stop("'data' must have at least two columns")
  }

  # Trend test dataframe
  mdt <- MDT_test(data, width = width)
  mot <- MOT_test(data, covar = covar, width = width)
  var1 <- mkUnivariate(data[,1])
  var2 <- mkUnivariate(data[,2])

  statistic <- c(mdt$statistic, mot$statistic, var1$statistic, var2$statistic)
  p.value <- c(pvalue_range(mdt$p.value), pvalue_range(mot$p.value), pvalue_range(var1$p.value), pvalue_range(var2$p.value))
  test <- c("Multivariate", "Multivariate", "Univariate", "Univariate")

  results <- data.frame(statistic, p.value, test,
                        row.names = c('MDT', 'MOT', 'MK variable 1', 'MK variable 2')

  )
  return(results)

}


# HELPER: p-value rounder

pvalue_round <- function(pvalue) {
  if (pvalue <= 1e-15) {
    pvalue <- "<= 1e-15"
  } else if (pvalue <= 1e-10) {
    pvalue <- "<= 1e-10"
  } else if (pvalue <= 1e-5) {
    pvalue <- "<= 1e-5"
  } else {
    pvalue <- sprintf("%.5f", pvalue)
  }
  return(pvalue)
}

pvalue_range <- function(pvalue) {
  if (pvalue <= 1e-15) {
    pvalue <- "<= 1e-15"
  } else if (pvalue <= 1e-10) {
    pvalue <- "<= 1e-10"
  } else if (pvalue <= 1e-5) {
    pvalue <- "<= 1e-5"
  }
  return(pvalue)
}
