#' Plot Trait Network Graph
#'
#' This function visualizes the trait network graph generated by the `PTN` function.
#'
#' @param graph An igraph object representing the trait network.
#' @param style A numeric value that determines the plotting style (default is 1).
#' @param vertex.size Numeric value for the size of vertices in the plot (default is 20).
#' @param vertex.label.cex Numeric value for the scaling factor of vertex labels (default is 0.6).
#'
#' @return
#' An object of class `igraph`. This function generates a visualization of the trait network graph.
#' When style = 1, it displays a community structure plot.
#' When style = 2, it displays a circular layout plot where vertex colors represent community membership,
#' edge thickness represents correlation strength, and edge color represents the sign of the correlation (black for positive, red for negative).
#'
#' @details
#' The function uses the `cluster_edge_betweenness` algorithm to identify communities
#' in the graph and assigns community membership to vertices. It offers two
#' plotting styles:
#' - Style 1: Plots the community structure.
#' - Style 2: Plots the graph in a circular layout with vertex colors representing communities.
#' The vertex size and label size can be customized using vertex.size and vertex.label.cex parameters respectively.
#'
#' @references
#' 1. He, N., Li, Y., Liu, C., et al. (2020). Plant trait networks: improved resolution of the dimensionality of adaptation.
#' Trends in Ecology & Evolution, 35(10), 908-918.
#' 2. Li, Y., Liu, C., Sack, L., Xu, L., Li, M., Zhang, J., & He, N. (2022). Leaf trait network architecture shifts with
#' species‐richness and climate across forests at continental scale. Ecology Letters, 25(6), 1442-1457.
#'
#' @examples
#' data(PFF)
#' PFF_traits <- PFF[, c("Height", "Leaf_area","LDMC","SLA","SRL","SeedMass","FltDate",
#'                       "FltDur","Leaf_Cmass","Leaf_Nmass","Leaf_CN","Leaf_Pmass",
#'                       "Leaf_NP","Leaf_CP","Root_Cmass","Root_Nmass","Root_CN")]
#' PFF_traits <- na.omit(PFF_traits)
#' head(PFF_traits)
#' ptn_result <- PTN(traits_matrix = PFF_traits, rThres = 0.2, pThres = 0.05)
#' PTN_plot(ptn_result, style = 1, vertex.size = 20, vertex.label.cex = 0.6)
#' PTN_plot(ptn_result, style = 2, vertex.size = 20, vertex.label.cex = 0.6)
#'
#' @importFrom igraph cluster_edge_betweenness membership layout_in_circle V
#' @export
PTN_plot <- function(graph, style = 1,vertex.size = 20,vertex.label.cex = 0.6) {
  comm <- suppressWarnings(igraph::cluster_edge_betweenness(graph, weights = NA))
  igraph::V(graph)$community <- igraph::membership(comm)
  layout <- igraph::layout_in_circle(graph, order = order(igraph::V(graph)$community))
  if (style == 1) {
    plot(comm, graph,vertex.size = vertex.size,vertex.label.cex = vertex.label.cex,vertex.label.color = "black")
  } else if (style == 2) {
    # Get the correlation coefficients of the edges
    edge_correlation <- igraph::E(graph)$correlation
    # Set the line width (according to the absolute value of the correlation coefficient)
    edge_width <- abs(edge_correlation) * 5  # The multiplier can be adjusted to change the line thickness range
    # Set the color of the edge (blue for negative correlation, green for positive correlation)
    edge_color <- ifelse(edge_correlation > 0, "green", "blue")
    # Drawing
    plot(graph, layout = layout,
         vertex.color = igraph::V(graph)$community,
         vertex.label.cex = vertex.label.cex,
         vertex.label.dist = 0,
         vertex.size = vertex.size,
         vertex.label.color = "black",
         vertex.label.font = 2,
         edge.width = edge_width,    # Add line width
         edge.color = edge_color)    # Add line colors
  } else {
    stop("Invalid style. Please choose 1 or 2.")
  }
}
