#' @title Fits Structural Equation Models to Simulated Data Using lavaan.
#' @description
#' This function applies a pre-specified SEM model to previously generated data sets (e.g., from sim.skewed() or sim.normal()) by utilizing the lavaan package.
#' After model estimation, fit indices and parameter estimates with their standard errors are exported to a Comma-Separated Values (CSV) file named All_Results.csv.
#' Each row in this file corresponds to the results of a single simulation.
#' Most columns are self-explanatory; however, the second column (Notes) requires further clarification. This column indicates the convergence status of the model:
#'  CONVERGE – The model converged without any issues.
#'  NONCONVERGE – The model failed to converge; in this case, all values in the row are recorded as NA.
#'  WARNING – The model converged but produced warnings (e.g., negative variance estimates). Depending on the warning type, some values may be recorded as NA.
#' To run the simulation, previously generated data sets (either via the package functions or other software) must be stored in the same folder as the dataset list file (Data_List.dat) within the working directory.
#'
#' @author Fatih Orcan
#' @importFrom lavaan cfa parameterEstimates fitmeasures standardizedSolution lavTech lavInspect
#' @importFrom utils read.table write.csv
#' @param model A Lavaan model
#' @param PEmethod The parameter estimation method. The default is ML.
#' @param dataList List of the names of data sets generated earlier either with the package functions or any other software.
#' @param f.loc File location. It indicates where the simulated data sets and "dataList" are located.
#' @param missing A specification for handling missing data. As in the lavaan package (See lavOptions)
#' @param Ordered Logical, If TRUE, variables are treated as ordered categorical; otherwise, as continuous.
#' @export
#' @examples
#'
#' # Step 1: Generate data
#' fc<-fcors.value(nf=3, cors=c(1,.5,.6,.5,1,.4,.6,.4,1))
#' fl<-loading.value(nf=3, fl.loads=c(.5,.5,.5,0,0,0,0,0,0,0,0,.6,.6,.6,0,0,0,0,0,0,0,0,.4,.4))
#' sim.normal(nd=10, ss=100, fcors=fc, loading<-fl,  f.loc=tempdir())
#'
#' # Step 2: Specify the model
#' lavaanM<-'
#' #CFA Model
#' f1	=~ NA*x1 + x2 + x3
#' f2	=~ NA*x4 + x5 + x6
#' f3 =~ NA*x7 + x8
#' #Factor Correlations
#' f1	~~ f2
#' f1	~~ f3
#' f2	~~ f3
#' #Factor variance
#' f1	~~ 1*f1
#' f2	~~ 1*f2
#' f3	~~ 1*f3
#' '
#' dl<-"Data_List.dat"  # must be available in the working directory
#'
#' # Step 3: Fit the model across simulated data
#'
#' fit.simulation(model=lavaanM, PEmethod="MLR", Ordered=FALSE, dataList=dl, f.loc=tempdir())
#'

fit.simulation<-function(model, PEmethod="ML", Ordered=FALSE, dataList="Data_List.dat", f.loc,
                         missing=NULL){

  data.names<-read.table(paste(f.loc, "/", dataList,sep=""), header = FALSE)

  fit.names<-c("rep#","Notes", "chisq", "df", "pvalue","chisq.scaled", "df.scaled", "pvalue.scaled", "chisq.scaling.factor",
               "baseline.chisq", "baseline.df", "baseline.pvalue", "cfi","cfi.robust","cfi.scaled", "tli","tli.robust","tli.scaled", "srmr",
               "srmr_bentler","srmr_mplus", "rmsea","rmsea.robust","rmsea.scaled", "rmsea.ci.lower", "rmsea.ci.upper", "rmsea.pvalue", "logl","aic", "bic")
  par.names1<-c("est","se","pvalue")
  par.names2<-c("std.est","std.se","pvalue")

  veri<-read.table(paste(f.loc,"/", data.names[1,],sep=""))

  colnames(veri)<-c("ID", paste("x",seq(1:(dim(veri)[2]-1)),sep=""))
  veri<-veri[,-1]

  sonuc<-cfa(model,veri, estimator= PEmethod, ordered = Ordered )

  #P.Est<-parameterEstimates(sonuc)
  #Sp.Est<-standardizedSolution(sonuc)
  tum.sonuc<-matrix(NA,dim(data.names)[1],(length(fit.names)+(dim(parameterEstimates(sonuc))[1])*6))

  for (i in 1:dim(data.names)[1]){  #Simulation replication start

    veri<-read.table(paste(f.loc,"/",data.names[i,], sep = ""))
    colnames(veri)<-c("ID", paste("x",seq(1:(dim(veri)[2]-1)),sep=""))
    veri<-veri[,-1]


    sonuc<-cfa(model,veri, estimator =PEmethod, ordered = Ordered)
    tum.sonuc[i,1]<-i

    P.Est<-parameterEstimates(sonuc)
    Sp.Est<-standardizedSolution(sonuc)

    if(lavTech(sonuc, "converged")==TRUE){
      tum.sonuc[i,2:length(fit.names)]<-round(fitmeasures(sonuc)[fit.names[-1]],3)

      for(k in 1:(dim(P.Est)[1])){
        tum.sonuc[i,length(fit.names)+6*k-5]<-round(P.Est[k,"est"],3)
        tum.sonuc[i,length(fit.names)+6*k-4]<-round(P.Est[k,"se"],3)
        tum.sonuc[i,length(fit.names)+6*k-3]<-round(P.Est[k,"pvalue"],3)
        tum.sonuc[i,length(fit.names)+6*k-2]<-round(Sp.Est[k,"est.std"],3)
        tum.sonuc[i,length(fit.names)+6*k-1]<-round(Sp.Est[k,"se"],3)
        tum.sonuc[i,length(fit.names)+6*k]<-round(Sp.Est[k,"pvalue"],3)
      }

      print(paste(round(100*i/dim(data.names)[1],2),"% has completed...", sep=""))
      if(lavInspect(sonuc, "post.check")==TRUE){tum.sonuc[i,2]<-c("CONVERGED")}
      if(lavInspect(sonuc, "post.check")==FALSE){tum.sonuc[i,2]<-c("WARNING")}

    }

    if(lavTech(sonuc, "converged")==FALSE){
      for(k in 1:(dim(P.Est)[1])){
        tum.sonuc[i,]<-NA
        tum.sonuc[i,1]<-i
      }

      print(paste(round(100*i/dim(data.names)[1],2),"% has completed...", sep=""))
      tum.sonuc[i,2]<-c("NOT_CONVERGED")
    }
  } # replication end

  print("All Done !!!")
  colnames(tum.sonuc)<-c(paste("x",seq(1:(length(fit.names)+(dim(P.Est)[1])*6)),sep = ""))

  colnames(tum.sonuc)[1:length(fit.names)]<-c(fit.names)
  for(k in 1:(dim(P.Est)[1])){
    eft<-paste(P.Est[k,c("lhs","op","rhs")],sep = "", collapse="")
    colnames(tum.sonuc)[length(fit.names)+6*k-5]<-eft
    colnames(tum.sonuc)[length(fit.names)+6*k-4]<-par.names1[2]
    colnames(tum.sonuc)[length(fit.names)+6*k-3]<-par.names1[3]
    colnames(tum.sonuc)[length(fit.names)+6*k-2]<-par.names2[1]
    colnames(tum.sonuc)[length(fit.names)+6*k-1]<-par.names2[2]
    colnames(tum.sonuc)[length(fit.names)+6*k]<-par.names2[3]
  }
  write.csv(tum.sonuc, file= paste(f.loc,"/All_Results.csv", sep = ""), row.names = FALSE)
}
