% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parse_config.R
\name{parse_config}
\alias{parse_config}
\title{Parse MetaRVM Configuration File}
\usage{
parse_config(config_file, return_object = FALSE)
}
\arguments{
\item{config_file}{Character string. Path to a YAML configuration file containing
model parameters and settings.}

\item{return_object}{Logical. If \code{TRUE}, returns a \code{MetaRVMConfig}
object for method chaining and enhanced functionality. If \code{FALSE}
(default), returns a named list for backward compatibility.}
}
\value{
If \code{return_object = FALSE} (default), returns a named list containing:
\describe{
\item{N_pop}{Number of population groups}
\item{pop_map}{Data.table with population mapping and demographics}
\item{S_ini, E_ini, I_asymp_ini, I_presymp_ini, I_symp_ini, H_ini, D_ini, P_ini, V_ini, R_ini}{Initial compartment populations}
\item{vac_time_id, vac_counts, vac_mat}{Vaccination schedule data}
\item{m_wd_d, m_wd_n, m_we_d, m_we_n}{Contact mixing matrices}
\item{ts, tv, ve, dv, de, dp, da, ds, dh, dr, pea, psr, phr}{Disease parameter matrices (nsim × N_pop)}
\item{start_date}{Simulation start date as Date object}
\item{sim_length}{Simulation length in days}
\item{nsim}{Number of simulation instances}
\item{random_seed}{Random seed used (if any)}
\item{delta_t}{Time step size (fixed at 0.5)}
\item{chk_file_names, chk_time_steps, do_chk}{Checkpointing configuration}
}

If \code{return_object = TRUE}, returns a \code{MetaRVMConfig} object with
methods for parameter access and validation.
}
\description{
Reads and parses a YAML configuration file for MetaRVM simulations, extracting
all necessary parameters for epidemic modeling including population data,
disease parameters, mixing matrices, vaccination schedules, and simulation settings.
}
\details{
The function processes a YAML configuration file with the following
main sections:

\strong{Simulation Configuration:}
\itemize{
\item \code{random_seed}: Optional random seed for reproducibility in case of stochastic simulations or stochastic parameters
\item \code{nsim}: Number of simulation instances (default: 1)
\item \code{start_date}: Simulation start date in MM/DD/YYYY format
\item \code{length}: Simulation length in days
\item \code{checkpoint_dir}: Optional checkpoint directory for saving intermediate results
\item \code{checkpoint_dates}: Optional list of dates to save checkpoints.
\item \code{restore_from}: Optional path to restore simulation from checkpoint
}

\strong{Population Data:}
\itemize{
\item \code{mapping}: CSV file path containing population mapping with demographic categories age, race, zone.
The file must contains columns \code{population_id}, \code{age}, \code{race}, \code{zone},
where the \code{population_id} is defined using natural numbers.
\item \code{initialization}: CSV file with initial population states. The file must contains columns \code{population_id}, \code{N}, \code{S0}, \code{I0}, \code{V0}, \code{R0}.
\item \code{vaccination}: CSV file with vaccination schedule over time. The first column must be dates
in MM/DD/YYYY format. The rest of the columns must corresponds to respective
subpopulations in the numeric order of population_id.
}

\strong{Mixing Matrices:}
Contact matrices for different time periods. Each CSV file must have a matrix of order (N_pop x N_pop), where,
N_pop is the number of subpopulations. It is assumed that the i-th row and j-th column correspond to i-th and j-th subpopulations.
\itemize{
\item \code{weekday_day}, \code{weekday_night}: Weekday contact patterns
\item \code{weekend_day}, \code{weekend_night}: Weekend contact patterns
}

\strong{Disease Parameters:}
Epidemiological parameters (can be scalars or distributions):
\itemize{
\item \code{ts}: Transmission rate for symptomatic individuals
\item \code{tv}: Transmission rate for vaccinated individuals
\item \code{ve}: Vaccine effectiveness
\item \code{de, dp, da, ds, dh, dr}: Duration parameters for different disease states
\item \code{pea, psr, phr}: Probability parameters for disease transitions
}

\strong{Sub-population Parameters:}
\code{sub_disease_params} allows specification of different parameter values
for specific demographic categories (e.g., age groups, races).

The function supports stochastic parameters through distribution specifications
with \code{dist}, \code{mu}, \code{sd}, \code{shape}, \code{rate}, etc.
}
\section{Parameter Distributions}{

Disease parameters can be specified as distributions for stochastic modeling:
\itemize{
\item \strong{lognormal}: \code{dist: "lognormal", mu: value, sd: value}
\item \strong{gamma}: \code{dist: "gamma", shape: value, rate: value}
\item \strong{uniform}: \code{dist: "uniform", min: value, max: value}
\item \strong{beta}: \code{dist: "beta", shape1: value, shape2: value}
\item \strong{gaussian}: \code{dist: "gaussian", mean: value, sd: value}
}
}

\section{File Requirements}{

\strong{Population mapping file} must contain columns:
\itemize{
\item \code{population_id}: Unique identifier for each population group, natural numbers
\item \code{age}: Age category (e.g., "0-4", "5-11", "12-17", "18-49", "50-64", "65+")
\item \code{race}: Race/ethnicity category
\item \code{zone}: Geographic zone identifier
}

\strong{Population initialization file} must contain:
\code{N} (total population), \code{S0}, \code{I0}, \code{V0}, \code{R0} (initial compartment counts)

\strong{Vaccination file} must contain:
\code{date} (MM/DD/YYYY format) and vaccination counts for each population group
}

\examples{
\donttest{
options(odin.verbose = FALSE)
example_config <- system.file("extdata", "example_config.yaml", package = "MetaRVM")
# Parse configuration file and return list (backward compatible)
config <- parse_config(example_config)

# Parse and return MetaRVMConfig object for method chaining
config_obj <- parse_config(example_config, return_object = TRUE)

# Access parameters from config object
config_obj$get("N_pop")
config_obj$list_parameters()

# Use with MetaRVM simulation
results <- metaRVM(config_obj)
}

}
\seealso{
\code{\link{metaRVM}} for running simulations with parsed configuration
\code{\link{MetaRVMConfig}} for the configuration object class
\code{\link{process_vac_data}} for vaccination data processing
}
\author{
Arindam Fadikar
}
