% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mosallocStepwiseFirst.R
\name{mosallocStepwiseFirst}
\alias{mosallocStepwiseFirst}
\title{Multiobjective sample allocation for constraint multivariate and
multidomain optimal allocation in survey sampling (a stepwise optimality
procedure is processed first to force Pareto optimality of the solution)}
\usage{
mosallocStepwiseFirst(
  D,
  d,
  A = NULL,
  a = NULL,
  C = NULL,
  c = NULL,
  l = 2,
  u = NULL,
  opts = list(sense = "max_precision", init_w = 1, mc_cores = 1L, max_iters = 100L)
)
}
\arguments{
\item{D}{(type: matrix)
The objective matrix. A matrix of either precision or cost units.}

\item{d}{(type: vector)
The objective vector. A vector of either fixed precision components
(e.g. finite population corrections) or fixed costs.}

\item{A}{(type: matrix)
A matrix of precision units for precision constraints.}

\item{a}{(type: vector)
The right-hand side vector ofthe precision constraints.}

\item{C}{(type: matrix)
A matrix of cost coefficients for cost constraints}

\item{c}{(type: vector)
The right-hand side vector of the cost constraints.}

\item{l}{(type: vector)
A vector of lower box constraints.}

\item{u}{(type: vector)
A vector of upper box constraints.}

\item{opts}{(type: list)
The options used by the algorithms:
\cr \code{$sense} (type: character) Sense of optimization
(default = \code{"max_precision"}, alternative \code{"min_cost"}).
\cr \code{$init_w} (type numeric or matrix) Preference weightings
(default = 1; The weight for first objective component must be 1).
\cr \code{$mc_cores} (type: integer) The number of cores for parallelizing
multiple input weightings stacked rowwise (default = 1L).
\cr \code{max_iters} (type: integer) The maximum number of iterations
(default = 100L).}
}
\value{
The function \code{mosallocStepwiseFirst()} returns a list
containing the following components:

\code{$w} The initial preference weighting \code{opts$init_w}.

\code{$n} The vector of optimal sample sizes.

\code{$J} The optimal objective vector.

\code{$Objective} The objective value with respect to decision
funtional f. \code{NULL} if \code{opts$f = NULL}.

\code{$Utopian} Always \code{NULL} (consitency to \code{mosalloc()}
output). \code{NULL} if \code{opts$f = NULL}.

\code{$Normal} The vector normal to the Pareto frontier at
\code{$J}.

\code{$dfJ} Always \code{NULL} (consitency to \code{mosalloc()}
output).

\code{$Sensitivity} The dual variables of the objectives and
constraints.

\code{$Qbounds} The Quality bounds of the Lorentz cones.

\code{$Dbounds} The weighted objective constraints ($w * $J).

\code{$Scalepar} An internal scaling parameter.

\code{$Ecosolver} A list of ECOSolveR returns including:
\cr \code{...$Ecoinfostring} The info string of
\code{ECOSolveR::ECOS_csolve()}.
\cr \code{...$Ecoredcodes} The redcodes of \code{ECOSolveR::ECOS_csolve()}.
\cr \code{...$Ecosummary} Problem summary of \code{ECOSolveR::ECOS_csolve()}.

\code{$Timing} Run time info.

\code{$Iteration} Always \code{NULL} (consitency to
\code{mosalloc()} output).
}
\description{
Computes solutions to standard sample allocation problems
under various precision and cost restrictions. The input data is
transformed and parsed to the Embedded COnic Solver (ECOS) from the
'ECOSolveR' package. Multiple survey purposes are optimized
simultaneously through a stepwise weighted Chebyshev minimization which
forces Pareto optimality of solutions (cf. \code{mosalloc()}).
}
\examples{
# Artificial population of 50 568 business establishments and 5 business
# sectors (data from Valliant, R., Dever, J. A., & Kreuter, F. (2013).
# Practical tools for designing and weighting survey samples. Springer.
# https://doi.org/10.1007/978-1-4614-6449-5, Example 5.2 pages 133-9)

# See also https://umd.app.box.com/s/9yvvibu4nz4q6rlw98ac/file/297813512360
# file: Code 5.3 constrOptim.example.R

Nh <- c(6221, 11738, 4333, 22809, 5467) # stratum sizes
ch <- c(120, 80, 80, 90, 150) # stratum-specific cost of surveying

# Revenues
mh.rev <- c(85, 11, 23, 17, 126) # mean revenue
Sh.rev <- c(170.0, 8.8, 23.0, 25.5, 315.0) # standard deviation revenue

# Employees
mh.emp <- c(511, 21, 70, 32, 157) # mean number of employees
Sh.emp <- c(255.50, 5.25, 35.00, 32.00, 471.00) # std. dev. employees

# Proportion of estabs claiming research credit
ph.rsch <- c(0.8, 0.2, 0.5, 0.3, 0.9)

# Proportion of estabs with offshore affiliates
ph.offsh <- c(0.06, 0.03, 0.03, 0.21, 0.77)

budget <- 300000 # overall available budget
n.min  <- 100 # minimum stratum-specific sample size

#----------------------------------------------------------------------------
# Problem: Minimization of the maximum relative variation of estimates for
# the total revenue, the number of employee, the number of businesses claimed
# research credit and the number of businesses with offshore affiliates
# subject to cost restrictions

l <- rep(n.min, 5) # minimum sample size ber stratum
u <- Nh            # maximum sample size per stratum
C <- rbind(ch, ch * c(-1, -1, -1, 0, 0))
c <- c(budget, - 0.5 * budget)
A <- NULL # no precision constraint
a <- NULL # no precision constraint

# Variance components for multidimensional objective
D <- rbind(Sh.rev**2 * Nh**2/sum(Nh * mh.rev)**2,
           Sh.emp**2 * Nh**2/sum(Nh * mh.emp)**2,
           ph.rsch * (1 - ph.rsch) * Nh**3/(Nh - 1)/sum(Nh * ph.rsch)**2,
           ph.offsh * (1 - ph.offsh) * Nh**3/(Nh - 1)/sum(Nh * ph.offsh)**2)

d <- as.vector(D \%*\% (1 / Nh)) # finite population correction

opts = list(sense = "max_precision",
            init_w = 1,
            mc_cores = 1L,
            max_iters = 100L)

res1 <- mosallocStepwiseFirst(D = D, d = d, C = C, c = c, l = l, u = u,
                              opts = opts)
w <- res1$J[1] / res1$J
w # [1] 1.000000 3.879692 2.653655 1.000000

opts = list(sense = "max_precision",
            init_w = w,
            mc_cores = 1L,
            max_iters = 100L)

res2 <- mosallocStepwiseFirst(D = D, d = d, C = C, c = c, l = l, u = u,
                              opts = opts)
res2$w # [1] 1.000000 3.879692 2.653655 1.000000

# Compare to function mosalloc (without stepwise procedure)
opts = list(sense = "max_precision",
            f = NULL, df = NULL, Hf = NULL,
            init_w = w,
            mc_cores = 1L, pm_tol = 1e-05,
            max_iters = 100L, print_pm = FALSE)
res3 <- mosalloc(D = D, d = d, C = C, c = c, l = l, u = u, opts = opts)

# Compare objectives
rbind(res1$J, res2$J, res3$J)
#            [,1]         [,2]         [,3]       [,4]
#[1,] 0.00170589 0.0004396972 0.0006428453 0.00170589
#[2,] 0.00170589 0.0004396971 0.0006428420 0.00170589
#[3,] 0.00170589 0.0004396971 0.0006428440 0.00170589

# Compare optimal sample sizes
rbind(res1$n, res2$n, res3$n)
#          [,1]     [,2]     [,3]     [,4]     [,5]
# [1,] 958.0510 290.7446 147.1789 602.8856 638.2686
# [2,] 958.0455 290.7447 147.1871 602.8847 638.2692
# [3,] 958.0488 290.7446 147.1822 602.8853 638.2688

}
\references{
See:

Willems, F. (2025). A Framework for Multiobjective and Uncertain Resource
Allocation Problems in Survey Sampling based on Conic Optimization
(Doctoral dissertation). Trier University.
\doi{10.25353/ubtr-9200-484c-5c89}.
}
