% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/constructArestrSTRS.R
\name{constructArestrSTRS}
\alias{constructArestrSTRS}
\title{Constructor for precision constraints}
\usage{
constructArestrSTRS(X_var, X_tot, N, list, fpc = TRUE)
}
\arguments{
\item{X_var}{(type: \code{matrix})
A matrix containing stratum- (rows) and variable- (columns) specific
precision units.}

\item{X_tot}{(type: \code{matrix})
A matrix containing stratum- (rows) and variable- (columns) specific
totals.}

\item{N}{(type: \code{vector})
A vector of stratum sizes.}

\item{list}{(type: \code{list})
A list of lists taking subpopulation- (domain/area) specific arguments.
Elements are lists containing the following components which inturn
correspond to one specific precision restriction:
\cr \code{..$stratum_id} (type: \code{numeric})
A vector containing the indices of the strata considered for the current
restriction. The indices must coincide with the corresponding row numbers of
\code{X_var} and \code{X_tot}.
\cr \code{..$variate} (type: \code{character} or \code{numeric})
The column name or column index of \code{X_var} to be addressed.
\cr \code{..$measure} (type: \code{character}) "RSE" (relative standard
error) or "VAR" (variance).
\cr \code{..$bound} (type: \code{numeric})
An upper bound to "RSE" (or "VAR").
\cr \code{..$name} (type: \code{character})
The name of the subpopulation (domain/area).}

\item{fpc}{(type: \code{logical})
A \code{TRUE} or \code{FALSE} statement indicating whether the finite
population correction should be considered.}
}
\value{
The function \code{constructArestrSTRS()} returns a list containing

\code{A} (type: \code{matrix}) a precision matrix for the
quality restrictions and

\code{a} (type: \code{vector}) a precision vector for the
corresponding right-hand side
\cr useable as input to the multiobjective allocation function
\code{mosalloc()}.
}
\description{
A helper function for generating precision matrix \code{A}
and corresponding right-hand side \code{a} under stratified random
sampling (STRS) as input to the multiobjective allocation function
\code{mosalloc()}.
}
\examples{
# Artificial population of 50 568 business establishments and 5 business
# sectors (data from Valliant, R., Dever, J. A., & Kreuter, F. (2013).
# Practical tools for designing and weighting survey samples. Springer.
# https://doi.org/10.1007/978-1-4614-6449-5, Example 5.2 pages 133-9)

# See also https://umd.app.box.com/s/9yvvibu4nz4q6rlw98ac/file/297813512360
# file: Code 5.3 constrOptim.example.R

Nh <- c(6221, 11738, 4333, 22809, 5467) # stratum sizes

# Revenues
mh.rev <- c(85, 11, 23, 17, 126) # mean revenue
Sh.rev <- c(170.0, 8.8, 23.0, 25.5, 315.0) # standard deviation revenue

# Employees
mh.emp <- c(511, 21, 70, 32, 157) # mean number of employees
Sh.emp <- c(255.50, 5.25, 35.00, 32.00, 471.00) # std. dev. employees

# Proportion of establishments claiming research credit
ph.rsch <- c(0.8, 0.2, 0.5, 0.3, 0.9)

# Proportion of establishments with offshore affiliates
ph.offsh <- c(0.06, 0.03, 0.03, 0.21, 0.77)

# Matrix containing stratum-specific variance components
X_var <- cbind(Sh.rev**2,
               Sh.emp**2,
               ph.rsch * (1 - ph.rsch) * Nh/(Nh - 1),
               ph.offsh * (1 - ph.offsh) * Nh/(Nh - 1))
colnames(X_var) <- c("rev", "emp", "rsch", "offsh")

# Matrix containing stratum-specific totals
X_tot <- cbind(mh.rev, mh.emp, ph.rsch, ph.offsh) * Nh
colnames(X_tot) <- c("rev", "emp", "rsch", "offsh")

# Examples
#----------------------------------------------------------------------------
# Example 1: Assume we require at maximum 5 \% relative standard error (RSE)
# for estimates of the proportion of businesses with offshore affiliates.
#
# The input \code{A} and \code{a} to \code{mosalloc} can be calculated as
# follows:
A <- matrix(ph.offsh * (1 - ph.offsh) * Nh**3/(Nh - 1)/sum(Nh * ph.offsh)**2,
nrow = 1)
a <- sum(ph.offsh * (1 - ph.offsh) * Nh**2/(Nh - 1)
)/sum(Nh * ph.offsh)**2 + 0.05**2

# Using \code{constructArestrSTRS()} this can also be done via
list <- list(list(stratum_id = 1:5, variate = "offsh", measure = "RSE",
                  bound = 0.05, name = "pop"))
Ac <- constructArestrSTRS(X_var, X_tot, Nh, list, fpc = TRUE)

# or equivalently by
list <- list(list(stratum_id = 1:5, variate = 4, measure = "RSE",
                  bound = 0.05, name = "pop"))
Ac <- constructArestrSTRS(X_var, X_tot, Nh, list, fpc = TRUE)

# Evaluation of the output
Ac$A - A
Ac$a - a

# Example 2: Assume we require at maximum 5 \% relative standard error for
# estimates of the proportion of businesses with offshore affiliates and
# for estimates of the proportion of businesses claiming research credit
# separately for strata 1:2 and 3:5 each.

list <- list(list(stratum_id = 1:2, variate = "offsh", measure = "RSE",
                  bound = 0.05, name = "D1"),
             list(stratum_id = 3:5, variate = "offsh", measure = "RSE",
                  bound = 0.05, name = "D2"),
             list(stratum_id = 1:2, variate = "rsch", measure = "RSE",
                  bound = 0.05, name = "D1"),
             list(stratum_id = 3:5, variate = "rsch", measure = "RSE",
                  bound = 0.05, name = "D2"))
Ac <- constructArestrSTRS(X_var, X_tot, Nh, list, fpc = TRUE)

}
