% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeScales.R
\name{makeScales}
\alias{makeScales}
\title{Synthesise rating-scale data with given first and second moments and a
predefined correlation matrix}
\usage{
makeScales(n, means, sds, lowerbound = 1, upperbound = 5, items = 1, cormatrix)
}
\arguments{
\item{n}{(positive, int) sample-size - number of observations}

\item{means}{(real) target means: a vector of length k
of mean values for each scale item}

\item{sds}{(positive, real) target standard deviations: a vector of length k
of standard deviation values for each scale item}

\item{lowerbound}{(positive, int) a vector of length k
(same as rows & columns of correlation matrix) of values for lower bound
of each scale item (e.g. '1' for a 1-5 rating scale). Default = 1.}

\item{upperbound}{(positive, int) a vector of length k
(same as rows & columns of correlation matrix) of values for upper bound
of each scale item (e.g. '5' for a 1-5 rating scale). Default = 5.}

\item{items}{(positive, int) a vector of length k of number of items in each scale. Default = 1.}

\item{cormatrix}{(real, matrix) the target correlation matrix:
a square symmetric positive-semi-definite matrix of values ranging
between -1 and +1, and '1' in the diagonal.}
}
\value{
a dataframe of rating-scale values
}
\description{
\code{makeScales()} generates a dataframe of random discrete
values so the data replicate a rating scale,
and are correlated close to a predefined correlation matrix.

\code{makeScales()} is wrapper function for:
\itemize{
\item \code{\link[=lfast]{lfast()}}, generates a dataframe that best fits the desired
moments, and
\item \code{\link[=lcor]{lcor()}}, which rearranges values in each column of the dataframe
so they closely match the desired correlation matrix.
}
}
\examples{

## Example 1: four correlated items (questions)

### define parameters

n <- 16
dfMeans <- c(2.5, 3.0, 3.0, 3.5)
dfSds <- c(1.0, 1.0, 1.5, 0.75)
lowerbound <- rep(1, 4)
upperbound <- rep(5, 4)

corMat <- matrix(
  c(
    1.00, 0.30, 0.40, 0.60,
    0.30, 1.00, 0.50, 0.70,
    0.40, 0.50, 1.00, 0.80,
    0.60, 0.70, 0.80, 1.00
  ),
  nrow = 4, ncol = 4
)

scale_names <- c("Q1", "Q2", "Q3", "Q4")
rownames(corMat) <- scale_names
colnames(corMat) <- scale_names

### apply function

df1 <- makeScales(
  n = n, means = dfMeans, sds = dfSds,
  lowerbound = lowerbound, upperbound = upperbound, cormatrix = corMat
)

### test function

str(df1)

#### means
apply(df1, 2, mean) |> round(3)

#### standard deviations
apply(df1, 2, sd) |> round(3)

#### correlations
cor(df1) |> round(3)



## Example 2: five correlated Likert scales

### a study on employee engagement and organizational climate:
# Job Satisfaction (JS)
# Organizational Commitment (OC)
# Perceived Supervisor Support (PSS)
# Work Engagement (WE)
# Turnover Intention (TI) (reverse-related to others).

### define parameters

n <- 128
dfMeans <- c(3.8, 3.6, 3.7, 3.9, 2.2)
dfSds <- c(0.7, 0.8, 0.7, 0.6, 0.9)
lowerbound <- rep(1, 5)
upperbound <- rep(5, 5)
items <- c(4, 4, 3, 3, 3)

corMat <- matrix(
  c(
    1.00, 0.72, 0.58, 0.65, -0.55,
    0.72, 1.00, 0.54, 0.60, -0.60,
    0.58, 0.54, 1.00, 0.57, -0.45,
    0.65, 0.60, 0.57, 1.00, -0.50,
    -0.55, -0.60, -0.45, -0.50, 1.00
  ),
  nrow = 5, ncol = 5
)

scale_names <- c("JS", "OC", "PSS", "WE", "TI")
rownames(corMat) <- scale_names
colnames(corMat) <- scale_names

### apply function

df2 <- makeScales(
  n = n, means = dfMeans, sds = dfSds,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items, cormatrix = corMat
)

### test function

str(df2)

#### means
apply(df2, 2, mean) |> round(3)

#### standard deviations
apply(df2, 2, sd) |> round(3)

#### correlations
cor(df2) |> round(3)

}
