% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_ration_growthfac.R
\name{calc_ration_growthfac}
\alias{calc_ration_growthfac}
\title{Calculate growth increments}
\usage{
calc_ration_growthfac(
  k,
  Linf,
  nsc,
  nfish,
  l_bound,
  u_bound,
  mid,
  W_a,
  W_b,
  phi_min,
  vary_growth = TRUE,
  growth_eff = 0.5,
  growth_eff_decay = 0.11
)
}
\arguments{
\item{k}{A numeric vector of length \code{nfish} representing the von Bertalanffy growth parameter \code{(1/yr)} for each species.}

\item{Linf}{A numeric vector of length \code{nfish} representing the asymptotic length of each species.}

\item{nsc}{A numeric value representing the number of length classes in the model.}

\item{nfish}{A numeric value representing the number of species in the model.}

\item{l_bound}{A numeric vector of length \code{nsc} representing the lower bounds of the length classes.}

\item{u_bound}{A numeric vector of length \code{nsc} representing the upper bounds of the length classes.}

\item{mid}{A numeric vector of length \code{nfish} representing the mid-point of the length classes.}

\item{W_a}{A numeric vector of length \code{nfish} representing the parameter \code{a} in the length-weight conversion. See 'Details' for more information.}

\item{W_b}{A numeric vector of length \code{nfish} representing the parameter \code{b} in the length-weight conversion. See 'Details' for more information.}

\item{phi_min}{A numeric value representing the time step of the model.}

\item{vary_growth}{A logical statement indicating whether growth efficiency should vary for each species (\code{vary_growth=TRUE}) or be fixed at the value given by \code{fixed_growth} (\code{vary_growth=FALSE}). The default is \code{FALSE}.}

\item{growth_eff}{If \code{vary_growth==TRUE}, \code{growth_eff} is a numeric representing the growth efficiencies of a fish of length 0. If \code{vary_growth==FALSE}, \code{growth_eff} is a numeric value of length \code{1} representing a fixed growth efficiency for all fish. The default is 0.5.}

\item{growth_eff_decay}{A numeric value specifying the rate at which growth efficiency decreases as length approaches \code{Linf}. The default is 0.11.}
}
\value{
A list object containing \code{ration}, \code{sc_Linf}, \code{wgt} and \code{g_eff}. \code{ration} is a matrix with dimensions \code{nsc} and \code{nfish} representing the amount of food required for fish of a given species and length class to grow according to the von Bertalanffy growth curve in a time step. \code{sc_Linf} is a numeric vector of length \code{nfish} representing the length class at which each species reaches \code{Linf}. \code{wgt} is a matrix with dimensions \code{nsc} and \code{nfish} representing the weight of each species in each length class. \code{g_eff} is a matrix with dimensions \code{nsc} and \code{nfish} representing the growth efficiency of each species in each length class.
}
\description{
Calculates the amount of food required for fish of a given species and length class to grow according to the von Bertalanffy growth curve in a time step.
}
\details{
The weight increments of the \code{i}th species in the \code{j}th length class is calculated by determining the amount an individual will grow in one time step, \code{phi_min}, if it were to follow the von Bertalanffy growth curve

\code{L22=(Linf[i]-mid[j])*(1-exp(-k[i]*phi_min))}.

The weight of a fish at the mid-point of the size class is calculated using the length-weight relationship

\code{wgt[j,i] = a[i]*mid[j]^b[i]},

and similarly the expected change in weight of the the fish is calculated as

\code{growth_inc = (W_a[i]*L22^W_b[i])}.

It also has a growth efficiency

\code{g_eff[j, i]=(1-(wgt[j,i]/(W_a[i]*Linf[i]^W_b[i]))^growth_eff_decay)*growth_eff}

if \code{vary_growth==TRUE} or \code{g_eff[j, i]=growth_eff} otherwise.

\code{ration} is then calculated by

\code{growth_inc*(1/g_eff[j, i])}.
}
\examples{
# Set up the inputs to the function - species-independent parameters
nfish <- nrow(NS_par)
nsc <- 32
maxsize <- max(NS_par$Linf)*1.01 # the biggest size is 1\% bigger than the largest Linf
l_bound <- seq(0, maxsize, maxsize/nsc); l_bound <- l_bound[-length(l_bound)]
u_bound <- seq(maxsize/nsc, maxsize, maxsize/nsc)
mid <- l_bound+(u_bound-l_bound)/2

# Set up the inputs to the function - species-specific parameters
Linf <- NS_par$Linf # the von-Bertalanffy asymptotic length of each species (cm).
W_a <- NS_par$W_a # length-weight conversion parameter.
W_b <- NS_par$W_b # length-weight conversion parameter.
k <- NS_par$k # the von-Bertalnaffy growth parameter.
Lmat <- NS_par$Lmat # the length at which 50\\\% of individuals are mature (cm).

# Get phi_min
tmp <- calc_phi(k, Linf, nsc, nfish, u_bound, l_bound, calc_phi_min=FALSE,
                  phi_min=0.1) # fixed phi_min
phi_min <- tmp$phi_min

# Calculate growth increments
tmp <- calc_ration_growthfac(k, Linf, nsc, nfish, l_bound, u_bound, mid, W_a, W_b, phi_min)
ration <- tmp$ration
sc_Linf <- tmp$sc_Linf
wgt <- tmp$wgt
g_eff <- tmp$g_eff
}
\references{
von Bertalanffy, L. (1957). Quantitative Laws in Metabolism and Growth. \emph{The Quarterly Review of Biology}, 32:217-231
}
