# =============================================================================
# MODULE 5: STOCHASTIC PROFILE
# =============================================================================

#' Compute Comprehensive Stochastic Profile
#'
#' Returns all stochastic metrics for a sample compared to controls.
#'
#' @param sample_values Numeric vector of sample measurements
#' @param pc_values Numeric vector of positive control measurements
#' @param nc_values Numeric vector of negative control measurements
#' @return List containing all computed stochastic metrics
#' @export
#' @examples
#' pc <- rnorm(20, 100, 10)
#' nc <- rnorm(20, 20, 5)
#' sample <- rnorm(20, 80, 15)
#' profile <- compute_stochastic_profile(sample, pc, nc)
compute_stochastic_profile <- function(sample_values, pc_values, nc_values) {
  sample_mean <- mean(sample_values, na.rm = TRUE)
  sample_var <- stats::var(sample_values, na.rm = TRUE)
  sample_cv <- cv(sample_values)
  sample_fano <- fano_factor(sample_values)
  pc_mean <- mean(pc_values, na.rm = TRUE)
  pc_var <- stats::var(pc_values, na.rm = TRUE)
  pc_cv <- cv(pc_values)
  nc_mean <- mean(nc_values, na.rm = TRUE)
  nc_var <- stats::var(nc_values, na.rm = TRUE)
  nc_cv <- cv(nc_values)

  sr <- function(a, b) {
    if (is.na(a) || is.na(b) || !is.finite(a) || !is.finite(b) ||
        abs(b) < .Machine$double.eps) return(NA_real_)
    a / b
  }

  list(
    mean = sample_mean, var = sample_var, cv = sample_cv, fano = sample_fano,
    cv_ratio_pc = sr(sample_cv, pc_cv),
    cv_ratio_nc = sr(sample_cv, nc_cv),
    fano_ratio_pc = sr(sample_fano, fano_factor(pc_values)),
    stochastic_index = normalized_stochastic_index(sample_var, pc_var, nc_var),
    ssmd_vs_nc = ssmd(sample_values, nc_values),
    ssmd_vs_pc = ssmd(sample_values, pc_values),
    snr = snr_vs_control(sample_values, nc_values),
    js_vs_pc = jensen_shannon(sample_values, pc_values),
    js_vs_nc = jensen_shannon(sample_values, nc_values),
    js_ratio = sr(jensen_shannon(sample_values, pc_values),
                  jensen_shannon(sample_values, nc_values)),
    energy_vs_pc = energy_distance(sample_values, pc_values),
    energy_vs_nc = energy_distance(sample_values, nc_values),
    energy_ratio = sr(energy_distance(sample_values, pc_values),
                      energy_distance(sample_values, nc_values)),
    wasserstein_vs_pc = wasserstein_1d(sample_values, pc_values),
    wasserstein_vs_nc = wasserstein_1d(sample_values, nc_values),
    wasserstein_ratio = sr(wasserstein_1d(sample_values, pc_values),
                           wasserstein_1d(sample_values, nc_values)),
    log_dist_pc = log_euclidean_distance(sample_values, pc_values),
    log_dist_nc = log_euclidean_distance(sample_values, nc_values)
  )
}
