% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_cond_density_quantile.R
\name{fit_cond_density_quantile}
\alias{fit_cond_density_quantile}
\title{Estimate conditional density pi(y|x) via quantile process + quotient estimator}
\usage{
fit_cond_density_quantile(
  dat,
  y_col = "Y",
  delta_col = "delta",
  x_cols,
  taus = seq(0.05, 0.95, by = 0.01),
  h = NULL,
  method = c("rq", "qrf"),
  enforce_monotone = TRUE,
  tail_decay = TRUE,
  num_extra_points = 10L,
  decay_factor = 0.8,
  dens_floor = 1e-10,
  eps = 1e-08,
  gap_min = 0.01,
  seed = NULL,
  ...
)
}
\arguments{
\item{dat}{data.frame in long format, containing outcome, missingness indicator, and covariates.}

\item{y_col}{name of outcome column (observed Y, may contain NA).}

\item{delta_col}{name of missingness indicator (1 observed, 0 missing).}

\item{x_cols}{character vector of covariate column names (include time if desired).}

\item{taus}{grid of quantile levels in (0,1) at which the quantile process is evaluated.}

\item{h}{Bandwidth(s) for quotient. Either a scalar or a numeric vector of length \code{length(taus)}.
If \code{NULL}, a tau-specific bandwidth vector \eqn{h(\tau)} is computed via
\code{quantreg::bandwidth.rq}, and automatically shrunk near the
boundaries to ensure \eqn{\tau\pm h(\tau)\in(0,1)}.}

\item{method}{quantile engine: \code{"rq"} (linear quantile regression) or \code{"qrf"} (quantile random forest).}

\item{enforce_monotone}{logical; if TRUE, apply isotonic regression to the predicted quantile curve in \eqn{\tau}
for each \eqn{x} to reduce quantile crossing.}

\item{tail_decay}{logical; if TRUE, add extra tail points with geometric decay before interpolation.}

\item{num_extra_points}{number of extra tail points on each side when \code{tail_decay=TRUE}.}

\item{decay_factor}{decay factor in (0,1) for tail densities when \code{tail_decay=TRUE}.}

\item{dens_floor}{lower bound for density to avoid numerical issues.}

\item{eps}{small stabilizer for denominator \code{pmax(Qplus-Qminus, eps)}.}

\item{gap_min}{minimum spacing for tail extrapolation points.}

\item{seed}{optional seed.}

\item{...}{extra arguments passed to the underlying quantile engine:
\describe{
\item{\code{rq}}{passed to \code{quantreg::rq.fit}, e.g. \code{rq_method="br"}.}
\item{\code{qrf}}{passed to \code{quantregForest::quantregForest}, e.g. \code{ntree=500}.}
}}
}
\value{
A list containing fitted objects and prediction functions:
\describe{
\item{\code{predict_Q(x_new, taus_use)}}{
Returns the estimated conditional quantiles
\deqn{
        \widehat Q_Y(\tau \mid x)
    }
for \eqn{\tau \in (0,1)} specified by \code{taus_use},
evaluated at new covariate values \code{x_new}.
The output is a numeric matrix with one row per covariate vector \eqn{x}
and one column per quantile level \eqn{\tau}.
}

\item{\code{predict_density(x_new, y_new)}}{
Returns the estimated conditional density
\deqn{
      \widehat \pi(y \mid x),
    }
evaluated at specified (x,y) pairs.
The inputs \code{x_new} and \code{y_new} are paired row-wise, so that the
r-th row of \code{x_new} is evaluated at \code{y_new[r]}.
}
}
}
\description{
Fits a conditional quantile function \eqn{\widehat Q_Y(\tau\mid x)} using pooled observed data
(working-independence), and estimates the conditional density through the quotient estimator
along the quantile curve:
\deqn{
\widehat\pi\{\widehat Q(\tau\mid x)\mid x\}
= \frac{2h(\tau)}{\widehat Q(\tau+h(\tau)\mid x)-\widehat Q(\tau-h(\tau)\mid x)}.
}
For numerical stability, the quantile curve can be monotone-adjusted (isotonic regression),
and tail decay extrapolation can be used before interpolation to \eqn{\pi(y\mid x)}.
}
\examples{
## ------------------------------------------------------------
## Case A: Conditional density evaluated at a single point (x, y)
## ------------------------------------------------------------
## This illustrates the most basic usage: estimating pi(y | x)
## at one covariate value x and one response value y.

dat <- generate_clustered_mar(
  n = 200, m = 4, d = 2,
  target_missing = 0.3, seed = 1
)
fit <- fit_cond_density_quantile(
  dat,
  y_col = "Y", delta_col = "delta",
  x_cols = c("X1", "X2"),
  taus = seq(0.05, 0.95, by = 0.02),
  method = "rq",
  seed = 1
)
## a single covariate value x
x1 <- matrix(c(0.2, -1.0), nrow = 1)
colnames(x1) <- c("X1", "X2")
## estimate pi(y | x) at y = 0.5
fit$predict_density(x1, y_new = 0.5)


## ------------------------------------------------------------
## Case B: Conditional density as a function of y (density curve)
## ------------------------------------------------------------
## Here we fix x and evaluate pi(y | x) over a grid of y values,
## which produces an estimated conditional density curve.

y_grid <- seq(-3, 3, length.out = 201)
## reuse the same x by repeating it to match the y-grid
x_rep <- x1[rep(1, length(y_grid)), , drop = FALSE]
f_grid <- fit$predict_density(x_rep, y_grid)

## ------------------------------------------------------------
## True conditional density under the data generator
## ------------------------------------------------------------
## Data are generated as:
##   Y = X^T beta + b + eps,
##   b ~ N(0, sigma_b^2),  eps ~ N(0, sigma_eps^2)
## Hence the marginal conditional density is:
##   Y | X = x ~ N(x^T beta, sigma_b^2 + sigma_eps^2)

beta_true <- c(0.5, 0.6)
sigma_b_true <- 0.7
sigma_eps_true <- 1.0
mu_true <- drop(x1 \%*\% beta_true)
sd_true <- sqrt(sigma_b_true^2 + sigma_eps_true^2)
f_true <- stats::dnorm(y_grid, mean = mu_true, sd = sd_true)


## ------------------------------------------------------------
## Visualization: estimated vs true conditional density
## (use smooth.spline on log-density for a smoother display)
## ------------------------------------------------------------

## smooth the estimated curve for visualization
ok <- is.finite(f_grid) & (f_grid > 0)
sp <- stats::smooth.spline(y_grid[ok], log(f_grid[ok]), spar = 0.85)
f_smooth <- exp(stats::predict(sp, y_grid)$y)

ymax <- max(c(f_smooth, f_true), na.rm = TRUE)
plot(
  y_grid, f_smooth,
  type = "l", lwd = 2,
  xlab = "y",
  ylab = expression(hat(pi)(y ~ "|" ~ x)),
  ylim = c(0, 1.2 * ymax),
  main = "Conditional density at a fixed x: estimated vs true"
)
grid(col = "gray85", lty = 1)
lines(y_grid, f_true, lwd = 2, lty = 2)
legend(
  "topright",
  legend = c("Estimated (smoothed)", "True (generator)"),
  lty = c(1, 2), lwd = c(2, 2), bty = "n"
)

}
