#' Compute binomial probabilities over categorical variable levels
#'
#' @param DF A data frame containing `binVar` and `catVar`
#' @param binVar Binary variable for binomial probabilities
#' @param catVar Categorical variable over which binomial probabilities are computed
#' @param targetLevel Positive response level for `binVar`
#' @param keepNA Missing data handling option: `ifany` (the default), `no`, or `always`
#' @param keepLevels Optional subset of `catVar` levels to be used in the analysis; default `NULL` retains all levels
#' @param cLevel Confidence level for binomial probabilities (default 0.95)
#'
#' @return Data frame with one row for each `catVar` level in the analysis and these 6 columns:
#'   * `Level` the `catVar` level
#'   * `nWith` the number of records with `catVar` equal to `Level` and `binVar` equal to `targetLevel`
#'   * `nTotal` the total number of records with `catVar` equal to `Level`
#'   * `pEst` the estimated probability that `binVar` equals `targetLevel`
#'   * `loCI` the lower `cLevel` confidence limit for `pEst`
#'   * `upCI` the upper `cLevel` confidence limit for `pEst`
#' @export
#'
#' @examples
#' catVar <- c(rep("A", 100), rep("B", 100), rep("C", 100))
#' binVar <- c(rep(0,80),rep(1,20), rep(0,50),rep(1,50), rep(0,20),rep(1,80))
#' DF <- data.frame(catVar = catVar, binVar = binVar)
#' BinomialCIsByCategorical(DF, "binVar", "catVar", 1)
BinomialCIsByCategorical <- function(DF, binVar, catVar, targetLevel,
                                     keepNA = "ifany", keepLevels = NULL, cLevel = 0.95){
  #
  stopifnot("DF must be a data frame"= is.data.frame(DF))
  stopifnot("binVar not found in DF"= binVar %in% colnames(DF))
  stopifnot("catVar not found in DF"= catVar %in% colnames(DF))
  stopifnot("binVar never equal to targetLevel"= length(which(DF[, binVar] == targetLevel)) > 0)
  stopifnot("keepNA must be 'ifany', 'no', or 'always'"= keepNA %in% c('ifany', 'no', 'always'))
  stopifnot("cLevel not between 0 and 1"= cLevel > 0 & cLevel < 1)
  #
  nTbl <- table(DF[, catVar], useNA = keepNA)
  bindex <- which(DF[, binVar] == targetLevel)
  xTbl <- table(DF[bindex, catVar], useNA = keepNA)
  if (is.null(keepLevels)){
    useLevels <- names(nTbl)
  } else {
    useLevels <- intersect(names(nTbl), keepLevels)
    stopifnot("keepLevels values not found in catVar"= length(useLevels) > 0)
  }
  nLevels <- length(useLevels)
  #
  nWith <- vector("numeric", nLevels)
  nTotal <- vector("numeric", nLevels)
  pEst <- vector("numeric", nLevels)
  loCI <- vector("numeric", nLevels)
  upCI <- vector("numeric", nLevels)
  #
  for (i in 1:nLevels){
    lvl <- useLevels[i]
    if (is.na(lvl)){
      nIndex <- which(is.na(names(nTbl)))
      n <- nTbl[nIndex]
      xIndex <- which(is.na(names(xTbl)))
      x <- ifelse(is.na(xTbl[xIndex]), 0, xTbl[xIndex])
    } else {
      n <- nTbl[lvl]
      x <- ifelse(is.na(xTbl[lvl]), 0, xTbl[lvl])
    }
    nTotal[i] <- n
    nWith[i] <- x
    CIobj <- PropCIs::addz2ci(x, n, cLevel)
    pEst[i] <- CIobj$estimate
    loCI[i] <- CIobj$conf.int[1]
    upCI[i] <- CIobj$conf.int[2]
  }
  #
  outFrame <- data.frame(Level = useLevels, nWith = nWith, nTotal = nTotal,
                         pEst = pEst, loCI = loCI, upCI = upCI)
  #
  #  Specify class as BinomCIframe, inheriting data.frame instead of default list
  #
  class(outFrame) <- c("BinomCIframe", "data.frame")
  #
  return(outFrame)
}

