#' @title Generate Data with Elliptically Distributed Covariates
#' @description This function generates a dataset for a linear model where the covariate matrix X follows an elliptical distribution.
#' @details
#' The function generates a response vector Y based on the linear model Y = X %*% beta + e.
#' The covariate matrix X is generated from one of three types of elliptical distributions:
#' 1.  `Elliptical-Normal`: Based on a multivariate normal distribution structure.
#' 2.  `Elliptical-t`: Based on a multivariate t-distribution structure.
#' 3.  `Elliptical-cov`: Based on a custom covariance matrix adjusted via its eigenvalues.
#' The error term 'e' is drawn from a standard normal distribution.
#' @param n The number of observations (rows) to generate.
#' @param p The number of predictors/dimensions (columns) for the covariate matrix X.
#' @param dist_type A character string specifying the type of elliptical distribution for X.
#'        Must be one of "Elliptical-Normal", "Elliptical-t", or "Elliptical-cov".
#' @return A list containing the following components:
#' \item{X}{An n x p matrix of covariates from the specified elliptical distribution.}
#' \item{Y}{A numeric vector of n responses.}
#' \item{e}{A numeric vector of n error terms from a standard normal distribution.}
#' @export
#' @importFrom stats qchisq qf rnorm runif rt
#' @importFrom MASS mvrnorm
#' @references
#' Guo, G., Song, H. & Zhu, L. The COR criterion for optimal subset selection in distributed estimation. \emph{Statistics and Computing}, 34, 163 (2024). \doi{10.1007/s11222-024-10471-z}
#'
#' Guo, G., Sun, Y., Qian, G., & Wang, Q. (2022). LIC criterion for optimal subset selection in distributed interval estimation. \emph{Journal of Applied Statistics}, 50(9), 1900-1920. \doi{10.1080/02664763.2022.2053949}.
#'
#' Chang, D., Guo, G. (2024). LIC: An R package for optimal subset selection for distributed data. \emph{SoftwareX}, 28, 101909.
#'
#' Jing, G., & Guo, G. (2025). TLIC: An R package for the LIC for T distribution regression analysis. \emph{SoftwareX}, 30, 102132.
#'
#' Chang, D., & Guo, G. Research on Distributed Redundant Data Estimation Based on LIC. \emph{IAENG International Journal of Applied Mathematics}, 55(1), 1-6 (2025).
#'
#' Gao, H., & Guo, G. LIC for Distributed Skewed Regression. \emph{IAENG International Journal of Applied Mathematics}, 55(9), 2925-2930 (2025).
#'
#' Zhang, C., & Guo, G. (2025). The optimal subset estimation of distributed redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 270-277.
#'
#' Jing, G., & Guo, G. (2025). Student LIC for distributed estimation. \emph{IAENG International Journal of Applied Mathematics}, 55(3), 575-581.
#'
#' Liu, Q., & Guo, G. (2025). Distributed estimation of redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 332-337.
#' @examples
#' # Generate 100 observations with 5 predictors from an Elliptical-Normal distribution
#' data_normal <- eerr(n = 100, p = 5, dist_type = "Elliptical-Normal")
#' str(data_normal)
#'
#' # Generate 100 observations with 3 predictors from an Elliptical-cov distribution
#' data_cov <- eerr(n = 100, p = 3, dist_type = "Elliptical-cov")
#' pairs(data_cov$X) # Visualize the relationships between covariates
#'
eerr <- function(n, p, dist_type) {

  # --- 1. Generate beta coefficients and error term ---
  beta <- sort(stats::runif(p, 1, 5))
  e <- stats::rnorm(n)

  # --- 2. Generate covariate matrix X based on dist_type ---
  if (dist_type == "Elliptical-Normal") {
    mu <- rep(0, p)
    Sigma <- diag(p)
    X <- MASS::mvrnorm(n = n, mu = mu, Sigma = Sigma)

  } else if (dist_type == "Elliptical-t") {
    df <- 5
    mu <- rep(0, p)
    Sigma <- diag(p)
    # Standard way to generate multivariate t-distributed data
    Y <- MASS::mvrnorm(n = n, mu = mu, Sigma = Sigma)
    u <- stats::rchisq(n, df = df)
    X <- Y / sqrt(u / df)

  } else if (dist_type == "Elliptical-cov") {
    A <- matrix(stats::rnorm(p^2), nrow = p)
    scale <- A %*% t(A)
    eigen_decomp <- eigen(scale)
    eigen_values <- eigen_decomp$values
    eigen_vectors <- eigen_decomp$vectors
    adjust_factors <- stats::runif(p, 0.8, 1.2)
    eigen_values_new <- eigen_values * adjust_factors
    scale_new <- eigen_vectors %*% diag(eigen_values_new) %*% t(eigen_vectors)
    mu <- rep(0, p)
    X <- MASS::mvrnorm(n = n, mu = mu, Sigma = scale_new)

  } else {
    stop("Unknown distribution type: '", dist_type,
         "'. Please use 'Elliptical-Normal', 'Elliptical-t', or 'Elliptical-cov'.")
  }

  # --- 3. Calculate response Y ---
  Y <- X %*% beta + e

  # --- 4. Return the list of data ---
  return(list(X = X, Y = Y, e = e))
}
