#' Minimum Average Partial (MAP) Test
#'
#' @description
#' The MAP test by Velicer (1976)was originally designed for
#' determining the number of components in PCA but is used in EFA as well.
#' At it score, the averaged partial correlations after excluding the
#' variance that can be explained by the previous components are compared among
#' different solutions and the one with the smallest value is chosen (therefore, MAP test).
#' @seealso \code{\link[EFAfactors]{MAP}}
#'
#' @param response A required \code{N} × \code{I} matrix or data.frame consisting of the responses
#'        of \code{N} individuals to \code{I} items.
#' @param fa A string that determines the method used to obtain factors loadings. If \code{"pc"}, it represents
#'           Principal Component Analysis (PCA); if \code{"fa"}, it represents Maximun Likelihhod Estimation (a widely
#'           used Factor Analysis method; @seealso \code{\link[psych]{fa}};
#'           Auerswald & Moshagen, 2019). (Default = \code{"pc"})
#' @param nfact.max The maximum number of factors discussed by MAP. (default = 10)
#' @param cor.type A character string indicating which correlation coefficient (or covariance) is
#'                to be computed. One of \code{"pearson"} (default), \code{"kendall"}, or
#'                \code{"spearman"}. @seealso \code{\link[stats]{cor}}.
#' @param use an optional character string giving a method for computing covariances in the presence of missing values. This
#'          must be one of the strings \code{"everything"}, \code{"all.obs"}, \code{"complete.obs"}, \code{"na.or.complete"},
#'          or \code{"pairwise.complete.obs"} (default). @seealso \code{\link[stats]{cor}}.
#' @param vis A Boolean variable that will print the factor retention results when set to \code{TRUE}, and will not print
#'          when set to \code{FALSE}. (default = \code{TRUE})
#' @param plot A Boolean variable that will print the MAP plot when set to \code{TRUE}, and will not print it when set to
#'          \code{FALSE}. @seealso \code{\link[EFAfactors]{plot.MAP}}. (Default = \code{TRUE})
#'
#' @return An object of class \code{MAP}, which is a \code{list} containing the following components:
#' \item{nfact}{The number of factors to retain by both Optimal Coordinate and PA.}
#' \item{MAP.values}{the averaged squared partial correlations for 1 to \code{nfact.max} factors.}
#'
#' @references
#'  Velicer, W. F. (1976). Determining the number of components from the matrix of partial correlations. Psychometrika, 41(3), 321–327. https://doi.org/10.1007/BF02293557
#'
#'  Goretzko, D. (2025). How many factors to retain in exploratory factor analysis? A critical overview of factor retention methods. Psychological methods, Advance online publication. https://doi.org/10.1037/met0000733
#'
#' @examples
#' library(EFAfactors)
#' set.seed(123)
#'
#' ##Take the data.bfi dataset as an example.
#' data(data.bfi)
#'
#' response <- as.matrix(data.bfi[, 1:25]) ## loading data
#' response <- na.omit(response) ## Remove samples with NA/missing values
#'
#' ## Transform the scores of reverse-scored items to normal scoring
#' response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1
#'
#' \donttest{
#'  MAP.obj <- MAP(response, plot=FALSE)
#'
#'  ## MAP plot
#'  plot(MAP.obj)
#'
#' }
#'
#' @importFrom stats prcomp
#' @importFrom psych fa factor.scores
#' @export
MAP <- function(response, fa = "pc", nfact.max=10,
                cor.type = "pearson", use = "pairwise.complete.obs",
                vis = TRUE, plot = TRUE) {
  if (!any(rep(use, 5) == c("everything", "all.obs", "complete.obs",
                            "na.or.complete", "pairwise.complete.obs")))
    stop("'use' must be one of the strings 'everything', 'all.obs', 'complete.obs', 'na.or.complete', or 'pairwise.complete.obs' !")
  if (!any(rep(fa, 2) == c("pc", "fa")))
    stop("'type' must be one of the strings 'fa' or 'pc' !")

  N <- dim(response)[1]
  I <- dim(response)[2]
  response <- scale(response)

  remove.nfact <- function(response, nfact) {
    if(fa == "pc"){
      pca.res <- prcomp(response, center = FALSE, scale. = FALSE)
      Loading <- pca.res$rotation[, 1:nfact]
      scores.nfact <- pca.res$x[, 1:nfact]
    }else if(fa == "fa"){
      fa.res <- fa(response, nfactors = nfact, rotate = "none", fm = "ml")
      Loading <- fa.res$loadings[, 1:nfact]
      scores.nfact <- factor.scores(response, fa.res)$scores
    }

    response.nfact <- scores.nfact %*% t(Loading)

    response.residual <- response - response.nfact
    return(response.residual)
  }

  MAP.values <- sapply(1:nfact.max, function(f){
    if(f > I - 1){
      return(Inf)
    }
    tryCatch({
      response.residual <- remove.nfact(response, f)
      cor.residual <- cor(response.residual, method = cor.type, use = use)
      mean(cor.residual[upper.tri(cor.residual)]^2)
    }, error = function(e){ Inf })
  })

  nfact <- which.min(MAP.values)

  MAP.obj <- list(MAP.values=MAP.values, nfact=nfact)
  class(MAP.obj) <- "MAP"

  if (vis)
    print(MAP.obj)
  if (plot)
    plot(MAP.obj)

  return(MAP.obj)
}
