#' Two-Sample CLX Test
#' Performs a two-sample CLX test for high-dimensional data to compare the means of two 
#' samples.
#' This function implements the CLX test, which is suitable for high-dimensional data with small 
#'sample sizes.
#' It estimates the covariance matrix, and calculates the test statistic and p-value.
#' @param X A matrix representing the first sample, where rows are variables and columns are observations.
#' @param Y A matrix representing the second sample, where rows are variables and columns #'are observations.
#' @param alpha The significance level for the test.
#'
#' @return A list containing:
#' \item{statistics}{The test statistic (Momega).}
#' \item{p.value}{The p-value of the test.}
#' \item{alternative}{The alternative hypothesis ("two.sided").}
#' \item{method}{The method used ("Two-Sample CLX test").}
#'
#' @examples
#' # Example usage:
#' set.seed(123)
#' p <- 100  # Number of variables
#' n1 <- 20  # Sample size for X
#' n2 <- 20  # Sample size for Y
#' X <- matrix(rnorm(n1 * p), nrow = p, ncol = n1)
#' Y <- matrix(rnorm(n2 * p, mean = 0.5), nrow = p, ncol = n2)
#' result <- CLX(X, Y, alpha = 0.05)
#' print(result)
#'
#' @export
CLX = function(X,Y,alpha) {
  p = nrow(X)
  n1 = ncol(X)
  n2 = ncol(Y)
  Xbar = rowMeans(X,2)
  Ybar = rowMeans(Y,2)
  Xm = X - Xbar
  Ym = Y - Ybar
  SS = (Xm %*% t(Xm) + Ym %*% t(Ym))/(n1+n2)
  btheta = matrix(0, p, p)
  for (i in 1:p) {
    for (j in i:p) {
      xi = Xm[i,]
      xj = Xm[j,]
      yi = Ym[i,]
      yj = Ym[j,]
      btheta[i,j] = (sum((xi*xj - SS[i,j])^2) + sum((yi*yj - SS[i,j])^2))/(n1+n2)
    }
  }
  theta = btheta + t(btheta)
  theta = theta - diag(diag(theta)/2)
  Sigma = SS * as.numeric( abs(SS) >= 2*sqrt(theta*log(p)/(n1+n2)))
  eig = eigen(Sigma)
  s = diag(eig$values)
  v = eig$vectors
  Sigma = v %*% diag(pmax(diag(s), log(p)/(n1+n2))) %*% t(v)
  X = solve(Sigma, X)
  Y = solve(Sigma, Y)
  Xbar = rowMeans(X)
  Ybar = rowMeans(Y)
  Xm = X - Xbar
  Ym = Y - Ybar
  S1 = (Xm %*% t(Xm))/n1
  S2 = (Ym %*% t(Ym))/n2
  Z = Xbar - Ybar
  om = n1/(n1+n2)*diag(S1) + n2/(n1+n2)*diag(S2)
  Momega = n1*n2/(n1+n2)*max(Z^2/om)
  Xn = as.numeric( Momega >= 2*log(p) - log(log(p)) - log(pi) - 2* log(log(1/(1-alpha))))
  p.val = exp(exp(-(Momega - 2*log(p) + log(log(p)) + log(pi))/2))
  p.val = 1- 1/p.val
  res = list(statistics = Momega, p.value = p.val, alternative = "two.sided",
             method = "Two-Sample CLX test")
  return(res)
}