% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fast_appli.R
\name{direct_discrete_FWER}
\alias{direct_discrete_FWER}
\title{Direct Application of Multiple Testing Procedures to Dataset}
\usage{
direct_discrete_FWER(
  dat,
  test_fun,
  test_args = NULL,
  alpha = 0.05,
  independence = FALSE,
  single_step = TRUE,
  critical_values = FALSE,
  select_threshold = 1,
  preprocess_fun = NULL,
  preprocess_args = NULL
)
}
\arguments{
\item{dat}{input data; must be suitable for the first parameter of the provided \code{preprocess_fun} function or, if \code{preprocess_fun} is \code{NULL}, for the first parameter of the \code{test_fun} function.}

\item{test_fun}{function \strong{from package \code{\link[DiscreteTests:DiscreteTests-package]{DiscreteTests}}}, i.e. one whose name ends with \verb{*_test_pv} and which performs hypothesis tests and provides an object with \eqn{p}-values and their support sets; can be specified by a single character string (which is automatically checked for being a suitable function \strong{from that package} and may be abbreviated) or a single function object.}

\item{test_args}{optional named list with arguments for \code{test_fun}; the names of the list fields must match the test function's parameter names. The first parameter of the test function (i.e. the data) MUST NOT be included!}

\item{alpha}{single real number strictly between 0 and 1 indicating the target FWER level.}

\item{independence}{single boolean specifying whether the \eqn{p}-values are statistically independent or not.}

\item{single_step}{single boolean specifying whether to perform a single-step (\code{TRUE}) or step-down (\code{FALSE}; the default) procedure.}

\item{critical_values}{single boolean specifying whether critical constants are to be computed.}

\item{select_threshold}{single real number strictly between 0 and 1 indicating the largest raw \eqn{p}-value to be considered, i.e. only \eqn{p}-values below this threshold are considered and the procedures are adjusted in order to take this selection effect into account; if \code{select_threshold = 1} (the default), all raw \eqn{p}-values are selected.}

\item{preprocess_fun}{optional function for pre-processing the input \code{data}; its result must be suitable for the first parameter of the \code{test_fun} function.}

\item{preprocess_args}{optional named list with arguments for \code{preprocess_fun}; the names of the list fields must match the pre-processing function's parameter names. The first parameter of the test function (i.e. the data) MUST NOT be included!}
}
\value{
A \code{DiscreteFWER} S3 class object whose elements are:
\item{Rejected}{rejected raw \eqn{p}-values.}
\item{Indices}{indices of rejected hypotheses.}
\item{Num_rejected}{number of rejections.}
\item{Adjusted}{adjusted \eqn{p}-values.}
\item{Critical_constants}{critical values (only exists if computations where performed with \code{critical_values = TRUE}).}
\item{Data}{list with input data.}
\item{Data$Method}{character string describing the performed algorithm, e.g. 'Discrete Bonferroni procedure'.}
\item{Data$Raw_pvalues}{observed \eqn{p}-values.}
\item{Data$pCDFlist}{list of the \eqn{p}-value supports.}
\item{Data$FWER_level}{FWER level \code{alpha}.}
\item{Data$Independence}{boolean indicating whether the \eqn{p}-values were considered as independent.}
\item{Data$Single_step}{boolean indicating whether a single-step or step-down procedure was performed.}
\item{Data$Data_name}{the respective variable names of the input data.}
\item{Select}{list with data related to \eqn{p}-value selection; only exists if \code{select_threshold < 1}.}
\item{Select$Threshold}{\eqn{p}-value selection threshold (\code{select_threshold}).}
\item{Select$Effective_Thresholds}{results of each \eqn{p}-value CDF evaluated at the selection threshold.}
\item{Select$Pvalues}{selected \eqn{p}-values that are \eqn{\leq} selection threshold.}
\item{Select$Indices}{indices of \eqn{p}-values \eqn{\leq} selection threshold.}
\item{Select$Scaled}{scaled selected \eqn{p}-values.}
\item{Select$Number}{number of selected \eqn{p}-values \eqn{\leq} selection threshold.}
}
\description{
Apply one of the various FWER adaptation procedures, with or without
computing the critical constants, to a data set of 2x2 contingency tables
using statistical test functions from package
\code{\link[DiscreteTests:DiscreteTests-package]{DiscreteTests}}. If necessary,
functions for pre-processing can be passed as well.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

# Computation of p-values and their supports with Fisher's exact test
library(DiscreteTests)  # for Fisher's exact test
test_results <- fisher_test_pv(df)
raw_pvalues <- test_results$get_pvalues()
pCDFlist <- test_results$get_pvalue_supports()

DBonf <- direct_discrete_FWER(df, "fisher")
summary(DBonf)

DHolm <- direct_discrete_FWER(df, "fisher_test_pv", single_step = FALSE)
summary(DHolm)

DBonf_bin <- direct_discrete_FWER(X1 + X2, "binom_test_pv", 
                                  list(n = N1 + N2, p = 0.05))
summary(DBonf_bin)

DHolm_bin <- direct_discrete_FWER(X1 + X2, "binom", 
                                  list(n = N1 + N2, p = 0.05),
                                  single_step = TRUE)
summary(DHolm_bin)

}
